// Cellular Automata example for linear cell patterns
// by Takehiko Nagakura (rev. 2013.4.31)
//
// This sample is derived from Daniel Shiffman's ebook
//         The Nature of Code
// See the site below for the original chapter.
// http://natureofcode.com/book/chapter-7-cellular-automata/

// Cells are array of black or white squares
// Each takes a state of 1 or 0: 1 means white cell. 0 means black cell
int[] cells = {1,0,1,0,0,0,0,1,0,1,1,1,0,0,0,1,1,1,0,0};

// Another array to temporarily store (copy) the new states of the cells
// You need this when you compute the transformation of current cells to next generation.
int[] newcells = new int[cells.length]; //[bold]

// Size of cell drawn on the screen.
int cellsize = 40;

// slows down the iteration for ease of view
int delat_mil_seconds = 1000;

void setup() {
  size(900,150);
}

void draw()  {
  delay(delat_mil_seconds); // slow down the iteration a bit.
  
  for (int i = 1; i < cells.length-1; i++) {
    // You will look at 3 adjacent cells. Depending on what
    // colors they have, you will decide the color of the
    // center cell for the next generation.
    int left   = cells[i-1];
    int middle = cells[i];
    int right  = cells[i+1];
    
    // The transformation rule is described in a separate function.
    int newstate = rules(left,middle,right);
    
    // Temporarily saves the new state (color) in the new array
    newcells[i] = newstate; //[bold]
  }
  
  // Coopy the new generation cells to the current generation.
  cells = newcells; //[bold]

  // Loop through every cell to draw them on screen.
  // background(0);
  for (int i = 0; i < cells.length; i++) {
    if (cells[i] == 1) fill(255); // white cell if the state is 1
    else fill(0); // black cell if the state is 0
    stroke(0); // black frame for each cell
    rect(cellsize + i*cellsize,cellsize,cellsize,cellsize);
  }
}

// This is to compute the new color of a cell based on the
// current colors of that cell plus its left and right cells.
// Here we have 8 rules that are hard-coded.

int rules (int a, int b, int c) {
    if      (a == 1 && b == 1 && c == 1) return 0;
    else if (a == 1 && b == 1 && c == 0) return 1;
    else if (a == 1 && b == 0 && c == 1) return 0;
    else if (a == 1 && b == 0 && c == 0) return 1;
    else if (a == 0 && b == 1 && c == 1) return 1;
    else if (a == 0 && b == 1 && c == 0) return 0;
    else if (a == 0 && b == 0 && c == 1) return 1;
    else if (a == 0 && b == 0 && c == 0) return 0;
    return 0;  // Just in cases where the states do not match one of the eight possibilities above
  }
