// Cellular Automata example for linear cell patterns
// by Takehiko Nagakura (rev. 2013.4.31)
//
// This sample is derived from Daniel Shiffman's ebook
//         The Nature of Code
// See the site below for the original chapter.
// http://natureofcode.com/book/chapter-7-cellular-automata/

// Cells are array of black or white squares
// Each takes a state of 1 or 0: 1 means white cell. 0 means black cell
int[] cells = {1,0,1,0,0,0,0,1,0,1,1,1,0,0,0,1,1,1,0,0};

// Another array to temporarily store (copy) the states of cells
// when you compute the transformation of cells into next generation.
int[] newcells = new int[cells.length]; //[bold]

// Used to describe the pattern of transformations. (See below)
int[] ruleset = {0,1,0,1,1,0,1,0};

// Size of cell drawn on the screen.
int cellsize = 30;

int counter = 0;

void setup() {
  size(900,820);
  
  arrayCopy(cells, newcells); // initialize
  // draw the first row separately at the top
  draw_cell_row ( 7*cellsize, cellsize); 
}

void draw()  {
  delay(1000); // slow down the iteration a bit.
  
  counter = counter + 1;
  
  for (int i = 1; i < cells.length-1; i++) {
    // You will look at 3 adjacent cells. Depending on what
    // colors they have, you will decide the color of the
    // center cell for the next generation.
    int left   = cells[i-1];
    int middle = cells[i];
    int right  = cells[i+1];
    
    // The transformation rule is described in a separate function.
    int newstate = rules(left,middle,right);
    
    // Temporarily saves the new state (color) in the new array
    newcells[i] = newstate; //[bold]
  }
  
  // Copy the new generation cells to the current generation.
  cells = newcells; //[bold]
  // Draws the new row of cells
  draw_cell_row ( 7*cellsize, cellsize * (counter + 2) ) ;
}

/////////////////////// cells ///////////////////////////////////////////

 // This draws a row of cells. Topleft corner of the row is x=top, y=left.
void draw_cell_row(int left, int top) {
    // Loop through every cell to draw them on screen.
    fill(0);
    text(counter, left - 20, top + 15);
    for (int i = 0; i < cells.length; i++) {
      draw_one_cell(left + i*cellsize, top, cells[i]);
  }
}

 // This draws one cell (white or black) at a given position.
void draw_one_cell(int posx, int posy, int state) {
  if (state == 1) fill(255); // white cell if the state is 1
  else fill(0); // black cell if the state is 0
  stroke(0); // black frame for each cell
  rect( posx, posy, cellsize, cellsize );
}

/////////////////////// rules ///////////////////////////////////////////

// This is to compute the new color of a cell based on the
// current colors of that cell plus its left and right cells.
// Here we have 8 rules.

int rules (int a, int b, int c) {
    if      (a == 1 && b == 1 && c == 1) return ruleset[0];
    else if (a == 1 && b == 1 && c == 0) return ruleset[1];
    else if (a == 1 && b == 0 && c == 1) return ruleset[2];
    else if (a == 1 && b == 0 && c == 0) return ruleset[3];
    else if (a == 0 && b == 1 && c == 1) return ruleset[4];
    else if (a == 0 && b == 1 && c == 0) return ruleset[5];
    else if (a == 0 && b == 0 && c == 1) return ruleset[6];
    else if (a == 0 && b == 0 && c == 0) return ruleset[7];
    // For this function to be valid, we have to make sure something is returned
    // in cases where the states do not match one of the eight possibilities.
    // We know this is impossible given the rest of our code, but Processing does not.
    return 0;
  }
  
