/**
 * Web interface to the music system.
 */

import fs from 'node:fs';
import path from 'node:path';
import * as esbuild from 'esbuild';
import express from 'express';

const app = express();

const project = 'ex19-music-starting';

app.get('/bundle/:example.js', async function(req, res, next) {
  res.contentType('application/javascript');
  const bundle = await esbuild.build({
    stdin: {
      contents: `
        import JZZ from 'jzz';
        import { Tiny } from 'jzz-synth-tiny';
        Tiny(JZZ);
        JZZ.synth.Tiny.register('Synth'); // use WebAudio-based synthesizer
        await import('./examples/${req.params.example}.js');
      `,
      resolveDir: 'dist',
    },
    bundle: true,
    sourcemap: "inline",
    format: 'esm',
    write: false,
    alias: {
      'node:assert': './node_modules/@jspm/core/nodelibs/browser/assert',
    },
  });
  for (const file of bundle.outputFiles) {
    res.write(file.text);
  }
  res.end();
});

app.get('/', async function(req, res, next) {
  res.contentType('text/html');
  res.write(`
    <html>
    <head><title>${project}</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5/dist/css/bootstrap.min.css" rel="stylesheet">
    </head>
    <body class="container m-5"><h1>${project} examples</h1>
  `);
  try {
    const ext = '.js';
    const files = await fs.promises.readdir(path.join(import.meta.dirname, 'examples'));
    for (const file of files.filter(file => file.endsWith(ext))) {
      const name = path.basename(file, ext);
      res.write(`<p><button>${name}</button></p>`);
    }
  } catch (err) {
    res.write(`<p>${err}</p>`);
  }
  res.end(`
    <script type="module">
      document.querySelectorAll('button').forEach(button => button.addEventListener('click', async () => {
        // import the example dynamically, because the example plays immediately at import time, but
        // browsers want this audio code to be triggered by a user click
        await import('/bundle/' + button.textContent + '.js?doNotCache=' + Date.now());
      }));
    </script>
    </body></html>
  `);
});

const port = 8080;
app.listen(port).on('listening', () => console.log(`${project} web server listening on http://localhost:${port}`));
