// PalmOS.c
// ARM stubs for Palm OS functions

#include "includes.h"
#include "pace.h"


MemPtr MemPtrNew(UInt32 size)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(size)
    };

    return (MemPtr) PACE_EXEC(MemPtrNew, kPceNativeWantA0);
}


MemPtr MemHandleLock(MemHandle h)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(h)
    };

    return (MemPtr) PACE_EXEC(MemHandleLock, kPceNativeWantA0);
}


Err MemHandleUnlock(MemHandle h)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(h)
    };

    return (Err) PACE_EXEC(MemHandleUnlock, 0);
}


UInt32 MemPtrSize(MemPtr p)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(p)
    };

    return (UInt32) PACE_EXEC(MemPtrSize, 0);
}


UInt32 MemHandleSize(MemHandle p)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(p)
    };

    return (UInt32) PACE_EXEC(MemHandleSize, 0);
}


void ClipboardAddItem(const ClipboardFormatType format, const void *ptr, UInt16 length)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD8(format)
        PACE_PARAMS_ADD32(ptr)
        PACE_PARAMS_ADD16(length)
    };

    PACE_EXEC(ClipboardAddItem, 0);
}


MemHandle ClipboardGetItem(const ClipboardFormatType format, UInt16 *length)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD8(format)
        PACE_PARAMS_ADD32(length)
    }; 
    MemHandle result;

    swap16P(length);

    result = (MemHandle) PACE_EXEC(ClipboardGetItem, kPceNativeWantA0);

    swap16P(length);

    return result;
}


FontID FntSetFont(FontID font)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD8(font)
    };

    return (FontID) PACE_EXEC(FntSetFont, 0);
}


void RctGetIntersection(const RectangleType *r1P, const RectangleType *r2P, RectangleType *r3P)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(r1P)
        PACE_PARAMS_ADD32(r2P)
        PACE_PARAMS_ADD32(r3P)
    };

    swapRect(r1P);
    swapRect(r2P);
    swapRect(r3P);

    PACE_EXEC(RctGetIntersection, 0);

    swapRect(r1P);
    swapRect(r2P);
    swapRect(r3P);
}


Err SndDoCmd(void * /*SndChanPtr*/ channelP, SndCommandPtr cmdP, Boolean noWait)
{
    // SndCommandType is badly aligned for ARM. 
    struct {
        uint8_t cmd;
        uint8_t reserved;
        uint8_t param1_a;
        uint8_t param1_b;
        uint8_t param1_c;
        uint8_t param1_d;
        uint16_t param2;
        uint16_t param3;
    } cmd68K;

    PACE_PARAMS = {
        PACE_PARAMS_ADD32(channelP)
        PACE_PARAMS_ADD32(&cmd68K)
        PACE_PARAMS_ADD8(noWait)
    };
    Err result;

    cmd68K.cmd = cmdP->cmd;
    cmd68K.reserved = cmdP->reserved;
    cmd68K.param1_a = (cmdP->param1 & 0xff000000) >> 24;
    cmd68K.param1_b = (cmdP->param1 & 0x00ff0000) >> 16;
    cmd68K.param1_c = (cmdP->param1 & 0x0000ff00) >> 8;
    cmd68K.param1_d = (cmdP->param1 & 0x000000ff) >> 0;
    cmd68K.param2 = swap16(cmdP->param2);
    cmd68K.param3 = swap16(cmdP->param3);

    result = (Err) PACE_EXEC(SndDoCmd, 0);

    // assume SndDoCmd doesn't modify cmdP

    return result;
}


UInt16 SysTicksPerSecond(void)
{
    PACE_PARAMS = {
        0
    };

    return (UInt16) PACE_EXEC(SysTicksPerSecond, 0);
}


UInt32 TimGetTicks(void)
{
    PACE_PARAMS = {
        0
    };

    return (UInt32) PACE_EXEC(TimGetTicks, 0);
}


UInt32 TimGetSeconds(void)
{
    PACE_PARAMS = {
        0
    };

    return (UInt32) PACE_EXEC(TimGetSeconds, 0);
}


void WinDrawChar (WChar theChar, Coord x, Coord y)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD16(theChar)
        PACE_PARAMS_ADD16(x)
        PACE_PARAMS_ADD16(y)
    };

    PACE_EXEC(WinDrawChar, 0);
}


void WinDrawChars (const Char *chars, Int16 len, Coord x, Coord y)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(chars)
        PACE_PARAMS_ADD16(len)
        PACE_PARAMS_ADD16(x)
        PACE_PARAMS_ADD16(y)
    };

    PACE_EXEC(WinDrawChars, 0);
}


void WinEraseRectangle (const RectangleType *rP, UInt16 cornerDiam)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(rP)
        PACE_PARAMS_ADD16(cornerDiam)
    };

    swapRect(rP);
    
    PACE_EXEC(WinEraseRectangle, 0);

    swapRect(rP);
}



void WinInvertRectangle (const RectangleType *rP, UInt16 cornerDiam)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(rP)
        PACE_PARAMS_ADD16(cornerDiam)
    };

    swapRect(rP);
    
    PACE_EXEC(WinInvertRectangle, 0);

    swapRect(rP);
}


void WinPushDrawState (void)
{
    PACE_PARAMS = {
        0
    };

    PACE_EXEC(WinPushDrawState, 0);
}


void WinPopDrawState (void)
{
    PACE_PARAMS = {
        0
    };

    PACE_EXEC(WinPopDrawState, 0);
}


Coord WinScaleCoord(Coord coord, Boolean ceiling)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD16(coord)
        PACE_PARAMS_ADD8(ceiling)
    };

    PACE_EXEC_SELECTOR(HDSelectorWinScaleCoord);
    return (Coord) PACE_EXEC(HighDensityDispatch, 0);
}


void WinScaleRectangle(RectangleType* rectP)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(rectP)
    };

    swapRect(rectP);

    PACE_EXEC_SELECTOR(HDSelectorWinScaleRectangle);
    PACE_EXEC(HighDensityDispatch, 0);

    swapRect(rectP);
}


Err WinScreenGetAttribute(WinScreenAttrType selector, UInt32* attrP)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD8(selector)
        PACE_PARAMS_ADD32(attrP)
    };
    Err result;

    swap32P(attrP);

    PACE_EXEC_SELECTOR(HDSelectorWinScreenGetAttribute);
    result = PACE_EXEC(HighDensityDispatch, 0);

    swap32P(attrP);

    return result;
}


void WinSetForeColorRGB (const RGBColorType* newRgbP, RGBColorType* prevRgbP)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(newRgbP)
        PACE_PARAMS_ADD32(prevRgbP)
    };

    PACE_EXEC(WinSetForeColorRGB, 0);
}


void WinSetBackColorRGB (const RGBColorType* newRgbP, RGBColorType* prevRgbP)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(newRgbP)
        PACE_PARAMS_ADD32(prevRgbP)
    };

    PACE_EXEC(WinSetBackColorRGB, 0);
}


void WinSetTextColorRGB (const RGBColorType* newRgbP, RGBColorType* prevRgbP)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(newRgbP)
        PACE_PARAMS_ADD32(prevRgbP)
    };

    PACE_EXEC(WinSetTextColorRGB, 0);
}


UInt16 	WinSetCoordinateSystem(UInt16 coordSys)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD16(coordSys)
    };

    PACE_EXEC_SELECTOR(HDSelectorWinSetCoordinateSystem);
    return (UInt16) PACE_EXEC(HighDensityDispatch, 0);
}


Int16         StrCaselessCompare(const Char *s1, const Char *s2)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(s1)
        PACE_PARAMS_ADD32(s2)
    };
    
    return (Int16) PACE_EXEC(StrCaselessCompare, 0);
}


Char *StrIToA(Char *s, Int32 i)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(s)
        PACE_PARAMS_ADD32(i)
    };

    return (Char *) PACE_EXEC(StrIToA, kPceNativeWantA0);
}


UInt32 PrefGetPreference(SystemPreferencesChoice choice)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD16(choice)
    };

    return (UInt32) PACE_EXEC(PrefGetPreference, 0);
}


void EvtAddUniqueEventToQueue(const EventType *eventP, UInt32 id, Boolean inPlace)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(eventP)
        PACE_PARAMS_ADD32(id)
        PACE_PARAMS_ADD8(inPlace)
    };

    PACE_EXEC(EvtAddUniqueEventToQueue, 0);
}


Err FtrGet(UInt32 creator, UInt16 featureNum, UInt32 *valueP)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(creator)
        PACE_PARAMS_ADD16(featureNum)
        PACE_PARAMS_ADD32(valueP)
    };
    Err result;

    swap32P(valueP);
    result = (Err) PACE_EXEC(FtrGet, 0);
    swap32P(valueP);

    return result;
}


Err FtrSet(UInt32 creator, UInt16 featureNum, UInt32 value)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(creator)
        PACE_PARAMS_ADD16(featureNum)
        PACE_PARAMS_ADD32(value)
    };

    return (Err) PACE_EXEC(FtrSet, 0);
}


Err FtrPtrNew(UInt32 creator, UInt16 featureNum, UInt32 size, void **newPtrP)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(creator)
        PACE_PARAMS_ADD16(featureNum)
        PACE_PARAMS_ADD32(size)
        PACE_PARAMS_ADD32(newPtrP)
    };
    Err result;

    swap32P(newPtrP);
    result = (Err) PACE_EXEC(FtrPtrNew, 0);
    swap32P(newPtrP);

    return result;
}


Err FtrPtrFree(UInt32 creator, UInt16 featureNum)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(creator)
        PACE_PARAMS_ADD16(featureNum)
    };

    return (Err) PACE_EXEC(FtrPtrFree, 0);
}


UInt16 FrmCustomAlert(UInt16 alertId, const Char *s1,
                      const Char *s2, const Char *s3)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD16(alertId)
        PACE_PARAMS_ADD32(s1)
        PACE_PARAMS_ADD32(s2)
        PACE_PARAMS_ADD32(s3)
    };

    return (UInt16) PACE_EXEC(FrmCustomAlert, 0);
}


Err DmSet(void *recordP, UInt32 offset, UInt32 bytes, UInt8 value)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(recordP)
        PACE_PARAMS_ADD32(offset)
        PACE_PARAMS_ADD32(bytes)
        PACE_PARAMS_ADD8(value)
    };

    return (Err) PACE_EXEC(DmSet, 0);
}


UInt16 DmNumDatabases(UInt16 cardNo)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD16(cardNo)
    };

    return (UInt16) PACE_EXEC(DmNumDatabases, 0);
}


LocalID DmGetDatabase(UInt16 cardNo, UInt16 index)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD16(cardNo)
        PACE_PARAMS_ADD16(index)
    };

    return (LocalID) PACE_EXEC(DmGetDatabase, 0);
}


Err DmDatabaseSize(UInt16 cardNo, LocalID dbID, UInt32 *numRecordsP,
                   UInt32 *totalBytesP, UInt32 *dataBytesP)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD16(cardNo)
        PACE_PARAMS_ADD32(dbID)
        PACE_PARAMS_ADD32(numRecordsP)
        PACE_PARAMS_ADD32(totalBytesP)
        PACE_PARAMS_ADD32(dataBytesP)
    };
    Err result;

    if (numRecordsP) swap32P(numRecordsP);
    if (totalBytesP) swap32P(totalBytesP);
    if (dataBytesP)  swap32P(dataBytesP);

    result = (Err) PACE_EXEC(DmDatabaseSize, 0);

    if (numRecordsP) swap32P(numRecordsP);
    if (totalBytesP) swap32P(totalBytesP);
    if (dataBytesP)  swap32P(dataBytesP);

    return result;
}


Err DmWrite(void *recordP, UInt32 offset, const void *srcP, UInt32 bytes)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(recordP)
        PACE_PARAMS_ADD32(offset)
        PACE_PARAMS_ADD32(srcP)
        PACE_PARAMS_ADD32(bytes)
    };

    return (Err) PACE_EXEC(DmWrite, 0);
}


MemHandle DmGetResource(DmResType type, DmResID resID)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(type)
        PACE_PARAMS_ADD16(resID)
    };
    
    return (MemHandle) PACE_EXEC(DmGetResource, kPceNativeWantA0);
}


UInt16 SysBatteryInfo(Boolean set, UInt16 *warnThresholdP, 
                      UInt16 *criticalThresholdP, Int16 *maxTicksP, 
                      SysBatteryKind* kindP, Boolean *pluggedIn, 
                      UInt8 *percentP)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD8(set)
        PACE_PARAMS_ADD32(warnThresholdP)
        PACE_PARAMS_ADD32(criticalThresholdP)
        PACE_PARAMS_ADD32(maxTicksP)
        PACE_PARAMS_ADD32(kindP)
        PACE_PARAMS_ADD32(pluggedIn)
        PACE_PARAMS_ADD32(percentP)
    };
    UInt16 result;

    if (warnThresholdP) swap16P(warnThresholdP);
    if (criticalThresholdP) swap16P(criticalThresholdP);
    if (maxTicksP) swap16P(maxTicksP);
    // SysBatteryKind, Boolean, and UInt8 are 1 byte each
    
    result = (UInt16) PACE_EXEC(SysBatteryInfo, 0);

    if (warnThresholdP) swap16P(warnThresholdP);
    if (criticalThresholdP) swap16P(criticalThresholdP);
    if (maxTicksP) swap16P(maxTicksP);

    return result;
}
