/**********
 * Copyright (c) 2005 Greg Parker.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY GREG PARKER ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **********/

#include "includes.h"
#include "formutils.h"
#include "data/prefs.h"
#include "data/connectionlist.h"
#include "rsrc/rsrc.h"
#include "ssh/ssh.h"
#include "forms/resize.h"

#include "kbdint0form.h"


static void KbdInt0SendResponses(Boolean ok) FORMS_SEGMENT;
static void LoadKbdInt0Form(void) FORMS_SEGMENT;

extern struct ssh_session_t *ss;


// globals for communication between KbdInt0GetResponses 
// and each form instantiation
static char *KbdInt0Name;
static char *KbdInt0Instructions;


void KbdInt0GetResponses(char *name, char *instructions)
{
    kbdint_log(__PRETTY_FUNCTION__);

    KbdInt0Name = name;
    KbdInt0Instructions = instructions;

    // set a default window title if no name was given
    if (KbdInt0Name[0] == '\0') {
        KbdInt0Name = "Notice";
    }

    // present instructions
    FrmPopupForm(KbdInt0FormID);
}

 
static void KbdInt0SendResponses(Boolean ok)
{
    char *responses[1] = {"unused"};

    kbdint_log(__PRETTY_FUNCTION__);
    
    // send responses to ssh (NULL means user cancel)
    // For non-cancel, send any old non-NULL value.
    ssh_use_kbdint(ss, ok ? responses : NULL);
}


// shorten form by delta pixels
// move every control at and after firstID up; leave others alone
static void ShortenForm(FormPtr frmP, UInt16 firstID, UInt16 delta)
{
    UInt16 id, index;
    RectangleType bounds;
    WinHandle winH;

    kbdint_log(__PRETTY_FUNCTION__);

    // move controls
    for (id = firstID; id <= KbdInt0FormLastID; id++) {
        PrvGetObjectBoundsByID(id, &bounds);
        bounds.topLeft.y -= delta;
        PrvSetObjectBoundsByID(id, &bounds);
    }

    // change form size and position
    winH = WinGetWindowHandle(frmP);
    WinGetBounds(winH, &bounds);
    bounds.topLeft.y += delta;
    bounds.extent.y -= delta;
    WinSetBounds(winH, &bounds);
}


static void LoadKbdInt0Form(void)
{
    FormPtr frmP = FrmGetActiveForm();
    MemHandle recordH;
    char *hostname;
    FieldPtr fieldP;
    RectangleType bounds;
    UInt16 height;

    kbdint_log(__PRETTY_FUNCTION__);

    // Set hostname (all prompts)
    recordH = ConnectionListReadSelectedRecord(&hostname, NULL, NULL);
    if (recordH  &&  hostname) {
        PrvSetFieldToValueByID(KbdInt0FormHostFieldID, hostname);
        MemHandleUnlock(recordH);
    }

    // Set form title (all prompts)
    FrmSetTitle(frmP, KbdInt0Name);

    // Set instructions (all prompts)
    PrvSetFieldToValueByID(KbdInt0FormInstructionFieldID, KbdInt0Instructions);

    // Adjust form height based on instructions length
    fieldP = (FieldPtr)PrvGetObjectByID(KbdInt0FormInstructionFieldID);    
    PrvGetObjectBoundsByID(KbdInt0FormInstructionFieldID, &bounds);
    height = FldGetTextHeight(fieldP);
    if (strlen(KbdInt0Instructions) == 0  ||  height == 0) {
        // Hide scroll bar and instructions
        PrvHideObjectByID(KbdInt0FormInstructionFieldID);
        PrvHideObjectByID(KbdInt0FormInstructionScrollbarID);
        ShortenForm(frmP, KbdInt0FormOKButtonID, bounds.extent.y + 4);
    } else {
        // Resize field and form and update scrollbar 
        PrvSetObjectHeightByID(KbdInt0FormInstructionFieldID, height);
        PrvSetObjectHeightByID(KbdInt0FormInstructionScrollbarID, height);
        ShortenForm(frmP, KbdInt0FormOKButtonID, bounds.extent.y - height);
        PrvUpdateScrollbarForField(KbdInt0FormInstructionFieldID, 
                                   KbdInt0FormInstructionScrollbarID);
    }
}


Boolean KbdInt0FormHandleEvent(EventPtr e)
{
    FormPtr frmP = FrmGetActiveForm();

    if (ResizeHandleEvent(e)) return true;

    kbdint_log(__PRETTY_FUNCTION__);

    switch (e->eType) {
    case frmOpenEvent:
        LoadKbdInt0Form();
        FrmDrawForm(frmP);
        return true;
        
    case keyDownEvent:
        if (!EvtKeydownIsVirtual(e)) {
            unsigned char c = e->data.keyDown.chr;
            if (c == '\n'  ||  c == '\r') {
                // enter - OK button
                KbdInt0SendResponses(true);
                FrmReturnToForm(0);
                return true;
            }
        }
        return false;

    case sclRepeatEvent:
        if (e->data.sclRepeat.scrollBarID == 
            KbdInt0FormInstructionScrollbarID)
        {
            PrvScrollField(KbdInt0FormInstructionFieldID, e);
        }
        return false;

    case ctlSelectEvent:
        switch (e->data.ctlSelect.controlID) {
        case KbdInt0FormOKButtonID:
            KbdInt0SendResponses(true);
            FrmReturnToForm(0);
            return true;

        case KbdInt0FormCancelButtonID:
            KbdInt0SendResponses(false);
            FrmReturnToForm(0);
            return true;

        default:
            return false;
        }

    case frmCloseEvent:
        return false;

    default:
        return false;
    }
}

