/**********
 * Copyright (c) 2004-2005 Greg Parker.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY GREG PARKER ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **********/

#include "includes.h"
#include "formutils.h"
#include "data/prefs.h"
#include "data/connectionlist.h"
#include "rsrc/rsrc.h"
#include "ssh/ssh.h"
#include "forms/resize.h"

#include "kbdintform.h"


static void KbdIntSendResponses(Boolean ok) FORMS_SEGMENT;
static void RetrieveResponse(void) FORMS_SEGMENT;
static void ForgetResponse(void) FORMS_SEGMENT;
static void SetResponseEcho(Boolean echo) FORMS_SEGMENT;
static void LoadKbdIntForm(void) FORMS_SEGMENT;
static void DoNext(void) FORMS_SEGMENT;
static void DoBack(void) FORMS_SEGMENT;

extern struct ssh_session_t *ss;


// globals for communication between KbdIntGetResponses 
// and each form instantiation
static char *KbdIntName;
static char *KbdIntInstructions;
static prompt_t *KbdIntPrompts;
static char **KbdIntResponses;
static int KbdIntPromptCount;
static int KbdIntCurrentPrompt;


void KbdIntGetResponses(char *name, char *instructions, 
                        prompt_t *prompts, int prompt_count)
{
    int i;

    kbdint_log(__PRETTY_FUNCTION__);

    KbdIntName = name;
    KbdIntInstructions = instructions;
    KbdIntPrompts = prompts;
    KbdIntPromptCount = prompt_count;
    KbdIntCurrentPrompt = 0;

    // set a default window title if no name was given
    if (KbdIntName[0] == '\0') {
        KbdIntName = "Log In";
    }

    // prepare empty response list
    KbdIntResponses = arena_malloc(prompt_count * sizeof(char *));
    for (i = 0; i < prompt_count; i++) {
        KbdIntResponses[i] = arena_strdup("");
    }

    // present first prompt form
    FrmPopupForm(KbdIntFormID);
}

 
static void KbdIntSendResponses(Boolean ok)
{
    int i;

    kbdint_log(__PRETTY_FUNCTION__);
    
    // send responses to ssh (NULL means user cancel)
    ssh_use_kbdint(ss, ok ? KbdIntResponses : NULL);

    // clean up
    for (i = 0; i < KbdIntPromptCount; i++) {
        arena_free(KbdIntResponses[i]);
    }
    arena_free(KbdIntResponses);
}


static void RetrieveResponse(void)
{
    FieldPtr responseFld = PrvGetObjectByID(KbdIntFormResponseFieldID);
    MemHandle responseH;
    char *response;

    kbdint_log(__PRETTY_FUNCTION__);

    FldReleaseFocus(responseFld);
    FldCompactText(responseFld);
    responseH = FldGetTextHandle(responseFld);
    FldSetTextHandle(responseFld, NULL);

    if (responseH) {
        response = MemHandleLock(responseH);
    } else {
        response = "";
    }

    arena_free(KbdIntResponses[KbdIntCurrentPrompt]);
    KbdIntResponses[KbdIntCurrentPrompt] = arena_strdup(response);

    if (responseH) {
        MemSet(response, MemHandleSize(responseH), 0);
        MemHandleUnlock(responseH);
        MemHandleFree(responseH);
    }
}


static void ForgetResponse(void)
{
    FieldPtr responseFld = PrvGetObjectByID(KbdIntFormResponseFieldID);
    MemHandle responseH;

    kbdint_log(__PRETTY_FUNCTION__);

    FldReleaseFocus(responseFld);
    FldCompactText(responseFld);
    responseH = FldGetTextHandle(responseFld);
    FldSetTextHandle(responseFld, NULL);

    if (responseH) {
        MemSet(MemHandleLock(responseH), MemHandleSize(responseH), 0);
        MemHandleUnlock(responseH);
        MemHandleFree(responseH);
    }
}


static void SetResponseEcho(Boolean echo)
{
    kbdint_log(__PRETTY_FUNCTION__);

    PrvSetControlValue(KbdIntFormEchoCheckboxID, echo);
    FldSetFont(PrvGetObjectByID(KbdIntFormResponseFieldID), 
               echo ? stdFont : (FontID)PasswordFontID);
}


// shorten form by delta pixels
// move every control at and after firstID up; leave others alone
// fixme always moves Gsi
static void ShortenForm(FormPtr frmP, UInt16 firstID, UInt16 delta)
{
    UInt16 id, index;
    RectangleType bounds;
    WinHandle winH;

    kbdint_log(__PRETTY_FUNCTION__);

    // move controls
    for (id = firstID; id <= KbdIntFormLastID; id++) {
        PrvGetObjectBoundsByID(id, &bounds);
        bounds.topLeft.y -= delta;
        PrvSetObjectBoundsByID(id, &bounds);
    }

    // change form size and position
    winH = WinGetWindowHandle(frmP);
    WinGetBounds(winH, &bounds);
    bounds.topLeft.y += delta;
    bounds.extent.y -= delta;
    WinSetBounds(winH, &bounds);

    // move Gsi (needs to be AFTER win change to prevent misdraw (fixme really?))
    for (index = 0; index < FrmGetNumberOfObjects(frmP); index++) {
        FormObjectKind type = FrmGetObjectType(frmP, index);
        if (type == frmGraffitiStateObj) {
            // PrvMoveGsi() here will cause misdraw!
            Coord x, y;
            FrmGetObjectPosition(frmP, index, &x, &y);
            y -= delta;
            FrmSetObjectPosition(frmP, index, x, y);
            GsiSetLocation(x, y);
            break;
        }
    }
}


static void LoadKbdIntForm(void)
{
    FormPtr frmP = FrmGetActiveForm();
    MemHandle recordH;
    char *hostname;
    FieldPtr fieldP;
    RectangleType bounds;
    UInt16 height;

    kbdint_log(__PRETTY_FUNCTION__);

    // Set hostname (all prompts)
    recordH = ConnectionListReadSelectedRecord(&hostname, NULL, NULL);
    if (recordH  &&  hostname) {
        PrvSetFieldToValueByID(KbdIntFormHostFieldID, hostname);
        MemHandleUnlock(recordH);
    }

    // Set form title (all prompts)
    FrmSetTitle(frmP, KbdIntName);

    // Set instructions (all prompts)
    PrvSetFieldToValueByID(KbdIntFormInstructionFieldID, KbdIntInstructions);

    // Adjust form height based on instructions length
    fieldP = (FieldPtr)PrvGetObjectByID(KbdIntFormInstructionFieldID);    
    PrvGetObjectBoundsByID(KbdIntFormInstructionFieldID, &bounds);
    height = FldGetTextHeight(fieldP);
    if (strlen(KbdIntInstructions) == 0  ||  height == 0) {
        // Hide scroll bar and instructions
        PrvHideObjectByID(KbdIntFormInstructionFieldID);
        PrvHideObjectByID(KbdIntFormInstructionScrollbarID);
        ShortenForm(frmP, KbdIntFormPromptFieldID, bounds.extent.y + 4);
    } else {
        // Resize field and form and update scrollbar 
        PrvSetObjectHeightByID(KbdIntFormInstructionFieldID, height);
        PrvSetObjectHeightByID(KbdIntFormInstructionScrollbarID, height);
        ShortenForm(frmP, KbdIntFormPromptFieldID, bounds.extent.y - height);
        PrvUpdateScrollbarForField(KbdIntFormInstructionFieldID, 
                                   KbdIntFormInstructionScrollbarID);
    }

    // Set prompt (all prompts)
    PrvSetFieldToValueByID(KbdIntFormPromptFieldID, 
                           KbdIntPrompts[KbdIntCurrentPrompt].msg);

    // Set existing response field contents
    PrvSetFieldToValueByID(KbdIntFormResponseFieldID, 
                           KbdIntResponses[KbdIntCurrentPrompt]);

    // Set focus to all of response field (all prompts)
    PrvSetFocusByID(KbdIntFormResponseFieldID);
    PrvFieldSelectAll(KbdIntFormResponseFieldID);

    // Set echo (all prompts)
    // If server prompt specifies non-echo, then set echoing to pref value.
    // Otherwise, enable echoing unconditionally and hide the checkbox.
    if (! KbdIntPrompts[KbdIntCurrentPrompt].echo) {
        SetResponseEcho(PrefsGetInt(prefEchoPassword, 0));
    } else {
        // field's default font is echo
        // fixme shorten form? would need to move GSI
        PrvHideObjectByID(KbdIntFormEchoCheckboxID);
    }

    // Set prompt count field (multiple prompts only)
    // Count field is "X of Y" for X current prompt and Y total prompts
    if (KbdIntPromptCount > 1) {
        char buf[30];
        snprintf(buf, sizeof(buf), "%d of %d", 
                 KbdIntCurrentPrompt+1, KbdIntPromptCount);
        PrvSetFieldToValueByID(KbdIntFormProgressFieldID, buf);
    }

    // Set buttons (all prompts)
    // Note that a single prompt can be both first and last.
    if (KbdIntCurrentPrompt == 0) {
        // First prompt - hide Back button
        PrvHideObjectByID(KbdIntFormBackButtonID);
    }
    if (KbdIntCurrentPrompt == KbdIntPromptCount - 1) {
        // Last prompt - "Next" becomes "OK"
        PrvSetControlLabel(KbdIntFormNextButtonID, "OK");
    }

}


static void DoNext(void) 
{
    kbdint_log(__PRETTY_FUNCTION__);

    // Save this response
    RetrieveResponse();

    // Update current index
    KbdIntCurrentPrompt++;

    if (KbdIntCurrentPrompt >= KbdIntPromptCount) {
        // All done. Send responses to ssh engine and clean up.
        KbdIntSendResponses(true);
        FrmReturnToForm(0);
    }
    else {
        // Kill this form and display the next.
        FrmReturnToForm(0);
        FrmPopupForm(KbdIntFormID);
    }
}


static void DoBack(void)
{
    kbdint_log(__PRETTY_FUNCTION__);

    // Save this response
    RetrieveResponse();

    // Update current index
    if (KbdIntCurrentPrompt == 0) fatal("bad kbdint index!");
    KbdIntCurrentPrompt--;
    
    // Kill this form and display the next
    FrmReturnToForm(0);
    FrmPopupForm(KbdIntFormID);
}


Boolean KbdIntFormHandleEvent(EventPtr e)
{
    FormPtr frmP = FrmGetActiveForm();

    if (ResizeHandleEvent(e)) return true;

    kbdint_log(__PRETTY_FUNCTION__);

    switch (e->eType) {
    case frmOpenEvent:
        LoadKbdIntForm();
        FrmDrawForm(frmP);
        return true;
        
    case keyDownEvent:
        if (!EvtKeydownIsVirtual(e)) {
            unsigned char c = e->data.keyDown.chr;
            if (c == '\n'  ||  c == '\r') {
                // enter in response - Next/OK button
                // enter in nothing - Next/OK button
                DoNext();
                return true;
            }
        }
        return false;

    case sclRepeatEvent:
        if (e->data.sclRepeat.scrollBarID == 
            KbdIntFormInstructionScrollbarID)
        {
            PrvScrollField(KbdIntFormInstructionFieldID, e);
        }
        return false;

    case ctlSelectEvent:
        switch (e->data.ctlSelect.controlID) {
        case KbdIntFormNextButtonID:
            DoNext();
            return true;

        case KbdIntFormBackButtonID:
            DoBack();
            return true;

        case KbdIntFormCancelButtonID:
            KbdIntSendResponses(false);
            FrmReturnToForm(0);
            return true;

        case KbdIntFormEchoCheckboxID:
            // change response field echoing
            PrefsPutInt(prefEchoPassword, e->data.ctlSelect.on);
            SetResponseEcho(e->data.ctlSelect.on);
            return true;

        default:
            return false;
        }

    case frmCloseEvent:
        // clear entered response, if any
        ForgetResponse();
        return false;

    case usrSetFocusEvent:
        PrvReallySetFocus(frmP, e);
        return true;

    default:
        return false;
    }
}

