/**********
 * Portions Copyright (c) 2004 Greg Parker.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY GREG PARKER ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **********/

/*
 * Code for PuTTY to import and export private key files in other
 * SSH clients' formats.
 */

// GrP pssh:
// based on PuTTY's import.c
// changed includes
// replace file I/O with memory operations
// remove export

#include "includes.h"
#include "ctype.h"
#include "openssh/buffer.h"
#include "openssh/bufaux.h"
#include "openssh/key.h"
#include "openssh/openbsd-compat/base64.h"
#include "openssl/evp/evp.h"
#include "openssl/md5/md5.h"

#include "keyimport.h"

static char fromhex(char c) SSH2_SEGMENT;
static int ber_read_id_len(Buffer *source, uint32_t *id, uint32_t *length, int *flags) SSH2_SEGMENT;
static struct openssh_key *load_openssh_key(char *buffer, uint16_t bufferlen) SSH2_SEGMENT;





#define PUT_32BIT(cp, value) do { \
  (cp)[3] = (unsigned char)(value); \
  (cp)[2] = (unsigned char)((value) >> 8); \
  (cp)[1] = (unsigned char)((value) >> 16); \
  (cp)[0] = (unsigned char)((value) >> 24); } while (0)

#define GET_32BIT(cp) \
    (((unsigned long)(unsigned char)(cp)[0] << 24) | \
    ((unsigned long)(unsigned char)(cp)[1] << 16) | \
    ((unsigned long)(unsigned char)(cp)[2] << 8) | \
    ((unsigned long)(unsigned char)(cp)[3]))


/* ----------------------------------------------------------------------
 * Helper routines. (The base64 ones are defined in sshpubk.c.)
 */

#define isbase64(c) (    ((c) >= 'A' && (c) <= 'Z') || \
                         ((c) >= 'a' && (c) <= 'z') || \
                         ((c) >= '0' && (c) <= '9') || \
                         (c) == '+' || (c) == '/' || (c) == '=' \
                         )


// assumes isxdigit(c)
static char fromhex(char c)
{
    if (isdigit(c)) return c - '0';
    else return 10 + (tolower(c) - 'a');
}

/*
 * Read an ASN.1/BER identifier and length pair.
 * 
 * Flags are a combination of the #defines listed below.
 * 
 * Returns 0 on failure, 1 on success
 */

/* ASN.1 tag classes. */
#define ASN1_CLASS_UNIVERSAL        (0 << 6)
#define ASN1_CLASS_APPLICATION      (1 << 6)
#define ASN1_CLASS_CONTEXT_SPECIFIC (2 << 6)
#define ASN1_CLASS_PRIVATE          (3 << 6)
#define ASN1_CLASS_MASK             (3 << 6)

/* Primitive versus constructed bit. */
#define ASN1_CONSTRUCTED            (1 << 5)

static int ber_read_id_len(Buffer *source, uint32_t *id, uint32_t *length, int *flags)
{
    uint8_t c;

#define NEXT(cc) \
    do { \
    if (buffer_len(source) == 0) return 0; \
    cc = buffer_get_char(source); \
    } while (0)

    NEXT(c);

    *flags = (c & 0xE0);
    if ((c & 0x1F) == 0x1F) {
	*id = 0;
	while (c & 0x80) {
	    *id = (*id << 7) | (c & 0x7F);
            NEXT(c);
	}
	*id = (*id << 7) | (c & 0x7F);
    } else {
	*id = c & 0x1F;
    }

    NEXT(c);

    if (c & 0x80) {
	unsigned int n = c & 0x7F;
        if (buffer_len(source) < n) return 0;
	*length = 0;
	while (n--) {
            NEXT(c);
	    *length = (*length << 8) | c;
        }
    } else {
	*length = c;
    }

    return 1;
#undef NEXT
}

#if 0
static int put_string(void *target, void *data, uint32_t len)
{
    unsigned char *d = (unsigned char *)target;

    PUT_32BIT(d, len);
    memcpy(d+4, data, len);
    return len+4;
}

static int put_mp(void *target, void *data, uint32_t len)
{
    unsigned char *d = (unsigned char *)target;
    unsigned char *i = (unsigned char *)data;

    if (*i & 0x80) {
        PUT_32BIT(d, len+1);
        d[4] = 0;
        memcpy(d+5, data, len);
        return len+5;
    } else {
        PUT_32BIT(d, len);
        memcpy(d+4, data, len);
        return len+4;
    }
}

/* Simple structure to point to an mp-int within a blob. */
struct mpint_pos { void *start; int bytes; };

int ssh2_read_mpint(void *data, int len, struct mpint_pos *ret)
{
    int bytes;
    unsigned char *d = (unsigned char *) data;

    if (len < 4)
        goto error;
    bytes = GET_32BIT(d);
    if (len < 4+bytes)
        goto error;

    ret->start = d + 4;
    ret->bytes = bytes;
    return bytes+4;

    error:
    ret->start = NULL;
    ret->bytes = -1;
    return len;                        /* ensure further calls fail as well */
}
#endif

/* ----------------------------------------------------------------------
 * Code to read and write OpenSSH private keys.
 */

struct openssh_key {
    int type;
    int encrypted;
    char iv[32];
    Buffer keyblob;
};


static struct openssh_key *load_openssh_key(char *buffer, uint16_t bufferlen)
{
    struct openssh_key *ret;
    char *errmsg = NULL, *p, *base64_start, *line;
    int base64_size;
    char *buffercopy;

    ret = arena_calloc(sizeof(*ret));

    // strsep mangles data. Don't modify passed-in buffer.
    buffercopy = arena_malloc(bufferlen);
    memcpy(buffercopy, buffer, bufferlen);
    buffer = buffercopy;

    line = strsep(&buffer, "\n");
    if (!line  ||  
	0 != strncmp(line, "-----BEGIN ", 11) ||
	0 != strcmp(line+strlen(line)-16, "PRIVATE KEY-----")) {
	errmsg = "File does not begin with OpenSSH key header";
	goto error;
    }
    if (!strcmp(line, "-----BEGIN RSA PRIVATE KEY-----"))
	ret->type = KEY_RSA;
    else if (!strcmp(line, "-----BEGIN DSA PRIVATE KEY-----"))
	ret->type = KEY_DSA;
    else {
	errmsg = "Unrecognised key type";
	goto error;
    }

    while (1) {
        line = strsep(&buffer, "\n");
        if (!line) {
	    errmsg = "Unexpected end of file";
	    goto error;
        }

        if (0 == strncmp(line, "-----END ", 9)  &&  
            strstr(line, "PRIVATE KEY-----"))
        {
            errmsg = "Key body not present";
            goto error;
        }

	if ((p = strchr(line, ':')) == NULL) {
            // done with headers
            // put line back into buffer
            buffer[-1] = '\n';
            buffer = line;
            break;
        }

        *p++ = '\0'; // replace ':' with '\0'
        while (*p  &&  isspace(*p)) p++;
        if (0 == strcmp(line, "Proc-Type")) {
            if (p[0] != '4' || p[1] != ',') {
                errmsg = "Proc-Type is not 4 (only 4 is supported)";
                goto error;
            }
            p += 2;
            if (0 == strcmp(p, "ENCRYPTED"))
                ret->encrypted = 1;
        }
        else if (0 == strcmp(line, "DEK-Info")) {
            int i;

            if (0 != strncmp(p, "DES-EDE3-CBC,", 13)) {
                // fixme use OpenSSL to support other ciphers
                errmsg = "Ciphers other than DES-EDE3-CBC not supported";
                goto error;
            }
            p += 13;
            // 16 hex digits follow
            for (i = 0; i < 8; i++) {
                if (!isxdigit(p[0])  ||  !isxdigit(p[1])) break;
                ret->iv[i] = (fromhex(p[0]) << 4) | fromhex(p[1]);
                p += 2;
            }
            if (i < 8) {
                errmsg = "Expected 16-digit iv in DEK-Info";
                goto error;
            }
        }
    }

    // done reading headers
    // read base64 data

    if (strchr(buffer, ':')) {
        errmsg = "Header found in body of key data";
        goto error;
    }

    base64_start = buffer;
    while (isspace(*buffer) || isbase64(*buffer)) buffer++;
    // buffer is now the last line, or something invalid
    // But be flexible about garbage after the end.
    if (0 != strncmp(buffer, "-----END ", 9)  ||  
        !strstr(buffer, "PRIVATE KEY-----"))
    {
        // last line is bogus
        errmsg = "Unexpected end of file";
        goto error;
    }

    // [base64_start, buffer) should be valid base64 data

    *buffer = '\0';
    base64_size = strlen(base64_start);

    buffer_init(&ret->keyblob);
    buffer_append_space(&ret->keyblob, base64_size);

    base64_size = b64_pton(base64_start, buffer_ptr(&ret->keyblob), base64_size);
    if (base64_size < 0) {
        errmsg = "Invalid base64 encoding";
        goto error;
    }
    
    if (base64_size == 0) {
	errmsg = "Key body not present";
	goto error;
    }

    if (ret->encrypted && base64_size % 8 != 0) {
	errmsg = "Encrypted key blob is not a multiple of cipher block size";
	goto error;
    }

    // trim buffer
    buffer_consume_end(&ret->keyblob, buffer_len(&ret->keyblob) - base64_size);

    arena_free(buffercopy);
    return ret;

    error:
    // if (errmsg) debug_printf("load_openssh_key %s", errmsg);
    if (ret->keyblob.buf) {
        // fixme clear too?
        buffer_free(&ret->keyblob);
    }
    memset(ret, 0, sizeof(*ret));
    arena_free(ret);
    arena_free(buffercopy);
    return NULL;
}

int openssh_encrypted(char *buffer, uint16_t bufferlen)
{
    struct openssh_key *key = load_openssh_key(buffer, bufferlen);
    int ret;

    if (!key)
	return 0;
    ret = key->encrypted;
    buffer_free(&key->keyblob); // fixme clear too?
    memset(key, 0, sizeof(*key)); // GrP fixme PuTTY bug?
    arena_free(key);
    return ret;
}

Key *openssh_read(char *buffer, uint16_t bufferlen, char *passphrase)
{
    struct openssh_key *key = load_openssh_key(buffer, bufferlen);
    Buffer *keyblob;
    int flags;
    uint32_t id, len;
    char *errmsg = NULL;
    Key *result = NULL;

    if (!key)
	return 0;

    keyblob = &key->keyblob;

    if (key->encrypted) {
	/*
	 * Derive encryption key from passphrase and iv/salt:
	 * 
	 *  - let block A equal MD5(passphrase || iv)
	 *  - let block B equal MD5(A || passphrase || iv)
	 *  - block C would be MD5(B || passphrase || iv) and so on
	 *  - encryption key is the first N bytes of A || B
	 */
	MD5_CTX md5c;
        const EVP_CIPHER *cipher;
        EVP_CIPHER_CTX ctx;
        
	unsigned char keybuf[32];
        uint8_t *kb;
        int written;
        int total;
        Buffer de;
        int decryptOK;

	MD5_Init(&md5c);
	MD5_Update(&md5c, passphrase, strlen(passphrase));
	MD5_Update(&md5c, key->iv, 8);
	MD5_Final(keybuf, &md5c);

	MD5_Init(&md5c);
	MD5_Update(&md5c, keybuf, 16);
	MD5_Update(&md5c, passphrase, strlen(passphrase));
	MD5_Update(&md5c, key->iv, 8);
	MD5_Final(keybuf+16, &md5c);

        // Decrypt keyblob
        cipher = EVP_des_ede3_cbc(); // fixme others

        kb = buffer_ptr(keyblob);
        total = buffer_len(keyblob);
        buffer_init(&de);
        buffer_append_space(&de, total);
        // extra pad needed by EVP_Decrypt
        buffer_append_space(&de, EVP_CIPHER_block_size(cipher));

        EVP_CIPHER_CTX_init(&ctx);
        EVP_DecryptInit_ex(&ctx, cipher, NULL, keybuf, key->iv);
        EVP_DecryptUpdate(&ctx, buffer_ptr(&de), &written, kb, total);
        decryptOK = EVP_DecryptFinal_ex(&ctx, written+(uint8_t *)buffer_ptr(&de), &total);
        EVP_CIPHER_CTX_cleanup(&ctx);

        buffer_free(keyblob); // fixme erase
        buffer_consume_end(&de, EVP_CIPHER_block_size(cipher));
        *keyblob = de;

        memset(keybuf, 0, sizeof(keybuf));
        memset(&md5c, 0, sizeof(md5c));

        if (!decryptOK) {
            // EVP discovered the decryption failure
            errmsg = "Incorrect passphrase";
            goto error;
        }
    }

    /*
     * Now we have a decrypted key blob, which contains an ASN.1
     * encoded private key. We must now untangle the ASN.1.
     *
     * We expect the whole key blob to be formatted as a SEQUENCE
     * (0x30 followed by a length code indicating that the rest of
     * the blob is part of the sequence). Within that SEQUENCE we
     * expect to see a bunch of INTEGERs. What those integers mean
     * depends on the key type:
     *
     *  - For RSA, we expect the integers to be 0, n, e, d, p, q,
     *    dmp1, dmq1, iqmp in that order. (The last three are d mod
     *    (p-1), d mod (q-1), inverse of q mod p respectively.)
     *
     *  - For DSA, we expect them to be 0, p, q, g, y, x in that
     *    order.
     */
    

    /* Expect the SEQUENCE header. Take its absence as a failure to decrypt. */
    if (!ber_read_id_len(keyblob, &id, &len, &flags)  ||  id != 16) {
	errmsg = "ASN.1 decoding failure";
	goto error;
    }

#define DECODE_INTEGER \
    do { \
    if (!ber_read_id_len(keyblob, &id, &len, &flags)  ||   \
        id != 2 || \
        buffer_len(keyblob) < len) { \
        errmsg = "ASN.1 decoding failure"; \
        goto error; \
    } \
    } while (0)

#define FETCH_INTEGER(bn) \
    if (!BN_bin2bn(buffer_ptr(keyblob), len, bn)) { \
        errmsg = "ASN.1 decoding failure"; \
        goto error; \
    } \
    buffer_consume(keyblob, len)

    // Make new keys
    result = key_new_private(key->type);
    
    // First integer: 0
    DECODE_INTEGER;
    if (len != 1 || buffer_get_char(keyblob) != 0) {
        errmsg = "Version number mismatch";
        goto error;
    }

    // Second integer: RSA n (pub), DSA p (pub)
    DECODE_INTEGER;
    if (key->type == KEY_RSA) {
        FETCH_INTEGER(result->rsa->n);
    } else {
        FETCH_INTEGER(result->dsa->p);
    }

    // Third integer: RSA e (pub), DSA q (pub)
    DECODE_INTEGER;
    if (key->type == KEY_RSA) {
        FETCH_INTEGER(result->rsa->e);
    } else {
        FETCH_INTEGER(result->dsa->q);
    }

    // Fourth integer: RSA d (priv), DSA g (pub)
    DECODE_INTEGER;
    if (key->type == KEY_RSA) {
        FETCH_INTEGER(result->rsa->d);
    } else {
        FETCH_INTEGER(result->dsa->g);
    }

    // Fifth integer: RSA p (priv), DSA y (priv)
    DECODE_INTEGER;
    if (key->type == KEY_RSA) {
        FETCH_INTEGER(result->rsa->p);
    } else {
        FETCH_INTEGER(result->dsa->pub_key);
    }

    // Sixth integer: RSA q (priv), DSA x (priv)
    DECODE_INTEGER;
    if (key->type == KEY_RSA) {
        FETCH_INTEGER(result->rsa->q);
    } else {
        FETCH_INTEGER(result->dsa->priv_key);
    }

    if (key->type == KEY_RSA) {
        // Seventh integer: RSA dmp1 (priv)
        DECODE_INTEGER;
        FETCH_INTEGER(result->rsa->dmp1);

        // Eighth integer: RSA dmq1 (priv)
        DECODE_INTEGER;
        FETCH_INTEGER(result->rsa->dmq1);

        // Ninth integer: RSA iqmp (priv)
        DECODE_INTEGER;
        FETCH_INTEGER(result->rsa->iqmp);
    }

#undef DECODE_INTEGER
#undef FETCH_INTEGER

    errmsg = NULL;                     /* no error */

 error:
    buffer_free(&key->keyblob); // fixme wipe
    memset(key, 0, sizeof(*key));
    arena_free(key);
    if (errmsg && result) { key_free(result); result = NULL; }
    return result;
}

/* ----------------------------------------------------------------------
 * Code to read ssh.com private keys.
 */

/*
 * The format of the base64 blob is largely ssh2-packet-formatted,
 * except that mpints are a bit different: they're more like the
 * old ssh1 mpint. You have a 32-bit bit count N, followed by
 * (N+7)/8 bytes of data.
 * 
 * So. The blob contains:
 * 
 *  - uint32 0x3f6ff9eb       (magic number)
 *  - uint32 size             (total blob size)
 *  - string key-type         (see below)
 *  - string cipher-type      (tells you if key is encrypted)
 *  - string encrypted-blob
 * 
 * (The first size field includes the size field itself and the
 * magic number before it. All other size fields are ordinary ssh2
 * strings, so the size field indicates how much data is to
 * _follow_.)
 * 
 * The encrypted blob, once decrypted, contains a single string
 * which in turn contains the payload. (This allows padding to be
 * added after that string while still making it clear where the
 * real payload ends. Also it probably makes for a reasonable
 * decryption check.)
 * 
 * The payload blob, for an RSA key, contains:
 *  - mpint e
 *  - mpint d
 *  - mpint n  (yes, the public and private stuff is intermixed)
 *  - mpint u  (presumably inverse of p mod q)
 *  - mpint p  (p is the smaller prime)
 *  - mpint q  (q is the larger)
 * 
 * For a DSA key, the payload blob contains:
 *  - uint32 0
 *  - mpint p
 *  - mpint g
 *  - mpint q
 *  - mpint y
 *  - mpint x
 * 
 * Alternatively, if the parameters are `predefined', that
 * (0,p,g,q) sequence can be replaced by a uint32 1 and a string
 * containing some predefined parameter specification. *shudder*,
 * but I doubt we'll encounter this in real life.
 * 
 * The key type strings are ghastly. The RSA key I looked at had a
 * type string of
 * 
 *   `if-modn{sign{rsa-pkcs1-sha1},encrypt{rsa-pkcs1v2-oaep}}'
 * 
 * and the DSA key wasn't much better:
 * 
 *   `dl-modp{sign{dsa-nist-sha1},dh{plain}}'
 * 
 * It isn't clear that these will always be the same. I think it
 * might be wise just to look at the `if-modn{sign{rsa' and
 * `dl-modp{sign{dsa' prefixes.
 * 
 * Finally, the encryption. The cipher-type string appears to be
 * either `none' or `3des-cbc'. Looks as if this is SSH2-style
 * 3des-cbc (i.e. outer cbc rather than inner). The key is created
 * from the passphrase by means of yet another hashing faff:
 * 
 *  - first 16 bytes are MD5(passphrase)
 *  - next 16 bytes are MD5(passphrase || first 16 bytes)
 *  - if there were more, they'd be MD5(passphrase || first 32),
 *    and so on.
 */


#if 0
// GrP fixme disabled for now

#define SSHCOM_MAGIC_NUMBER 0x3f6ff9eb

struct sshcom_key {
    char comment[256];                 /* allowing any length is overkill */
    unsigned char *keyblob;
    int keyblob_len, keyblob_size;
};

struct sshcom_key *load_sshcom_key(char *fp)
{
    struct sshcom_key *ret;
    char buffer[256];
    int len;
    char *errmsg, *p;
    int headers_done;
    char base64_bit[4];
    int base64_chars = 0;

    ret = smalloc(sizeof(*ret));
    ret->comment[0] = '\0';
    ret->keyblob = NULL;
    ret->keyblob_len = ret->keyblob_size = 0;

    fp = fopen(filename, "r");
    if (!fp) {
	errmsg = "Unable to open key file";
	goto error;
    }
    if (!sgets(buffer, sizeof(buffer), &fp) ||
	0 != strcmp(buffer, "---- BEGIN SSH2 ENCRYPTED PRIVATE KEY ----\n")) {
	errmsg = "File does not begin with ssh.com key header";
	goto error;
    }

    headers_done = 0;
    while (1) {
	if (!sgets(buffer, sizeof(buffer), &fp)) {
	    errmsg = "Unexpected end of file";
	    goto error;
	}
        if (!strcmp(buffer, "---- END SSH2 ENCRYPTED PRIVATE KEY ----\n"))
            break;                     /* done */
	if ((p = strchr(buffer, ':')) != NULL) {
	    if (headers_done) {
		errmsg = "Header found in body of key data";
		goto error;
	    }
	    *p++ = '\0';
	    while (*p && isspace((unsigned char)*p)) p++;
            /*
             * Header lines can end in a trailing backslash for
             * continuation.
             */
            while ((len = strlen(p)) > sizeof(buffer) - (p-buffer) -1 ||
                   p[len-1] != '\n' || p[len-2] == '\\') {
                if (len > (p-buffer) + sizeof(buffer)-2) {
                    errmsg = "Header line too long to deal with";
                    goto error;
                }
                if (!sgets(p+len-2, sizeof(buffer)-(p-buffer)-(len-2), &fp)) {
                    errmsg = "Unexpected end of file";
                    goto error;
                }
            }
            p[strcspn(p, "\n")] = '\0';
            if (!strcmp(buffer, "Comment")) {
                /* Strip quotes in comment if present. */
                if (p[0] == '"' && p[strlen(p)-1] == '"') {
                    p++;
                    p[strlen(p)-1] = '\0';
                }
                strncpy(ret->comment, p, sizeof(ret->comment));
                ret->comment[sizeof(ret->comment)-1] = '\0';
            }
	} else {
	    headers_done = 1;

	    p = buffer;
	    while (isbase64(*p)) {
                base64_bit[base64_chars++] = *p;
                if (base64_chars == 4) {
                    unsigned char out[3];

                    base64_chars = 0;

                    len = base64_decode_atom(base64_bit, out);

                    if (len <= 0) {
                        errmsg = "Invalid base64 encoding";
                        goto error;
                    }

                    if (ret->keyblob_len + len > ret->keyblob_size) {
                        ret->keyblob_size = ret->keyblob_len + len + 256;
                        ret->keyblob = srealloc(ret->keyblob, ret->keyblob_size);
                    }

                    memcpy(ret->keyblob + ret->keyblob_len, out, len);
                    ret->keyblob_len += len;
                }

		p++;
	    }
	}
    }

    if (ret->keyblob_len == 0 || !ret->keyblob) {
	errmsg = "Key body not present";
	goto error;
    }

    return ret;

    error:
    if (ret) {
	if (ret->keyblob) {
            memset(ret->keyblob, 0, ret->keyblob_size);
            sfree(ret->keyblob);
        }
        memset(&ret, 0, sizeof(ret));
	sfree(ret);
    }
    return NULL;
}

int sshcom_encrypted(char *filename, char **comment)
{
    struct sshcom_key *key = load_sshcom_key(filename);
    int pos, len, answer;

    *comment = NULL;
    if (!key)
        return 0;

    /*
     * Check magic number.
     */
    if (GET_32BIT(key->keyblob) != 0x3f6ff9eb)
        return 0;                      /* key is invalid */

    /*
     * Find the cipher-type string.
     */
    answer = 0;
    pos = 8;
    if (key->keyblob_len < pos+4)
        goto done;                     /* key is far too short */
    pos += 4 + GET_32BIT(key->keyblob + pos);   /* skip key type */
    if (key->keyblob_len < pos+4)
        goto done;                     /* key is far too short */
    len = GET_32BIT(key->keyblob + pos);   /* find cipher-type length */
    if (key->keyblob_len < pos+4+len)
        goto done;                     /* cipher type string is incomplete */
    if (len != 4 || 0 != memcmp(key->keyblob + pos + 4, "none", 4))
        answer = 1;

    done:
    *comment = dupstr(key->comment);
    memset(key->keyblob, 0, key->keyblob_size);
    sfree(key->keyblob);
    memset(&key, 0, sizeof(key));
    sfree(key);
    return answer;
}

int sshcom_read_mpint(void *data, int len, struct mpint_pos *ret)
{
    int bits;
    int bytes;
    unsigned char *d = (unsigned char *) data;

    if (len < 4)
        goto error;
    bits = GET_32BIT(d);

    bytes = (bits + 7) / 8;
    if (len < 4+bytes)
        goto error;

    ret->start = d + 4;
    ret->bytes = bytes;
    return bytes+4;

    error:
    ret->start = NULL;
    ret->bytes = -1;
    return len;                        /* ensure further calls fail as well */
}

static int sshcom_put_mpint(void *target, void *data, int len)
{
    unsigned char *d = (unsigned char *)target;
    unsigned char *i = (unsigned char *)data;
    int bits = len * 8 - 1;

    while (bits > 0) {
	if (*i & (1 << (bits & 7)))
	    break;
	if (!(bits-- & 7))
	    i++, len--;
    }

    PUT_32BIT(d, bits+1);
    memcpy(d+4, i, len);
    return len+4;
}

struct ssh2_userkey *sshcom_read(char *filename, char *passphrase)
{
    struct sshcom_key *key = load_sshcom_key(filename);
    char *errmsg;
    int pos, len;
    const char prefix_rsa[] = "if-modn{sign{rsa";
    const char prefix_dsa[] = "dl-modp{sign{dsa";
    enum { RSA, DSA } type;
    int encrypted;
    char *ciphertext;
    int cipherlen;
    struct ssh2_userkey *ret = NULL, *retkey;
    const struct ssh_signkey *alg;
    unsigned char *blob = NULL;
    int blobsize, publen, privlen;

    if (!key)
        return NULL;

    /*
     * Check magic number.
     */
    if (GET_32BIT(key->keyblob) != SSHCOM_MAGIC_NUMBER) {
        errmsg = "Key does not begin with magic number";
        goto error;
    }

    /*
     * Determine the key type.
     */
    pos = 8;
    if (key->keyblob_len < pos+4 ||
        (len = GET_32BIT(key->keyblob + pos)) > key->keyblob_len - pos - 4) {
        errmsg = "Key blob does not contain a key type string";
        goto error;
    }
    if (len > sizeof(prefix_rsa) - 1 &&
        !memcmp(key->keyblob+pos+4, prefix_rsa, sizeof(prefix_rsa) - 1)) {
        type = RSA;
    } else if (len > sizeof(prefix_dsa) - 1 &&
        !memcmp(key->keyblob+pos+4, prefix_dsa, sizeof(prefix_dsa) - 1)) {
        type = DSA;
    } else {
        errmsg = "Key is of unknown type";
        goto error;
    }
    pos += 4+len;

    /*
     * Determine the cipher type.
     */
    if (key->keyblob_len < pos+4 ||
        (len = GET_32BIT(key->keyblob + pos)) > key->keyblob_len - pos - 4) {
        errmsg = "Key blob does not contain a cipher type string";
        goto error;
    }
    if (len == 4 && !memcmp(key->keyblob+pos+4, "none", 4))
        encrypted = 0;
    else if (len == 8 && !memcmp(key->keyblob+pos+4, "3des-cbc", 8))
        encrypted = 1;
    else {
        errmsg = "Key encryption is of unknown type";
        goto error;
    }
    pos += 4+len;

    /*
     * Get hold of the encrypted part of the key.
     */
    if (key->keyblob_len < pos+4 ||
        (len = GET_32BIT(key->keyblob + pos)) > key->keyblob_len - pos - 4) {
        errmsg = "Key blob does not contain actual key data";
        goto error;
    }
    ciphertext = key->keyblob + pos + 4;
    cipherlen = len;
    if (cipherlen == 0) {
        errmsg = "Length of key data is zero";
        goto error;
    }

    /*
     * Decrypt it if necessary.
     */
    if (encrypted) {
	/*
	 * Derive encryption key from passphrase and iv/salt:
	 * 
	 *  - let block A equal MD5(passphrase)
	 *  - let block B equal MD5(passphrase || A)
	 *  - block C would be MD5(passphrase || A || B) and so on
	 *  - encryption key is the first N bytes of A || B
	 */
	struct MD5Context md5c;
	unsigned char keybuf[32], iv[8];

        if (cipherlen % 8 != 0) {
            errmsg = "Encrypted part of key is not a multiple of cipher block"
                " size";
            goto error;
        }

	MD5Init(&md5c);
	MD5Update(&md5c, passphrase, strlen(passphrase));
	MD5Final(keybuf, &md5c);

	MD5Init(&md5c);
	MD5Update(&md5c, passphrase, strlen(passphrase));
	MD5Update(&md5c, keybuf, 16);
	MD5Final(keybuf+16, &md5c);

	/*
	 * Now decrypt the key blob.
	 */
        memset(iv, 0, sizeof(iv));
	des3_decrypt_pubkey_ossh(keybuf, iv, ciphertext, cipherlen);

        memset(&md5c, 0, sizeof(md5c));
        memset(keybuf, 0, sizeof(keybuf));

        /*
         * Hereafter we return WRONG_PASSPHRASE for any parsing
         * error. (But only if we've just tried to decrypt it!
         * Returning WRONG_PASSPHRASE for an unencrypted key is
         * automatic doom.)
         */
        if (encrypted)
            ret = SSH2_WRONG_PASSPHRASE;
    }

    /*
     * Strip away the containing string to get to the real meat.
     */
    len = GET_32BIT(ciphertext);
    if (len > cipherlen-4) {
        errmsg = "containing string was ill-formed";
        goto error;
    }
    ciphertext += 4;
    cipherlen = len;

    /*
     * Now we break down into RSA versus DSA. In either case we'll
     * construct public and private blobs in our own format, and
     * end up feeding them to alg->createkey().
     */
    blobsize = cipherlen + 256;
    blob = smalloc(blobsize);
    privlen = 0;
    if (type == RSA) {
        struct mpint_pos n, e, d, u, p, q;
        int pos = 0;
        pos += sshcom_read_mpint(ciphertext+pos, cipherlen-pos, &e);
        pos += sshcom_read_mpint(ciphertext+pos, cipherlen-pos, &d);
        pos += sshcom_read_mpint(ciphertext+pos, cipherlen-pos, &n);
        pos += sshcom_read_mpint(ciphertext+pos, cipherlen-pos, &u);
        pos += sshcom_read_mpint(ciphertext+pos, cipherlen-pos, &p);
        pos += sshcom_read_mpint(ciphertext+pos, cipherlen-pos, &q);
        if (!q.start) {
            errmsg = "key data did not contain six integers";
            goto error;
        }

        alg = &ssh_rsa;
        pos = 0;
        pos += put_string(blob+pos, "ssh-rsa", 7);
        pos += put_mp(blob+pos, e.start, e.bytes);
        pos += put_mp(blob+pos, n.start, n.bytes);
        publen = pos;
        pos += put_string(blob+pos, d.start, d.bytes);
        pos += put_mp(blob+pos, q.start, q.bytes);
        pos += put_mp(blob+pos, p.start, p.bytes);
        pos += put_mp(blob+pos, u.start, u.bytes);
        privlen = pos - publen;
    } else if (type == DSA) {
        struct mpint_pos p, q, g, x, y;
        int pos = 4;
        if (GET_32BIT(ciphertext) != 0) {
            errmsg = "predefined DSA parameters not supported";
            goto error;
        }
        pos += sshcom_read_mpint(ciphertext+pos, cipherlen-pos, &p);
        pos += sshcom_read_mpint(ciphertext+pos, cipherlen-pos, &g);
        pos += sshcom_read_mpint(ciphertext+pos, cipherlen-pos, &q);
        pos += sshcom_read_mpint(ciphertext+pos, cipherlen-pos, &y);
        pos += sshcom_read_mpint(ciphertext+pos, cipherlen-pos, &x);
        if (!x.start) {
            errmsg = "key data did not contain five integers";
            goto error;
        }

        alg = &ssh_dss;
        pos = 0;
        pos += put_string(blob+pos, "ssh-dss", 7);
        pos += put_mp(blob+pos, p.start, p.bytes);
        pos += put_mp(blob+pos, q.start, q.bytes);
        pos += put_mp(blob+pos, g.start, g.bytes);
        pos += put_mp(blob+pos, y.start, y.bytes);
        publen = pos;
        pos += put_mp(blob+pos, x.start, x.bytes);
        privlen = pos - publen;
    }

    assert(privlen > 0);	       /* should have bombed by now if not */

    retkey = smalloc(sizeof(struct ssh2_userkey));
    retkey->alg = alg;
    retkey->data = alg->createkey(blob, publen, blob+publen, privlen);
    if (!retkey->data) {
	sfree(retkey);
	errmsg = "unable to create key data structure";
	goto error;
    }
    retkey->comment = dupstr(key->comment);

    errmsg = NULL; /* no error */
    ret = retkey;

    error:
    if (blob) {
        memset(blob, 0, blobsize);
        sfree(blob);
    }
    memset(key->keyblob, 0, key->keyblob_size);
    sfree(key->keyblob);
    memset(&key, 0, sizeof(key));
    sfree(key);
    return ret;
}

#endif


