/**********
 * Copyright (c) 2003-2004 Greg Parker.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY GREG PARKER ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **********/

#include "stdlib.h"

#include "ctype.h"
#include "assert.h"
#include "ssh/ssh.h"
#include "rsrc/rsrc.h"

uint8_t *memchr(uint8_t *buf, uint8_t ch, uint16_t bufLen)
{
    uint8_t *end = buf + bufLen;
    for ( ; buf < end; buf++) {
        if (*buf == ch) return buf;
    }
    return NULL;
}


char *strsep(char **stringp, const char *delim)
{
    char *pos;
    char *str = *stringp;
    if (!str) return NULL;

    // GrP only implemented for single-character delim
    assert(strlen(delim) == 1);

    pos = strchr(str, *delim);
    if (pos) {
        *pos = '\0';
        *stringp = pos+1;
    } else {
        *stringp = NULL;
    }
    return str;
}


int strcasecmp(const char *s1, const char *s2)
{
    return StrCaselessCompare(s1, s2);
}


char *strcasestr(const char *big, const char *little)
{
    int littlelen = strlen(little);

    const char *s = big;
    for (s = big; *s != '\0'; s++) {
        if (0 == memcmp(s, little, littlelen)) return (char *)s;
    }
    return NULL;
}


void printf(const char *format, ...)
{
    char buf[320];
    char *c;
    va_list args;
    struct ssh_session_t *ss = ssh_get_current();

    va_start(args, format);
    vsnprintf(buf, sizeof(buf), format, args);
    va_end(args);

    // Filter out nonprintable chars
    for (c = buf; *c != '\0'; c++) {
        if (!isprint(*c)  &&  !isspace(*c)) *c = '?';
    }

    if (ss) ssh_receive_bytes(ss, buf, strlen(buf));
    // ssh_update(ss);
    // DON'T update here - it can cause a crash 
}

void debug_printf(const char *format, ...)
{
    char buf[320];
    va_list args;
    RectangleType r = {{0, 0}, {160, 14}};
    WinHandle drawWindow;
    int len;

    len = snprintf(buf, sizeof(buf), "%lu ", TimGetTicks());

    va_start(args, format);
    vsnprintf(buf+len, sizeof(buf)-len, format, args);
    va_end(args);

    // print on top of program title box
    // will be visible even with vt100 problems and alerts
    // fixme this causes problems with hi-res drawing
    // drawWindow = WinSetDrawWindow(NULL);
    WinPushDrawState();
    WinEraseRectangle(&r, 0);
    WinDrawChars(buf, strlen(buf), 0, 0);
    WinPopDrawState();
    // WinSetDrawWindow(drawWindow);
}

int fatal(const char *format, ...)
{
    char buf[320];
    va_list args;

    va_start(args, format);
    vsnprintf(buf, sizeof(buf), format, args);
    va_end(args);

    // release version: display alert, then kill program
    FrmCustomAlert(FatalAlertID, buf, " ", " ");
    SysReset();  // fixme don't see a better way to kill the program cleanly

    // development version: don't obscure debug_printf output or vt100
    /*
      {
      struct ssh_session_t *ss = ssh_get_current();
      if (ss) {
      ssh_receive_bytes(ss, "\nFATAL:", 7);
      ssh_receive_bytes(ss, buf, strlen(buf));
      } else {
      debug_printf("%s", buf);
      while(1) { }
      }
    */

    return 0;
}



void assert_failed(uint32_t which)
{
    char intbuf[20];
    FrmCustomAlert(AlertFormID, "arm assert failed!", StrIToA(intbuf, which), " ");
}

