/**********
 * Copyright (c) 2004-2005 Greg Parker.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY GREG PARKER ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **********/

#include "includes.h"
#include "ctype.h"
#include "formutils.h"
#include "forms/passphraseform.h"
#include "forms/resize.h"
#include "data/memolist.h"
#include "data/publickeys.h"
#include "rsrc/rsrc.h"
#include "ssh/keyimport.h"

#include "memoform.h"


static Boolean ImportKeyFromMemo(void) FORMS_SEGMENT;


static Boolean ImportKeyFromMemo(void) 
{
    Boolean ok = true;
    Boolean encrypted;
    char *memoP, *memoEnd;
    char comment[81];
    uint16_t commentLen;
    char *lineEnd, *lineEndR, *lineEndN;
    char *passphrase = NULL;
    Key *key = NULL;
    MemHandle memoH = MemoListQuerySelectedRecord();
    if (!memoH) return false;

    memoP = MemHandleLock(memoH);
    memoEnd = memoP + MemPtrSize(memoP);

    // first line (up to 80 chars) is comment
    // UNLESS it looks like a key instead
    if (0 == strncmp(memoP, "-----BEGIN", 10)) {
        // no comment
        lineEnd = memoP;
        strcpy(comment, "unnamed key");
        commentLen = strlen("unnamed key");
    } else {
        lineEnd = memoEnd;
        lineEndR = strchr(memoP, '\r');
        lineEndN = strchr(memoP, '\n');
        if (lineEndR && lineEndR < lineEnd) lineEnd = lineEndR;
        if (lineEndN && lineEndN < lineEnd) lineEnd = lineEndN;
        commentLen = MIN(sizeof(comment)-1, lineEnd - memoP);
        memcpy(comment, memoP, commentLen);
        comment[commentLen] = '\0';
        while (isspace(*lineEnd)) lineEnd++; // skip any whitespace before key data
    }

    encrypted = openssh_encrypted(lineEnd, memoEnd - lineEnd);

    if (encrypted) {
        passphrase = GetPassphrase(comment);
        // empty passphrase is returned as empty string
        // user cancel is returned as NULL;
        ok = (passphrase != NULL);
    }

    if (ok) {
        key = openssh_read(lineEnd, memoEnd - lineEnd, passphrase);
        ok = (key != NULL);
        if (!ok) {
            FrmCustomAlert(AlertFormID, "Incorrect passphrase, or incorrectly formatted memo", " ", " "); // fixme better diagnostics
        }
    }
    if (ok) {
        ok = SaveImportedPublicKey(encrypted, comment, commentLen, key, 
                                   lineEnd, memoEnd - lineEnd);
        if (!ok) {
            FrmCustomAlert(AlertFormID, "Couldn't save key record", " ", " ");
        }
    }

    if (ok) {
        // fixme
        FrmCustomAlert(AlertFormID, "Successfully imported key '", comment, "'.");
    }

    if (key) key_free(key);
    if (passphrase) arena_free(passphrase);

    MemHandleUnlock(memoH);

    return ok;
}


Boolean MemoFormHandleEvent(EventPtr e)
{
    static int inited = 0;

    FormPtr frmP = FrmGetActiveForm();

    if (ResizeHandleEvent(e)) return true;
    if (MemoListHandleEvent(e)) return true;

    switch (e->eType) {
    case frmOpenEvent:
        if (!inited) {
            MemoListInit(); // allowed to fail == no memos
            inited = 1;
        }
        MemoListUpdate();
        FrmDrawForm(frmP);
        return true;

    case ctlSelectEvent:
        switch (e->data.ctlSelect.controlID) {

        case MemoFormCancelButtonID:
            FrmReturnToForm(0);
            break;

        case MemoFormImportButtonID:
            if (MemoListSelectedIndex() != noRecord) {
                if (ImportKeyFromMemo()) {
                    // successfully got a key - kill this form
                    FrmReturnToForm(0);
                }
            }
            break;

        default: 
            break;

        }
        return false;

    case frmCloseEvent:
        return false;

    default: 
        return false;
    }
}


void MemoFormResize(FormPtr frmP, Int16 dh, Int16 dv)
{
    static Int16 height = -1;
    PrvResizeTableAndScrollbar(frmP, dh, dv, 0, &height, 
                               FrmGetObjectIndex(frmP, MemoFormTableID),
                               FrmGetObjectIndex(frmP, MemoFormScrollbarID));

    MemoListUpdate();
}
