/*
  fortuna_accumulator.c
  Fortuna cryptographic PRNG
  As seen in Ferguson and Schneier "Practical Cryptography"
  Greg Parker     gparker@cs.stanford.edu     2003-11-16
*/
/**********
 * Copyright (c) 2003-2004 Greg Parker.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY GREG PARKER ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **********/

#include "includes.h"

#include "fortuna.h"
#include "fortuna_locl.h"


void accumulator_init(accumulator_t *a)
{
    int i;
    for (i = 0; i < POOL_COUNT; i++) {
        pool_t *pool = &a->pools[i];
        SHA256_Init(&pool->hash);
        pool->bytesBeforeUseful = MIN_POOL_BYTES;
    }
}


void accumulator_add_bytes(accumulator_t *a, uint8_t sourceID, 
                           uint8_t poolID, uint8_t *data, uint8_t len)
{
    pool_t *pool = &a->pools[poolID % POOL_COUNT];
    
    // Record length of addition to this pool
    if (pool->bytesBeforeUseful > 0) {
        if (pool->bytesBeforeUseful > len) {
            pool->bytesBeforeUseful -= len;
        } else {
            pool->bytesBeforeUseful = 0;
        }
    }

    // Add to the pool: sourceID, len, data
    SHA256_Update(&pool->hash, &sourceID, sizeof(sourceID));
    SHA256_Update(&pool->hash, &len, sizeof(len));
    SHA256_Update(&pool->hash, data, len);
}


int accumulator_pool_is_useful(accumulator_t *a, uint8_t poolID)
{
    return (0 == a->pools[poolID % POOL_COUNT].bytesBeforeUseful);
}


void accumulator_extract_pool(accumulator_t *a, uint8_t poolID, 
                              uint8_t *outBytes)
{
    pool_t *pool = &a->pools[poolID % POOL_COUNT];
    
    // Finish current hash
    SHA256_Final(outBytes, &pool->hash);
    
    // Re-hash final pool content into output
    SHA256_Init(&pool->hash);
    SHA256_Update(&pool->hash, outBytes, POOL_BYTES);
    SHA256_Final(outBytes, &pool->hash);

    // Start new hash for pool
    SHA256_Init(&pool->hash);
    pool->bytesBeforeUseful = MIN_POOL_BYTES;
}
