/**********
 * Copyright (c) 2003-2004 Greg Parker.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY GREG PARKER ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **********/

#include "stdlib.h"

#include "pace.h"
#include "ctype.h"
#include "assert.h"
#include "rsrc/rsrc.h"

// 68K function addresses (written by 68K code)
uintptr_t assert_failed_68K = 0;
uintptr_t fatal_68K = 0;

char *strsep(char **stringp, const char *delim)
{
    char *pos;
    char *str = *stringp;
    if (!str) return NULL;

    // GrP only implemented for single-character delim
    assert(strlen(delim) == 1);

    pos = strchr(str, *delim);
    if (pos) {
        *pos = '\0';
        *stringp = pos+1;
    } else {
        *stringp = NULL;
    }
    return str;
}


int strcasecmp(const char *s1, const char *s2)
{
    return StrCaselessCompare(s1, s2);
}


char *strcasestr(const char *big, const char *little)
{
    int littlelen = strlen(little);

    const char *s = big;
    for (s = big; *s != '\0'; s++) {
        if (0 == memcmp(s, little, littlelen)) return (char *)s;
    }
    return NULL;
}


void debug_printf(const char *format, ...)
{
    char buf[320];
    va_list args;
    RectangleType r = {{0, 0}, {160, 14}};
    WinHandle drawWindow;

    va_start(args, format);
    strcpy(buf, "ARM ");
    vsnprintf(buf+4, sizeof(buf)-4, format, args);
    va_end(args);

    // print on top of program title box
    // will be visible even with vt100 problems and alerts
    // fixme this causes problems with hi-res drawing
    // drawWindow = WinSetDrawWindow(NULL);
    WinPushDrawState();
    WinEraseRectangle(&r, 0);
    WinDrawChars(buf, strlen(buf), 0, 0);
    WinPopDrawState();
    // WinSetDrawWindow(drawWindow);
}


void complain(char *err)
{
    WinPushDrawState();
    WinSetCoordinateSystem(kCoordinatesStandard);
    FrmCustomAlert(AlertFormID, err, " ", " ");
    WinPopDrawState();
}

void complain_int(char *err, uint32_t i) 
{
    char intbuf[20];
    WinPushDrawState();
    WinSetCoordinateSystem(kCoordinatesStandard);
    FrmCustomAlert(AlertFormID, err, StrIToA(intbuf, i), " ");
    WinPopDrawState();
}


static int fatal_str(const char *s)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32("%s")
        PACE_PARAMS_ADD32(s)
    };

    return (int) PACE_EXEC_FN(fatal_68K, 0);
}

int fatal(const char *format, ...)
{
    char buf[320];
    va_list args;

    va_start(args, format);
    vsnprintf(buf, sizeof(buf), format, args);
    va_end(args);

    return fatal_str(buf);
}


void assert_failed(uint32_t which)
{
    PACE_PARAMS = {
        PACE_PARAMS_ADD32(which)
    };

    PACE_EXEC_FN(assert_failed_68K, 0);
}

