/**********
 * Copyright (c) 2003-2005 Greg Parker.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY GREG PARKER ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **********/

#include "includes.h"
#include "formutils.h"
#include "data/hostkeys.h"
#include "rsrc/rsrc.h"
#include "ssh/openssh/key.h"
#include "ssh/openssh/uuencode.h"
#include "forms/resize.h"

#include "hostkeydetailsform.h"


static Boolean LoadHostKeyDetailsForm(void) FORMS_SEGMENT;


static Boolean LoadHostKeyDetailsForm(void)
{
    MemHandle recordH;
    Boolean ok = true;
    char *hostnames;
    uint8_t *keyblob;
    uint16_t keybloblen;
    Key *key;

    recordH = HostKeysQuerySelectedRecord(&hostnames, &keyblob, &keybloblen);
    if (!recordH) return false;

    if (ok) ok = PrvSetFieldToValueByID(HostKeyDetailsFormHostFieldID, hostnames);
    
    if (ok) ok = ((key = key_from_blob(keyblob, keybloblen)) != NULL);
    if (ok) ok = PrvSetFieldToValueByID(HostKeyDetailsFormKeyTypeFieldID, key_type(key));
    if (ok) {
        // uuencode(blob) is what OpenSSH uses for known_hosts files
        uint8_t *uubuf = xmalloc(2*keybloblen);
        int uulen = uuencode(keyblob, keybloblen, uubuf, 2*keybloblen);
        if (uulen > 0) {
            ok = PrvSetFieldToValueByID(HostKeyDetailsFormKeyFieldID, uubuf);
        }
        xfree(uubuf);
    }

    PrvUpdateScrollbarForField(HostKeyDetailsFormKeyFieldID, 
                               HostKeyDetailsFormKeyScrollbarID);

    MemHandleUnlock(recordH);

    return ok;    
}


Boolean HostKeyDetailsFormHandleEvent(EventPtr e)
{
    FormPtr frmP = FrmGetActiveForm();

    if (ResizeHandleEvent(e)) return true;

    switch (e->eType) {
    case frmOpenEvent: {
        LoadHostKeyDetailsForm();
        FrmDrawForm(frmP);
        return true;
    }

    case sclRepeatEvent:
        if (e->data.sclRepeat.scrollBarID == HostKeyDetailsFormKeyScrollbarID){
            PrvScrollField(HostKeyDetailsFormKeyFieldID, e);
        }
        return false;

    case ctlSelectEvent:
        switch (e->data.ctlSelect.controlID) {
        case HostKeyDetailsFormOKButtonID:
            FrmReturnToForm(0);
            break;

        case HostKeyDetailsFormDeleteButtonID:
            if (0 == FrmAlert(DeleteHostKeyAlertID)) {
                HostKeysDeleteSelectedRecord();
                FrmReturnToForm(0);
            }
            break;

        default:
            break;
        }

        return true;

    case frmCloseEvent:
        return false;

    default: 
        return false;
    }
}

/*
static Boolean HostKeyDetailsFormResize(FormPtr frmP)
{
    Int16 dh, dv;
    Boolean changed = CollapseResizeForm(frmP, false, &dh, &dv);
    
    if (!changed) return false;

    // key data field: resize to width and height
    // key data scrollbar: resize to width and height
    // host field: resize to width (fixme)
    // key type field: resize to width (fixme)
    // everything else: move vertically

    CollapseMoveResizeFormObject(frmP, HostKeyDetailsFormOKButtonID, dh, dv, 0, 0);
    CollapseMoveResizeFormObject(frmP, HostKeyDetailsFormDeleteButtonID, dh, dv, 0, 0);

    PrvResizeFieldAndScrollbar(frmP, dh, dv, 
                               FrmGetObjectIndex(frmP, HostKeyDetailsFormKeyFieldID),
                               FrmGetObjectIndex(frmP, HostKeyDetailsFormKeyScrollbarID));
    
    return true;
}
*/
