/**********
 * Copyright (c) 2005 Greg Parker.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY GREG PARKER ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **********/

#include "includes.h"
#include "sshzlib.h"
#include "armstubs.h"
#include "peal.h"


static void *zlib_compress_init(void) SSH_SEGMENT;
static void zlib_compress_cleanup(void *ctx) SSH_SEGMENT;
static void *zlib_decompress_init(void) SSH_SEGMENT;
static void zlib_decompress_cleanup(void *ctx) SSH_SEGMENT;
static int zlib_compress_block(void *ctx, unsigned char *block, size_t len,
                               unsigned char **outblock, size_t *outlen) SSH_SEGMENT;
static int zlib_decompress_block(void *ctx, unsigned char *block, size_t len,
                                 unsigned char **outblock, size_t *outlen) SSH_SEGMENT;

static void *zlib_compress_init(void)
{
    static void *sym = NULL;
    if (!sym) sym = PealLookupSymbol(arm_module, __FUNCTION__ "_stub");
    if (!sym) fatal("no ARM symbol %s", __FUNCTION__ "_stub");
    compress_log(__PRETTY_FUNCTION__);

    return (void *) PealCall(arm_module, sym, NULL);
}


static void zlib_compress_cleanup(void *ctx)
{
    uint32_t param[1];
    static void *sym = NULL;
    if (!sym) sym = PealLookupSymbol(arm_module, __FUNCTION__ "_stub");
    if (!sym) fatal("no ARM symbol %s", __FUNCTION__ "_stub");
    compress_log(__PRETTY_FUNCTION__);

    param[0] = (uint32_t)ctx;
    PealCall(arm_module, sym, param);
}


static void *zlib_decompress_init(void)
{
    static void *sym = NULL;
    if (!sym) sym = PealLookupSymbol(arm_module, __FUNCTION__ "_stub");
    if (!sym) fatal("no ARM symbol %s", __FUNCTION__ "_stub");
    compress_log(__PRETTY_FUNCTION__);

    return (void *) PealCall(arm_module, sym, NULL);
}


static void zlib_decompress_cleanup(void *ctx)
{
    uint32_t param[1];
    static void *sym = NULL;
    if (!sym) sym = PealLookupSymbol(arm_module, __FUNCTION__ "_stub");
    if (!sym) fatal("no ARM symbol %s", __FUNCTION__ "_stub");
    compress_log(__PRETTY_FUNCTION__);

    param[0] = (uint32_t)ctx;
    PealCall(arm_module, sym, param);
}


static int zlib_compress_block(void *ctx, unsigned char *block, size_t len,
                               unsigned char **outblock, size_t *outlen)
{
    int result;
    uint32_t param[5];
    static void *sym = NULL;
    if (!sym) sym = PealLookupSymbol(arm_module, __FUNCTION__ "_stub");
    if (!sym) fatal("no ARM symbol %s", __FUNCTION__ "_stub");
    compress_log("%lx %lx %lx %lx %lx", ctx, block, len, outblock, outlen);

    param[0] = (uint32_t)ctx;
    param[1] = (uint32_t)block;
    param[2] = (uint32_t)len;
    param[3] = (uint32_t)outblock;
    param[4] = (uint32_t)outlen;
    result = (int) PealCall(arm_module, sym, param);
    compress_log(__PRETTY_FUNCTION__ " done");
    return result;
}


static int zlib_decompress_block(void *ctx, unsigned char *block, size_t len,
                                 unsigned char **outblock, size_t *outlen)
{
    int result;
    uint32_t param[5];
    static void *sym = NULL;
    if (!sym) sym = PealLookupSymbol(arm_module, __FUNCTION__ "_stub");
    if (!sym) fatal("no ARM symbol %s", __FUNCTION__ "_stub");
    compress_log("%lx %lx %lx %lx %lx", ctx, block, len, outblock, outlen);

    param[0] = (uint32_t)ctx;
    param[1] = (uint32_t)block;
    param[2] = (uint32_t)len;
    param[3] = (uint32_t)outblock;
    param[4] = (uint32_t)outlen;
    result = (int) PealCall(arm_module, sym, param);
    compress_log(__PRETTY_FUNCTION__ " done");
    return result;
}


const struct ssh_compress ssh_zlib_compress = {
    zlib_compress_init,
    zlib_compress_cleanup,
    zlib_compress_block
};

const struct ssh_compress ssh_zlib_decompress = {
    zlib_decompress_init,
    zlib_decompress_cleanup,
    zlib_decompress_block
};

