/**********
 * Copyright (c) 2004-2005 Greg Parker.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY GREG PARKER ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **********/

#include "includes.h"
#include "formutils.h"
#include "data/publickeys.h"
#include "rsrc/rsrc.h"
#include "ssh/openssh/key.h"
#include "ssh/openssh/uuencode.h"
#include "forms/resize.h"

#include "publickeydetailsform.h"


static Boolean LoadPublicKeyDetailsForm(void) FORMS_SEGMENT;
static Boolean SavePublicKeyDetailsForm(void) FORMS_SEGMENT;


static Boolean LoadPublicKeyDetailsForm(void)
{
    Boolean ok = true;
    MemHandle recordH;
    Boolean encrypted;
    char *comment;
    uint8_t *pubkey;
    uint16_t pubkeylen;
    uint8_t *privkey;
    uint16_t privkeylen;
    Key *key;

    recordH = PublicKeysQuerySelectedRecord(&encrypted, &comment, 
                                            &pubkey, &pubkeylen, 
                                            &privkey, &privkeylen);
    if (!recordH) return false;

    if (ok) ok = PrvSetFieldToValueByID(PublicKeyDetailsFormCommentFieldID, comment);
    
    if (ok) ok = ((key = key_from_blob(pubkey, pubkeylen)) != NULL);
    if (ok) {
        char *encrypted_msg = " (encrypted)";
        char *nonencrypted_msg = " (not encrypted)";
        char *buf = arena_malloc(strlen(key_type(key)) + strlen(nonencrypted_msg) + 1);
        strcpy(buf, key_type(key));
        strcat(buf, encrypted ? encrypted_msg : nonencrypted_msg);
        ok = PrvSetFieldToValueByID(PublicKeyDetailsFormKeyTypeFieldID, buf);
        arena_free(buf);
    }
    if (ok) {
        // uuencode(blob) is what OpenSSH uses for public key files
        uint8_t *uubuf = xmalloc(2*pubkeylen);
        int uulen = uuencode(pubkey, pubkeylen, uubuf, 2*pubkeylen);
        if (uulen > 0) {
            ok = PrvSetFieldToValueByID(PublicKeyDetailsFormKeyFieldID, uubuf);
        }
        xfree(uubuf);
    }

    PrvUpdateScrollbarForField(PublicKeyDetailsFormKeyFieldID, 
                               PublicKeyDetailsFormKeyScrollbarID);

    MemHandleUnlock(recordH);

    return ok;
}


static Boolean SavePublicKeyDetailsForm(void) 
{
    Boolean ok = false;
    MemHandle recordH;
    Boolean encrypted;
    char *comment;
    uint8_t *pubkey;
    uint16_t pubkeylen;
    uint8_t *privkey;
    uint16_t privkeylen;
    char *newComment;
    uint8_t *newpubkey;
    uint8_t *newprivkey;
    FieldPtr commentField;
    MemHandle textH;
    uint16_t len;

    recordH = PublicKeysQuerySelectedRecord(&encrypted, &comment, 
                                            &pubkey, &pubkeylen, 
                                            &privkey, &privkeylen);
    if (!recordH) return false;

    commentField = PrvGetObjectByID(PublicKeyDetailsFormCommentFieldID);
    if (! FldDirty(commentField)) {
        // nothing to do - success
        return true;
    }

    // retrieve new comment
    len = FldGetTextLength(commentField);
    newComment = arena_malloc(1+len);
    textH = FldGetTextHandle(commentField);
    memcpy(newComment, MemHandleLock(textH), len);
    MemHandleUnlock(textH);
    newComment[len] = '\0';

    // duplicate old key data
    newpubkey = arena_malloc(pubkeylen);
    memcpy(newpubkey, pubkey, pubkeylen);

    newprivkey = arena_malloc(privkeylen);
    memcpy(newprivkey, privkey, privkeylen);

    // write new record
    MemHandleUnlock(recordH);
    ok = WritePublicKeyRecord(encrypted, newComment, 
                              newpubkey, pubkeylen, 
                              newprivkey, privkeylen);
    
    arena_free(newComment);
    arena_free(newpubkey);
    arena_free(newprivkey);
    
    return true;
}


Boolean PublicKeyDetailsFormHandleEvent(EventPtr e)
{
    FormPtr frmP = FrmGetActiveForm();

    if (ResizeHandleEvent(e)) return true;

    switch (e->eType) {
    case frmOpenEvent: {
        LoadPublicKeyDetailsForm();
        FrmDrawForm(frmP);
        return true;
    }

    case sclRepeatEvent:
        if (e->data.sclRepeat.scrollBarID == 
            PublicKeyDetailsFormKeyScrollbarID)
        {
            PrvScrollField(PublicKeyDetailsFormKeyFieldID, e);
        }
        return false;

    case ctlSelectEvent:
        switch (e->data.ctlSelect.controlID) {
        case PublicKeyDetailsFormOKButtonID:
            if (!SavePublicKeyDetailsForm()) {
                // fixme complain?
            }
            FrmReturnToForm(0);
            break;

        case PublicKeyDetailsFormCancelButtonID:
            FrmReturnToForm(0);
            break;

        case PublicKeyDetailsFormDeleteButtonID:
            if (0 == FrmAlert(DeletePublicKeyAlertID)) {
                PublicKeysDeleteSelectedRecord();
                FrmReturnToForm(0);
            }
            break;

        default:
            break;
        }

        return true;

    case frmCloseEvent:
        return false;

    default: 
        return false;
    }
}
