/* FIR decimation filter with 196 taps and blackman window.  the first two taps on either end
 *  are not used since they are below 1.  tables are integrated and then rounded.  the table
 *  is symmetric, so it is run through backwards for the second half to save on table space.
 *  total memory used is 12*256*2B = 6kB.  it executes in 2.5us on 120MHz M4 with optimizations
 *  on.  gain is set with bitshift at the end.  uses modified I2S library.  the SNR is around
 *  60dB for loud signals, and has a pretty flat passband.  there is a lot of aliasing for
 *  signals up to 2*Fc, as it doesnt have a very sharp rolloff (more taps are required for
 *  that).  decimation by 64 using lookup tables.
 */

#define SAMD51_PDM 1

#include <Arduino.h>

#include <Adafruit_ZeroI2S.h>
#include <math.h>

/* max volume for 32 bit data */
#define VOLUME ( (1UL << 31) - 1)

/* create a buffer for both the left and right channel data */
#define BUFSIZE 1
uint32_t left;
uint32_t right;

const int16_t filter0[256] = {
  39, 37, 36, 34, 33, 32, 31, 29, 31, 30, 28, 27, 26, 25, 23, 22, 28, 27, 25, 24, 23, 22, 20, 19, 20, 19, 17, 16, 15, 14, 12, 11, 25, 24, 22, 21, 20, 19, 17, 16, 17, 16, 14, 13, 12, 11, 9, 8, 14, 13, 12, 10, 9, 8, 6, 5, 7, 6, 4, 3, 2, 0, -1, -2, 22, 21, 19, 18, 17, 16, 14, 13, 14, 13, 11, 10, 9, 8, 6, 5, 11, 10, 9, 7, 6, 5, 3, 2, 4, 3, 1, 0, -1, -3, -4, -5, 8, 7, 5, 4, 3, 2, 0, -1, 1, -1, -2, -3, -4, -6, -7, -8, -2, -3, -5, -6, -7, -8, -10, -11, -10, -11, -13, -14, -15, -16, -18, -19, 19, 18, 16, 15, 14, 13, 11, 10, 11, 10, 8, 7, 6, 5, 3, 2, 8, 7, 6, 4, 3, 2, 1, -1, 1, 0, -2, -3, -4, -5, -7, -8, 5, 4, 3, 1, 0, -1, -3, -4, -2, -3, -5, -6, -7, -9, -10, -11, -5, -6, -8, -9, -10, -11, -13, -14, -13, -14, -16, -17, -18, -19, -21, -22, 2, 1, 0, -2, -3, -4, -6, -7, -5, -6, -8, -9, -10, -12, -13, -14, -8, -9, -11, -12, -13, -14, -16, -17, -16, -17, -19, -20, -21, -22, -24, -25, -11, -12, -14, -15, -16, -17, -19, -20, -19, -20, -22, -23, -24, -25, -27, -28, -22, -23, -25, -26, -27, -28, -30, -31, -29, -31, -32, -33, -34, -36, -37, -39
};
const int16_t filter1[256] = {
  95, 73, 71, 49, 69, 47, 45, 23, 69, 46, 44, 22, 43, 20, 18, -4, 69, 47, 44, 22, 43, 21, 19, -4, 42, 20, 18, -4, 16, -6, -8, -30, 70, 48, 46, 24, 44, 22, 20, -2, 44, 21, 19, -3, 18, -4, -7, -29, 44, 22, 19, -3, 18, -4, -6, -29, 17, -5, -7, -29, -9, -31, -33, -55, 73, 51, 49, 26, 47, 25, 23, 1, 46, 24, 22, 0, 20, -2, -4, -26, 47, 24, 22, 0, 21, -1, -4, -26, 20, -2, -4, -27, -6, -28, -30, -52, 48, 26, 24, 2, 22, 0, -2, -24, 21, -1, -3, -25, -4, -27, -29, -51, 22, -1, -3, -25, -4, -26, -29, -51, -5, -27, -29, -51, -31, -53, -55, -77, 77, 55, 53, 31, 51, 29, 27, 5, 51, 29, 26, 4, 25, 3, 1, -22, 51, 29, 27, 4, 25, 3, 1, -21, 24, 2, 0, -22, -2, -24, -26, -48, 52, 30, 28, 6, 27, 4, 2, -20, 26, 4, 1, -21, 0, -22, -24, -47, 26, 4, 2, -20, 0, -22, -24, -46, -1, -23, -25, -47, -26, -49, -51, -73, 55, 33, 31, 9, 29, 7, 5, -17, 29, 6, 4, -18, 3, -19, -22, -44, 29, 7, 4, -18, 3, -19, -21, -44, 2, -20, -22, -44, -24, -46, -48, -70, 30, 8, 6, -16, 4, -18, -20, -42, 4, -19, -21, -43, -22, -44, -47, -69, 4, -18, -20, -43, -22, -44, -46, -69, -23, -45, -47, -69, -49, -71, -73, -95
};
const int16_t filter2[256] = {
  -135, -146, -138, -150, -128, -140, -132, -143, -115, -127, -119, -131, -109, -120, -112, -124, -100, -112, -104, -115, -93, -105, -97, -109, -81, -92, -84, -96, -74, -86, -78, -89, -82, -94, -86, -97, -75, -87, -79, -91, -63, -74, -66, -78, -56, -68, -60, -71, -48, -59, -51, -63, -41, -53, -44, -56, -28, -40, -32, -43, -21, -33, -25, -37, -61, -73, -65, -76, -54, -66, -58, -70, -42, -53, -45, -57, -35, -47, -39, -50, -26, -38, -30, -42, -20, -31, -23, -35, -7, -19, -11, -22, 0, -12, -4, -16, -8, -20, -12, -24, -2, -13, -5, -17, 11, -1, 7, -4, 18, 6, 14, 2, 26, 14, 23, 11, 33, 21, 29, 18, 45, 34, 42, 30, 52, 40, 49, 37, -37, -49, -40, -52, -30, -42, -34, -45, -18, -29, -21, -33, -11, -23, -14, -26, -2, -14, -6, -18, 4, -7, 1, -11, 17, 5, 13, 2, 24, 12, 20, 8, 16, 4, 12, 0, 22, 11, 19, 7, 35, 23, 31, 20, 42, 30, 38, 26, 50, 39, 47, 35, 57, 45, 53, 42, 70, 58, 66, 54, 76, 65, 73, 61, 37, 25, 33, 21, 43, 32, 40, 28, 56, 44, 53, 41, 63, 51, 59, 48, 71, 60, 68, 56, 78, 66, 74, 63, 91, 79, 87, 75, 97, 86, 94, 82, 89, 78, 86, 74, 96, 84, 92, 81, 109, 97, 105, 93, 115, 104, 112, 100, 124, 112, 120, 109, 131, 119, 127, 115, 143, 132, 140, 128, 150, 138, 146, 135
};
const int16_t filter3[256] = {
  -1048, -922, -891, -766, -856, -731, -700, -574, -817, -692, -661, -536, -626, -501, -470, -344, -775, -649, -618, -493, -583, -458, -427, -301, -545, -419, -388, -263, -353, -228, -197, -71, -728, -602, -571, -446, -536, -411, -380, -254, -497, -372, -341, -216, -306, -181, -150, -24, -455, -329, -298, -173, -263, -138, -107, 19, -224, -99, -68, 57, -33, 92, 123, 249, -676, -551, -520, -394, -485, -359, -328, -203, -446, -321, -290, -164, -255, -129, -98, 27, -403, -278, -247, -121, -212, -86, -55, 70, -173, -48, -17, 109, 18, 144, 175, 300, -356, -231, -200, -74, -165, -39, -8, 117, -126, -1, 31, 156, 65, 191, 222, 347, -83, 42, 73, 199, 108, 234, 265, 390, 147, 272, 303, 429, 338, 464, 495, 620, -620, -495, -464, -338, -429, -303, -272, -147, -390, -265, -234, -108, -199, -73, -42, 83, -347, -222, -191, -65, -156, -31, 1, 126, -117, 8, 39, 165, 74, 200, 231, 356, -300, -175, -144, -18, -109, 17, 48, 173, -70, 55, 86, 212, 121, 247, 278, 403, -27, 98, 129, 255, 164, 290, 321, 446, 203, 328, 359, 485, 394, 520, 551, 676, -249, -123, -92, 33, -57, 68, 99, 224, -19, 107, 138, 263, 173, 298, 329, 455, 24, 150, 181, 306, 216, 341, 372, 497, 254, 380, 411, 536, 446, 571, 602, 728, 71, 197, 228, 353, 263, 388, 419, 545, 301, 427, 458, 583, 493, 618, 649, 775, 344, 470, 501, 626, 536, 661, 692, 817, 574, 700, 731, 856, 766, 891, 922, 1048
};
const int16_t filter4[256] = {
  -3003, -2515, -2450, -1962, -2380, -1892, -1827, -1339, -2305, -1817, -1752, -1264, -1681, -1193, -1128, -640, -2225, -1737, -1672, -1184, -1601, -1114, -1048, -560, -1526, -1039, -973, -485, -903, -415, -350, 138, -2140, -1652, -1587, -1099, -1516, -1029, -963, -475, -1441, -954, -888, -400, -818, -330, -265, 223, -1361, -874, -808, -320, -738, -250, -185, 303, -663, -175, -110, 378, -40, 448, 513, 1001, -2050, -1562, -1497, -1009, -1426, -938, -873, -385, -1351, -863, -798, -310, -728, -240, -175, 313, -1271, -783, -718, -230, -648, -160, -95, 393, -573, -85, -20, 468, 50, 538, 603, 1091, -1186, -699, -633, -145, -563, -75, -10, 478, -488, 0, 65, 553, 135, 623, 688, 1176, -408, 80, 145, 633, 215, 703, 768, 1256, 290, 778, 843, 1331, 913, 1401, 1467, 1955, -1955, -1467, -1401, -913, -1331, -843, -778, -290, -1256, -768, -703, -215, -633, -145, -80, 408, -1176, -688, -623, -135, -553, -65, 0, 488, -478, 10, 75, 563, 145, 633, 699, 1186, -1091, -603, -538, -50, -468, 20, 85, 573, -393, 95, 160, 648, 230, 718, 783, 1271, -313, 175, 240, 728, 310, 798, 863, 1351, 385, 873, 938, 1426, 1009, 1497, 1562, 2050, -1001, -513, -448, 40, -378, 110, 175, 663, -303, 185, 250, 738, 320, 808, 874, 1361, -223, 265, 330, 818, 400, 888, 954, 1441, 475, 963, 1029, 1516, 1099, 1587, 1652, 2140, -138, 350, 415, 903, 485, 973, 1039, 1526, 560, 1048, 1114, 1601, 1184, 1672, 1737, 2225, 640, 1128, 1193, 1681, 1264, 1752, 1817, 2305, 1339, 1827, 1892, 2380, 1962, 2450, 2515, 3003
};
const int16_t filter5[256] = {
  -6206, -5057, -4952, -3804, -4842, -3693, -3588, -2439, -4727, -3578, -3473, -2324, -3363, -2214, -2109, -960, -4607, -3458, -3353, -2204, -3242, -2094, -1989, -840, -3127, -1979, -1873, -725, -1763, -615, -509, 639, -4482, -3333, -3228, -2079, -3118, -1969, -1864, -715, -3002, -1854, -1748, -600, -1638, -490, -384, 764, -2882, -1734, -1628, -480, -1518, -369, -264, 884, -1403, -254, -149, 1000, -39, 1110, 1215, 2364, -4352, -3203, -3098, -1950, -2988, -1839, -1734, -585, -2873, -1724, -1619, -470, -1509, -360, -255, 894, -2753, -1604, -1499, -350, -1388, -240, -135, 1014, -1273, -125, -19, 1129, 91, 1240, 1345, 2493, -2628, -1479, -1374, -225, -1263, -115, -10, 1139, -1148, 0, 106, 1254, 216, 1364, 1470, 2618, -1028, 120, 226, 1374, 336, 1485, 1590, 2738, 451, 1600, 1705, 2854, 1815, 2964, 3069, 4218, -4218, -3069, -2964, -1815, -2854, -1705, -1600, -451, -2738, -1590, -1485, -336, -1374, -226, -120, 1028, -2618, -1470, -1364, -216, -1254, -106, 0, 1148, -1139, 10, 115, 1263, 225, 1374, 1479, 2628, -2493, -1345, -1240, -91, -1129, 19, 125, 1273, -1014, 135, 240, 1388, 350, 1499, 1604, 2753, -894, 255, 360, 1509, 470, 1619, 1724, 2873, 585, 1734, 1839, 2988, 1950, 3098, 3203, 4352, -2364, -1215, -1110, 39, -1000, 149, 254, 1403, -884, 264, 369, 1518, 480, 1628, 1734, 2882, -764, 384, 490, 1638, 600, 1748, 1854, 3002, 715, 1864, 1969, 3118, 2079, 3228, 3333, 4482, -639, 509, 615, 1763, 725, 1873, 1979, 3127, 840, 1989, 2094, 3242, 2204, 3353, 3458, 4607, 960, 2109, 2214, 3363, 2324, 3473, 3578, 4727, 2439, 3588, 3693, 4842, 3804, 4952, 5057, 6206
};
const int16_t filter6[256] = {
  -10616, -8489, -8346, -6219, -8199, -6072, -5929, -3802, -8048, -5921, -5778, -3651, -5631, -3504, -3361, -1234, -7893, -5766, -5624, -3497, -5477, -3350, -3207, -1080, -5326, -3199, -3056, -929, -2909, -782, -639, 1488, -7735, -5608, -5466, -3339, -5319, -3192, -3049, -922, -5168, -3041, -2898, -771, -2751, -624, -481, 1646, -5013, -2886, -2743, -616, -2596, -469, -326, 1801, -2445, -318, -175, 1951, -28, 2098, 2241, 4368, -7574, -5447, -5304, -3177, -5157, -3030, -2887, -761, -5006, -2879, -2737, -610, -2590, -463, -320, 1807, -4852, -2725, -2582, -455, -2435, -308, -165, 1962, -2284, -157, -14, 2113, 133, 2260, 2403, 4530, -4694, -2567, -2424, -297, -2277, -150, -7, 2120, -2126, 1, 144, 2271, 291, 2418, 2561, 4688, -1971, 156, 298, 2425, 445, 2572, 2715, 4842, 596, 2723, 2866, 4993, 3013, 5140, 5283, 7410, -7410, -5283, -5140, -3013, -4993, -2866, -2723, -596, -4842, -2715, -2572, -445, -2425, -298, -156, 1971, -4688, -2561, -2418, -291, -2271, -144, -1, 2126, -2120, 7, 150, 2277, 297, 2424, 2567, 4694, -4530, -2403, -2260, -133, -2113, 14, 157, 2284, -1962, 165, 308, 2435, 455, 2582, 2725, 4852, -1807, 320, 463, 2590, 610, 2737, 2879, 5006, 761, 2887, 3030, 5157, 3177, 5304, 5447, 7574, -4368, -2241, -2098, 28, -1951, 175, 318, 2445, -1801, 326, 469, 2596, 616, 2743, 2886, 5013, -1646, 481, 624, 2751, 771, 2898, 3041, 5168, 922, 3049, 3192, 5319, 3339, 5466, 5608, 7735, -1488, 639, 782, 2909, 929, 3056, 3199, 5326, 1080, 3207, 3350, 5477, 3497, 5624, 5766, 7893, 1234, 3361, 3504, 5631, 3651, 5778, 5921, 8048, 3802, 5929, 6072, 8199, 6219, 8346, 8489, 10616
};
const int16_t filter7[256] = {
  -15906, -12534, -12364, -8992, -12193, -8821, -8651, -5279, -12020, -8647, -8478, -5106, -8307, -4934, -4765, -1392, -11846, -8473, -8304, -4931, -8132, -4760, -4591, -1218, -7959, -4587, -4418, -1045, -4246, -874, -704, 2668, -11670, -8297, -8128, -4756, -7957, -4584, -4415, -1042, -7784, -4411, -4242, -869, -4071, -698, -529, 2844, -7609, -4237, -4068, -695, -3896, -524, -354, 3018, -3723, -350, -181, 3191, -10, 3363, 3532, 6905, -11494, -8121, -7952, -4579, -7781, -4408, -4239, -866, -7608, -4235, -4066, -693, -3894, -522, -353, 3020, -7433, -4060, -3891, -519, -3720, -347, -178, 3195, -3547, -174, -5, 3368, 166, 3539, 3708, 7081, -7258, -3885, -3716, -343, -3544, -172, -2, 3370, -3371, 1, 171, 3543, 342, 3715, 3884, 7256, -3197, 176, 345, 3718, 516, 3889, 4058, 7431, 689, 4062, 4231, 7604, 4403, 7775, 7944, 11317, -11317, -7944, -7775, -4403, -7604, -4231, -4062, -689, -7431, -4058, -3889, -516, -3718, -345, -176, 3197, -7256, -3884, -3715, -342, -3543, -171, -1, 3371, -3370, 2, 172, 3544, 343, 3716, 3885, 7258, -7081, -3708, -3539, -166, -3368, 5, 174, 3547, -3195, 178, 347, 3720, 519, 3891, 4060, 7433, -3020, 353, 522, 3894, 693, 4066, 4235, 7608, 866, 4239, 4408, 7781, 4579, 7952, 8121, 11494, -6905, -3532, -3363, 10, -3191, 181, 350, 3723, -3018, 354, 524, 3896, 695, 4068, 4237, 7609, -2844, 529, 698, 4071, 869, 4242, 4411, 7784, 1042, 4415, 4584, 7957, 4756, 8128, 8297, 11670, -2668, 704, 874, 4246, 1045, 4418, 4587, 7959, 1218, 4591, 4760, 8132, 4931, 8304, 8473, 11846, 1392, 4765, 4934, 8307, 5106, 8478, 8647, 12020, 5279, 8651, 8821, 12193, 8992, 12364, 12534, 15906
};
const int16_t filter8[256] = {
  -21498, -16732, -16556, -11790, -16380, -11614, -11438, -6672, -16206, -11440, -11263, -6498, -11088, -6322, -6146, -1380, -16033, -11267, -11091, -6325, -10915, -6149, -5973, -1207, -10740, -5975, -5798, -1032, -5623, -857, -680, 4085, -15862, -11096, -10919, -6154, -10744, -5978, -5802, -1036, -10569, -5804, -5627, -861, -5452, -686, -509, 4257, -10396, -5631, -5454, -688, -5279, -513, -336, 4429, -5104, -338, -162, 4604, 14, 4780, 4956, 9722, -15693, -10927, -10751, -5985, -10575, -5809, -5633, -867, -10401, -5635, -5458, -692, -5283, -517, -341, 4425, -10228, -5462, -5285, -520, -5110, -344, -168, 4598, -4935, -170, 7, 4773, 182, 4948, 5125, 9891, -10057, -5291, -5114, -348, -4939, -173, 3, 4769, -4764, 2, 178, 4944, 353, 5119, 5296, 10062, -4591, 174, 351, 5117, 526, 5292, 5469, 10235, 701, 5467, 5643, 10409, 5819, 10585, 10761, 15527, -15527, -10761, -10585, -5819, -10409, -5643, -5467, -701, -10235, -5469, -5292, -526, -5117, -351, -174, 4591, -10062, -5296, -5119, -353, -4944, -178, -2, 4764, -4769, -3, 173, 4939, 348, 5114, 5291, 10057, -9891, -5125, -4948, -182, -4773, -7, 170, 4935, -4598, 168, 344, 5110, 520, 5285, 5462, 10228, -4425, 341, 517, 5283, 692, 5458, 5635, 10401, 867, 5633, 5809, 10575, 5985, 10751, 10927, 15693, -9722, -4956, -4780, -14, -4604, 162, 338, 5104, -4429, 336, 513, 5279, 688, 5454, 5631, 10396, -4257, 509, 686, 5452, 861, 5627, 5804, 10569, 1036, 5802, 5978, 10744, 6154, 10919, 11096, 15862, -4085, 680, 857, 5623, 1032, 5798, 5975, 10740, 1207, 5973, 6149, 10915, 6325, 11091, 11267, 16033, 1380, 6146, 6322, 11088, 6498, 11263, 11440, 16206, 6672, 11438, 11614, 16380, 11790, 16556, 16732, 21498
};
const int16_t filter9[256] = {
  -26652, -20518, -20358, -14224, -20202, -14068, -13909, -7774, -20051, -13917, -13757, -7623, -13601, -7467, -7307, -1173, -19904, -13769, -13610, -7476, -13454, -7320, -7160, -1026, -13302, -7168, -7009, -874, -6853, -719, -559, 5575, -19761, -13627, -13468, -7333, -13312, -7178, -7018, -884, -13160, -7026, -6866, -732, -6711, -576, -417, 5717, -13013, -6879, -6719, -585, -6564, -429, -270, 5865, -6412, -278, -118, 6016, 38, 6172, 6332, 12466, -19625, -13490, -13331, -7197, -13175, -7041, -6881, -747, -13023, -6889, -6729, -595, -6574, -439, -280, 5854, -12876, -6742, -6582, -448, -6427, -292, -133, 6001, -6275, -141, 19, 6153, 175, 6309, 6468, 12603, -12734, -6600, -6440, -306, -6284, -150, 9, 6144, -6133, 1, 161, 6295, 317, 6451, 6611, 12745, -5986, 148, 308, 6442, 464, 6598, 6758, 12892, 616, 6750, 6909, 13044, 7065, 13199, 13359, 19493, -19493, -13359, -13199, -7065, -13044, -6909, -6750, -616, -12892, -6758, -6598, -464, -6442, -308, -148, 5986, -12745, -6611, -6451, -317, -6295, -161, -1, 6133, -6144, -9, 150, 6284, 306, 6440, 6600, 12734, -12603, -6468, -6309, -175, -6153, -19, 141, 6275, -6001, 133, 292, 6427, 448, 6582, 6742, 12876, -5854, 280, 439, 6574, 595, 6729, 6889, 13023, 747, 6881, 7041, 13175, 7197, 13331, 13490, 19625, -12466, -6332, -6172, -38, -6016, 118, 278, 6412, -5865, 270, 429, 6564, 585, 6719, 6879, 13013, -5717, 417, 576, 6711, 732, 6866, 7026, 13160, 884, 7018, 7178, 13312, 7333, 13468, 13627, 19761, -5575, 559, 719, 6853, 874, 7009, 7168, 13302, 1026, 7160, 7320, 13454, 7476, 13610, 13769, 19904, 1173, 7307, 7467, 13601, 7623, 13757, 13917, 20051, 7774, 13909, 14068, 20202, 14224, 20358, 20518, 26652
};
const int16_t filter10[256] = {
  -30614, -23329, -23210, -15926, -23097, -15813, -15694, -8410, -22992, -15707, -15588, -8304, -15476, -8191, -8072, -788, -22893, -15609, -15489, -8205, -15377, -8093, -7973, -689, -15271, -7987, -7867, -583, -7755, -471, -351, 6933, -22801, -15517, -15398, -8114, -15285, -8001, -7882, -598, -15179, -7895, -7776, -492, -7663, -379, -260, 7024, -15081, -7796, -7677, -393, -7564, -280, -161, 7123, -7459, -174, -55, 7229, 57, 7342, 7461, 14745, -22717, -15433, -15314, -8030, -15201, -7917, -7798, -514, -15095, -7811, -7692, -408, -7579, -295, -176, 7108, -14997, -7712, -7593, -309, -7481, -196, -77, 7207, -7375, -90, 29, 7313, 141, 7426, 7545, 14829, -14905, -7621, -7502, -218, -7389, -105, 14, 7299, -7283, 1, 120, 7404, 233, 7517, 7636, 14920, -7184, 100, 219, 7503, 332, 7616, 7735, 15019, 438, 7722, 7841, 15125, 7954, 15238, 15357, 22641, -22641, -15357, -15238, -7954, -15125, -7841, -7722, -438, -15019, -7735, -7616, -332, -7503, -219, -100, 7184, -14920, -7636, -7517, -233, -7404, -120, -1, 7283, -7299, -14, 105, 7389, 218, 7502, 7621, 14905, -14829, -7545, -7426, -141, -7313, -29, 90, 7375, -7207, 77, 196, 7481, 309, 7593, 7712, 14997, -7108, 176, 295, 7579, 408, 7692, 7811, 15095, 514, 7798, 7917, 15201, 8030, 15314, 15433, 22717, -14745, -7461, -7342, -57, -7229, 55, 174, 7459, -7123, 161, 280, 7564, 393, 7677, 7796, 15081, -7024, 260, 379, 7663, 492, 7776, 7895, 15179, 598, 7882, 8001, 15285, 8114, 15398, 15517, 22801, -6933, 351, 471, 7755, 583, 7867, 7987, 15271, 689, 7973, 8093, 15377, 8205, 15489, 15609, 22893, 788, 8072, 8191, 15476, 8304, 15588, 15707, 22992, 8410, 15694, 15813, 23097, 15926, 23210, 23329, 30614
};
const int16_t filter11[256] = {
  -32767, -24726, -24666, -16626, -24615, -16574, -16514, -8473, -24571, -16531, -16471, -8430, -16419, -8378, -8318, -278, -24536, -16496, -16436, -8395, -16384, -8343, -8283, -243, -16341, -8300, -8240, -199, -8188, -148, -88, 7953, -24510, -16470, -16410, -8369, -16358, -8317, -8257, -217, -16314, -8274, -8214, -173, -8162, -122, -61, 7979, -16280, -8239, -8179, -138, -8127, -87, -27, 8014, -8084, -43, 17, 8057, 69, 8109, 8169, 16210, -24493, -16452, -16392, -8352, -16340, -8300, -8240, -199, -16297, -8256, -8196, -156, -8145, -104, -44, 7997, -16262, -8222, -8162, -121, -8110, -69, -9, 8031, -8066, -26, 34, 8075, 86, 8127, 8187, 16227, -16236, -8195, -8135, -95, -8084, -43, 17, 8058, -8040, 0, 60, 8101, 112, 8153, 8213, 16253, -8005, 35, 95, 8136, 147, 8188, 8248, 16288, 190, 8231, 8291, 16332, 8343, 16383, 16443, 24484, -24484, -16443, -16383, -8343, -16332, -8291, -8231, -190, -16288, -8248, -8188, -147, -8136, -95, -35, 8005, -16253, -8213, -8153, -112, -8101, -60, 0, 8040, -8058, -17, 43, 8084, 95, 8135, 8195, 16236, -16227, -8187, -8127, -86, -8075, -34, 26, 8066, -8031, 9, 69, 8110, 121, 8162, 8222, 16262, -7997, 44, 104, 8145, 156, 8196, 8256, 16297, 199, 8240, 8300, 16340, 8352, 16392, 16452, 24493, -16210, -8169, -8109, -69, -8057, -17, 43, 8084, -8014, 27, 87, 8127, 138, 8179, 8239, 16280, -7979, 61, 122, 8162, 173, 8214, 8274, 16314, 217, 8257, 8317, 16358, 8369, 16410, 16470, 24510, -7953, 88, 148, 8188, 199, 8240, 8300, 16341, 243, 8283, 8343, 16384, 8395, 16436, 16496, 24536, 278, 8318, 8378, 16419, 8430, 16471, 16531, 24571, 8473, 16514, 16574, 24615, 16626, 24666, 24726, 32767
};


// Use default pins in board variant
Adafruit_ZeroI2S i2s = Adafruit_ZeroI2S();

const int16_t *table[12] = {filter0, filter1, filter2, filter3, filter4, filter5, filter6, filter7, filter8, filter9, filter10, filter11};
const uint8_t reverse[256] = {
  0, 128, 64, 192, 32, 160, 96, 224, 16, 144, 80, 208, 48, 176, 112, 240, 8, 136, 72, 200, 40, 168, 104, 232, 24, 152, 88, 216, 56, 184, 120, 248, 4, 132, 68, 196, 36, 164, 100, 228, 20, 148, 84, 212, 52, 180, 116, 244, 12, 140, 76, 204, 44, 172, 108, 236, 28, 156, 92, 220, 60, 188, 124, 252, 2, 130, 66, 194, 34, 162, 98, 226, 18, 146, 82, 210, 50, 178, 114, 242, 10, 138, 74, 202, 42, 170, 106, 234, 26, 154, 90, 218, 58, 186, 122, 250, 6, 134, 70, 198, 38, 166, 102, 230, 22, 150, 86, 214, 54, 182, 118, 246, 14, 142, 78, 206, 46, 174, 110, 238, 30, 158, 94, 222, 62, 190, 126, 254, 1, 129, 65, 193, 33, 161, 97, 225, 17, 145, 81, 209, 49, 177, 113, 241, 9, 137, 73, 201, 41, 169, 105, 233, 25, 153, 89, 217, 57, 185, 121, 249, 5, 133, 69, 197, 37, 165, 101, 229, 21, 149, 85, 213, 53, 181, 117, 245, 13, 141, 77, 205, 45, 173, 109, 237, 29, 157, 93, 221, 61, 189, 125, 253, 3, 131, 67, 195, 35, 163, 99, 227, 19, 147, 83, 211, 51, 179, 115, 243, 11, 139, 75, 203, 43, 171, 107, 235, 27, 155, 91, 219, 59, 187, 123, 251, 7, 135, 71, 199, 39, 167, 103, 231, 23, 151, 87, 215, 55, 183, 119, 247, 15, 143, 79, 207, 47, 175, 111, 239, 31, 159, 95, 223, 63, 191, 127, 255
};

void setup() {

  i2s.begin(I2S_32_BIT, 44100);
  i2s.enableRx();
  pinMode(9,OUTPUT); // test pin
}

byte data = 0;
int32_t accumulator = 0;
uint32_t pastvalues[6] = {0};
byte pointer = 0;

void loop() {
 
    while(!(i2s.rxReady())) ; // wait for data
    i2s.read(&left, &right); // fetch a word (only fills left), LSB first (LSB is older data than MSB)
    
digitalWrite(9,1); // speed test

    pastvalues[pointer] = left; // store data in 6 word array
    pointer++;
    if (pointer == 6) pointer = 0;

    data--; // do decimation evey other word for /64
    if (data == 0) {
      for (byte m = 0; m < 3; m++) { // do forward half
        byte j = pointer + m;
        if (j >= 6) j -= 6;
        uint32_t temp = pastvalues[j];
        byte n = (m << 2);
        accumulator += table[n][(temp & 0xff)];
        temp >>= 8;
        accumulator += table[n+1][(temp & 0xff)];
        temp >>= 8;
        accumulator += table[n+2][(temp & 0xff)];
        temp >>= 8;
        accumulator += table[n+3][(temp & 0xff)];
      }
      for (byte m = 3; m < 6; m++) { // do backward half
        byte j = pointer + m;
        if (j >= 6) j -= 6;
        uint32_t temp = pastvalues[j];
        byte n = (20 - (m << 2));
        accumulator += table[n+3][reverse[(temp & 0xff)]];
        temp >>= 8;
        accumulator += table[n+2][reverse[(temp & 0xff)]];
        temp >>= 8;
        accumulator += table[n+1][reverse[(temp & 0xff)]];
        temp >>= 8;
        accumulator += table[n][reverse[(temp & 0xff)]];
      } 

digitalWrite(9,0); // speed test

      analogWrite(A0, ((accumulator >> 4) + 0x0800)); // play data
      accumulator = 0;
      data = 2;
    }
    
digitalWrite(9,0); // speed test

}
