 
PROTOCOL.C 0            @@/          00   @@TEXTMACA       Va  ȭ:lY                         /* Protocol.c
*
*  Written by: Bill Worzel
*
*  Copyright:   1995-96 by Apple Computer, Inc.  All rights reserved.
*
* This file is used both by the Macintosh and Windows versions of DILArchive; it contains
* nearly all the calls to the CDIL code other than the initialization routines. It is based on
* the SoupDrink example written by Rob Langhorne and David Fedor of Apple Computer.
*
* *
* Purpose:
*    To demonstrate the basic parts of CDILs based on the DILArchive transport protocol.
* The Archive function is the main call for getting data being archived or dearchived
* from the Newton PDA or sending previously archived data from the desktop machine to the Newton PDA.
* Note that the added "End " in the receiving protocol is due to the fact that the Newton PDA prefers
* to disconnect first.
*
* The protocol for DILArchive is as follows (direction is from Newton PDA):
*
*                      Sending:
*                          <connect>
*                          Newton->desktop "Sending"
*                          Newton->desktop <data format>
*                          desktop->Newton "READY"
*                          Newton->desktop <length> (of string)
*                          Newton->desktop <data>
*                          desktop->Newton "CONFIRMED" or "ABORT"
*                          Newton->desktop "End " or "More" or "Abrt"
*                          <disconnect>
*
*                      Receiving:
*                          <connect>
*                          Newton->desktop "Receive"
*                          desktop->Newton "READY FOR <data format>"
*                          Newton->desktop "Ready"
*                          desktop->Newton <null-terminated string>
*                          Newton->desktop "Conf" or "Abrt"
*                          desktop->Newton "END ARCHIVE" or "MORE" or "ABORT"
*                          Newton->desktop "End "
*                          <disconnect>
*
*
*/

#ifndef _windows
#define forMac      // Codewarrior doesn't have a handy way to do this.
#endif


#include <string.h>
#include <stdio.h>


#ifdef forMac   // ****** stuff for MAC-OS applications ******
#include <Types.h>#include <memory.h>
#include <Packages.h>
#include <Errors.h>
#include <quickdraw.h>
#include <fonts.h>
#include <dialogs.h>
#include <windows.h>
#include <menus.h>
#include <events.h>
#include <OSEvents.h>
#include <Desk.h>
#include <diskinit.h>
#include <OSUtils.h>
#include <resources.h>
#include <toolutils.h>
#include <AppleEvents.h>
#include <string.h>
#include <Strings.h>
#include <stdio.h>
#include <Types.h>
#include <memory.h>

#include "DILCPIPE.H"
#include "DILArchive.h"

#define Timeout 60*10      // this is currently in ticks but should be in milliseconds
//

#else           // ****** stuff for Windows applications *********

#include <windows.h>
#include "DILCPIPE.H"
#include "DILARCHI.H"
#define Timeout 10000    // time out is 1 second
typedef unsigned char   Str255[256];

#define CurrentTimeInSeconds() ((long)(GetTickCount() / 1000))

#endif

#include "Protocol.h"

extern CDILPipe     *ourPipe;
extern void         *gThisObject;       // This is the only DIL frame active at one time in this app
extern short        gXferMode;          // A hook in case we want to archive frames later - tracks what we are working with

FILE                *gArchFile=0;       // UNIX file pointer to file being archived to/from

/* Archive
 *
 * Purpose: Cover function for archiving or dearchiving based on first message received from Newton.
 *          Calls DoArchive to write strings from Newton to disk or DoDearchive to read them and send to Newton.
 */
CommErr Archive()
{
    char  response[kMAXSTR];
    long        length;
    CommErr     fErr;
    short       num_tries=0;

    // put up a status dialog to report on the pipe state and the archive process
    CHECKDILERROR ( InitializePipe() ); // CHECKDILERROR is a macro defined in Protocol.h

    CreateStatusDlog ();

    // make the connection!
    CHECKDILERROR ( DoConnectPipe () );

    length = 7;
    CHECKDILERROR ( ReadFromPipe ( response, &length, 0) );

    response[length] = '\0'; // make it a C string
    if (strncmp( "Sending", response, 7 ) == 0)
        fErr = DoArchive ( response );
    else if (strcmp("Receive",response) == 0)
        fErr = DoDearchive ( response );
    else {
        CHECKDILERROR ( EndArchive ( kDeskAbortError, true ) ); // CHECKDILERROR makes sure we're all tidy before abandoning ship
    }
    return ( fErr );
}


/* DoArchive
 *
 * Purpose: Creates file and writes string(s) from Newton PDA to file.
 */
CommErr DoArchive ( char *response )
{
    long        length,str_len;
    CommErr     fErr=0;
    short       retries=0;

    length = 4;
    CHECKDILERROR ( ReadFromPipe ( response, &length, 0 ) );
    response[length] = '\0'; // make it C string
    if (strncmp("STR ", response, 4) == 0)
        gXferMode = kStrings;
    else // if not string then should be frame
        gXferMode = kFrames;
        // put up standard file dialog and create archive file
    CHECKDILERROR ( CreateArchiveFile( gXferMode ) );
    SetArchiveStatus ( kConnected );

    if (gXferMode == kFrames)
    {
        ; // do bindings for FDIL - not implemented in this lab
    }

    /*  Initiate conversation; tell the Newton PDA that we're ready */
    length = 6; // send terminator too
    CHECKDILERROR( WriteToPipe( "READY\4", &length, 0));

    do { // Loop getting Newton PDA data
        SetArchiveStatus ( kArchiving );
        if ( gXferMode == kStrings ) {
            length = sizeof ( long );
#ifndef _windows
            fErr = ReadFromPipe ( (char *)&str_len, &length, 0 ); // ReadFromPipe is our routine to handle events while waiting for incoming
#else
            fErr = ReadFromPipe ( (char *)&str_len, &length, length ); // ReadFromPipe is our routine to handle events while waiting for incoming
#endif
            if ( !fErr )
                fErr = ReadFromPipe ( response, &str_len, 0 ); // first read got string length, now get string itself
            if ( !fErr ) // send CONFIRMED if no error
            {
                response[str_len] = '\0'; // make it a C string for convenience
                if ( (fErr = ArchiveString ( response )) != 0)
                {
                    EndArchive ( kDeskAbortError, true ); // unable to write string to disk
                    break;
                }
                SetArchiveStatus ( kConfirming );
                length = 10; // send null terminator too
                WriteToPipe ("CONFIRMED\4", &length, 0);
            } else
                break;

        } else {
            ; // this would be frames - not done in this lab
   }
        // look for More or End Archive message
        length = 4;
        fErr = ReadFromPipe ( response, &length, 0 );
        response[length] = '\0';
        if ( fErr )
            break;
    } while ( strcmp("More",response) == 0 );

    if ( !fErr )
    { // check what message we got - if we don't understand it, abort
        if ( strcmp ( "Abrt", response ) == 0 )
        {
            SetArchiveStatus ( kNewtAbortError );
            fErr = EndArchive ( fErr, true );
        } else if ( strcmp ( "End ", response ) == 0 ){
            SetArchiveStatus ( kEndArchive );
            fErr = EndArchive ( 0, true );
        } else { // protocol messed up - unknown message
            SetArchiveStatus ( kDeskAbortError );
            fErr = EndArchive ( fErr, true );
        }
    }

    CDPipeDisconnect(ourPipe);
    return fErr;
}


/* DoDearchive
 *
 * Purpose: Finds previously archived file, reads string(s) from it and sends to Newton PDA.
 */
CommErr DoDearchive ( char *response ){
    long        length;
    CommErr     fErr=0;
    char        msg_string[256], content_str[256];
    Boolean     first_time=true;

    CHECKDILERROR ( OpenArchiveFile() ); // Get file from user and open it for reading

    SetArchiveStatus ( kConnected );
    strcpy(msg_string, "READY FOR ");
    if ( gXferMode == kStrings )
        strcat ( msg_string, "STR " );
    else
        strcat ( msg_string, "FRAM" );
    length = strlen ( msg_string ) + 1;
    msg_string[length-1] = '\4'; // add terminator character then send what we're dearchiving (strings or frames)
    CHECKDILERROR (WriteToPipe(msg_string, &length, 0));
    length = 5;
    CHECKDILERROR ( ReadFromPipe ( response, &length, 0 ) );
    response[length] = '\0';
    if ( strcmp("Ready",response) == 0 )
    {
        if ( gXferMode == kStrings ) { // dearchiving strings
            while ( DearchiveString ( content_str ) != EOF ) { // Loop sending strings to Newton PDA
                SetArchiveStatus ( kDearchiving );
        if ( !first_time ) {
                    strcpy( msg_string, "MORE\4" );
                    length = 5L; // send terminator too
                    fErr = WriteToPipe(msg_string, &length, 0);
                } else
                    first_time = false;
                length = strlen ( content_str ) + 1;
                content_str[length-1] = '\4';
                fErr = WriteToPipe(content_str, &length, 0);
                if ( fErr )
                    break; // didn't go thru, abort
                SetArchiveStatus ( kConfirming );
                length = 4;
                ReadFromPipe ( response, &length, 0 );
                response[length] = '\0'; // make it a C string for convenience
                if ( strcmp ( "Abrt", response ) == 0 ) {
                    fErr = kNewtAbortError;
                    break;
                } else if ( strcmp ( "Conf", response ) != 0 ) { // if not "Conf" protocol is messed up!
                    fErr = kDeskAbortError;
                    break;
                }
            } // end while
        } else {
                ; // this would be frames - not done in this lab
        }
    } else
        fErr = kNewtAbortError; // we didn't get "Ready" so abort

    if ( fErr )
    {
        SetArchiveStatus ( kAborting );
        EndArchive ( fErr, false );
    } else {
        SetArchiveStatus ( kEndArchive );
        EndArchive ( 0, false );
    }

    CDPipeDisconnect(ourPipe);
    return fErr;
}


/* DoConnectPipe
 *
 * Purpose: Try to connect to Newton PDA using currently selected methods.
 */
CommErr DoConnectPipe ( )
{
       CommErr         retn_value = kCommErrNoErr;
    long        endTime = 0;

    // ourPipe is a global defined in DILArchive.c
    retn_value = CDPipeListen(ourPipe, Timeout, 0, 0);

#ifndef _windows
    CHECKDILERROR((short) CDPipeAccept(ourPipe));
#else
    // This code doesn't need to be done on the MacOS side, but is currently required
    // for Windows.  You need to loop a little bit for the connection state to be
    //set to kCDIL_ConnectPending.  It might not do that if there's nobody on the
    // other side of the cable, though!

    // Loop until the Newton device connects
    endTime = CurrentTimeInSeconds() + kTimeoutInSecs;  // time at which we should stop looping
    while (CurrentTimeInSeconds() < endTime) {
        if (CDGetPipeState(ourPipe) == kCDIL_ConnectPending) {
            CHECKDILERROR((short) CDPipeAccept(ourPipe));
            break;
        } else
            CDIdle(ourPipe);
    }

    if (CurrentTimeInSeconds() >= endTime) {    // did we time out?
        CDPipeDisconnect(ourPipe) ;
        retn_value =  kOurTimeoutError;
    } else
       retn_value =  kCommErrNoErr;
#endif

       return retn_value;
}


/* ReadFromPipe
 *
 * Purpose: try to read 'length' bytes from pipe and pack them into 'response' string
 *          Purpose of this routine is to wait for data to arrive and while waiting keep
 *          event processing going. This keeps desktop machine from "freezing".
 *          Macintosh version calls routine EventLoop to check for and handle events.
 *          Since we may not get all the bytes in a single read, keep track of bytes read and
 *          keep trying to get all requested bytes until we time out.
 */
CommErr ReadFromPipe ( char *response, long *length, long swapping )
{
    Boolean     eom;
    short       num_tries=0;
    CommErr     readErr;
    CDIL_State  state;
    long        read_len, length_read=0;

    // *** Students write this
    state = CDGetPipeState(ourPipe);
    if (state != kCDIL_Connected && state != kCDIL_Busy )
        return kPipeNotReady;

    readErr = kTimeout;
    while ( (*length-length_read) && (num_tries++<=kNumRetries) && (readErr==kTimeout) )
    // this translates to: we haven't received total requested bytes,
    // haven't given up trying and
    // haven't failed a read because of something other than a timeout
    {
        read_len = *length-length_read; // necessary because if we timed out we read 0 bytes so we must reset it
readErr = CDPipeRead ( ourPipe, (void *) response, &read_len, &eom, swapping, Encoding, Timeout, 0, 0 );
        if ( read_len != 0 ) {
            response += read_len;
            length_read += read_len;
        }
        EventLoop(); // if you don't need to service events, make this a do-nothing routine
    }
    if ( readErr == kCommErrNoErr )
        if ( *length != length_read )
            readErr = kUnknownError;
    return readErr;
}


/* WriteToPipe
 *
 * Purpose: try to write 'length' bytes to pipe from 'message' string
 *          Purpose of this routine is to encapsulate writing. 'swapping'
 *                     controls whether big endian/little endian byte swapping needs
 *                     happen
 */
CommErr WriteToPipe ( char *message, long *length, long swapping )
{
       return ( CDPipeWrite(ourPipe, (void *)message, length, true, swapping, Encoding, Timeout, 0, 0) );
}


/* ErrorStrings
 *
 * Purpose: return error string based on error number from the CDIL or FDIL
 */
char* ErrorStrings(CommErr theErr, char* theString)
{
    switch (theErr)
        {
        case kOurUserCancelled:  strcpy((char*) theString,
            (char*) "Archive Cancelled"); break;
        case kOurTimeoutError:  strcpy((char*) theString,
            (char*) "Timeout error. (determined by Archive)"); break;
        case kDeskAbortError:   strcpy((char*) theString,
            (char*) "Protocol error. (determined by Archive)"); break;
        case kNewtAbortError:   strcpy((char*) theString,
            (char*) "Abort error. (determined by Newton)"); break;
        case kUnknownError: strcpy((char*) theString,
            (char*) "Unknown OS error. (determined by Archive)"); break;
        case -97:  strcpy((char*) theString,
            (char*) "The port is busy. Quit the relevant application or restart if necessary (from ATalk Driver)");  break;

        // CDIL errors (see DILCPipe.h)
        case -28701:  strcpy((char*) theString,
            (char*) "Error on memory allocation.");  break;
        case -28702:  strcpy((char*) theString,
            (char*) "DIL pipe was set to a bad state.");  break;
        case -28703:  strcpy((char*) theString,
            (char*) "An unknown exception has occurred.");  break;
        case -28704:  strcpy((char*) theString,
            (char*) "The queue of asynchronous calls is full.");  break;
        case -28705:  strcpy((char*) theString,
            (char*) "Pipe has not been initialized.");  break;
        case -28706:  strcpy((char*) theString,
            (char*) "Parameter passed in was invalid.");  break;
        case -28707:  strcpy((char*) theString,
            (char*) "Pipe is not ready for operation.");  break;
        case -28708:  strcpy((char*) theString,
            (char*) "Timeout during DIL operation.");  break;

        // FDIL errors (see HLFDIL.h)
        case -28801:  strcpy((char*) theString,
            (char*) "Out of heap memory");  break;
        case -28802:  strcpy((char*) theString,
  (char*) "Out of temporary or other memory");  break;
        case -28803:  strcpy((char*) theString,
            (char*) "Unknown slot");  break;
        case -28804:  strcpy((char*) theString,
            (char*) "Slot size exceeded");  break;
        case -28805:  strcpy((char*) theString,
            (char*) "Slot size is required");  break;
        case -28806:  strcpy((char*) theString,
            (char*) "Unexpected data type");  break;

        // other errors
        case -28003:  strcpy((char*) theString,
            (char*) "The communication operation was aborted.");  break;
        case -28009:  strcpy((char*) theString,
            (char*) "Bad connection detected");  break;
        case -28017:  strcpy((char*) theString,
            (char*) "Out of memory");  break;
        case -28029:  strcpy((char*) theString,
            (char*) "Cannot connect to modem; no response.");  break;
        case -28030:  strcpy((char*) theString,
            (char*) "Disconnection detected.");  break;
        case -28100:  strcpy((char*) theString,
            (char*) "Disconnect occurred while reading.");  break;
        case -28101:  strcpy((char*) theString,
            (char*) "An error occurred while reading.");  break;
        case -28102:  strcpy((char*) theString,
            (char*) "The communication tool was not found");  break;
        case -28103:  strcpy((char*) theString,
            (char*) "Bad modem tool version");  break;
        default: sprintf(theString, "%ld", theErr);
        }
    return (char*) theString;
}

/* CreateArchiveFile
 *
 * Purpose: Using UNIX file I/O create and open a file and mark it as being for archiving strings or frames
 */
CommErr CreateArchiveFile( short type )
{
    CommErr err=0;
    char    name[256];

    // note that program doesn't know about file errors so we just return kUnknownError
    err = CreateFileName ( name ); // this is in file DILArchive.c as it is platform specific
    if ( !err ) {
        gArchFile = fopen ( name, "wb+" ); // create binary file as string files have weird limits
        if ( !gArchFile )
            err = kUnknownError;
        else {
            if ( type == kStrings )
                fputs ( "ARKS", gArchFile );
            else
                fputs ( "ARKF", gArchFile );
        }
    } else
        err = kUnknownError;
    return err;
}


/* OpenArchiveFile
 *
 * Purpose: Open existing archive file using the UNIX file I/O library and read whether it is for strings or frames
 */
CommErr OpenArchiveFile()
{
    char            name[256]; // 256 in case path information is included
    CommErr         err=kCommErrNoErr;
    int                             converted;

    // note that program doesn't know about file errors so we just return kUnknownError
    err = GetFileName ( name ); // this is in file DILArchive.c as it is platform specific
    if ( !err ) {
        gArchFile = fopen( name, "rb" );
        if ( !gArchFile )
            err = kUnknownError;
        else {
                        converted = fscanf ( gArchFile, "%4s", name );
            if ( converted != EOF && converted != 0) { // file begins with type of archive (strings or frames)
                if ( strcmp ( name, "ARKS" ) == 0 )
                    gXferMode = kStrings;
                else if ( strcmp ( name, "ARKF" ) == 0 )
                    gXferMode = kFrames;
                else {
                    err = kUnknownError; // file not one of ours.
                    fclose ( gArchFile );
                    gArchFile = 0;
                }
            } else
                err = kUnknownError;
        }
    } else
        err = kUnknownError; // user cancelled
    return err;
}


/* CreateArchiveFile
 *
 * Purpose: Using UNIX file I/O create and open a file and mark it as being for archiving strings or frames
 */
CommErr ArchiveString ( char *from_str )
{
    short   length;
    CommErr err=0;

    // we're going to assume we're at the right place to write a string
    length = strlen ( from_str );
    if ( fprintf( gArchFile, "%4hd%s", length, from_str ) == EOF )
        err = kUnknownError;

    return err;
}


/* DearchiveString
 *
 * Purpose: Using UNIX file I/O read first/next string from file and return to caller as 'to_str'
 */
CommErr DearchiveString ( char *to_str )
{
    int   length;
    CommErr err;

    // we're going to assume we're at the right place to get a string
    if ( fscanf ( gArchFile, "%4d", &length ) == EOF )
        return EOF;
    err = (CommErr)fgets ( to_str, length+1, gArchFile ); // length+1 because fgets read n-1 chars from stream
    if ( !err ) { // gets returns NULL if read failed
        if ( feof(gArchFile) )
            err = EOF;
        else
            err = kUnknownError;
    } else
        err = kCommErrNoErr;
    return err;
}


/* EndArchive
 *
 * Purpose: Terminate archiving as gracefully as possible including closing file and notifying Newton PDA of end.
 */
CommErr EndArchive ( CommErr endState, Boolean sending )
{
    long    length;    char    response[256];

    SetArchiveStatus ( endState );
    if ( endState  )
    {
        if ( endState >= kUnknownError )
        {
            length = 6L; // send terminator character
            WriteToPipe ("ABORT\4", &length, 0);
        }

    } else { // no error
        if ( !sending )
        {
            length = 12L; // send null terminator too (changed to \4 instead)
            WriteToPipe ("END ARCHIVE\4", &length, 0);
        }
    }

    length = 4;
    ReadFromPipe ( response, &length, 0 ); // wait for acknowledgement we're ending

    if ( gArchFile ) {
        fclose ( gArchFile );
        gArchFile = 0;
    }

    return endState;
}
                                    v   v   R         O  Y      	&   
PROTOCOL.C   TEXTMACA     TEXTMACA                   :  S     $         J                                    G       Y  Y  P1             H 	Monaco                              =w  =w4  5~  5  3           v   v   R J    R MPSR  MWBB   *           "       n                                                            