/*
Copyright (C) 1999/2000 Hendrik Lipka.
All rights reserved.

This program is free software; you can redistribute it and/or modify it under the terms
of the GNU General Public License as published by the Free Software Foundation;
either version 2 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along with this
program; if not, write to the Free Software Foundation, Inc., 675 Mass Ave,
Cambridge, MA 02139, USA or see http://www.gnu.org

If you encounter problems or have a suggestion, please mail to:
hli@hendriklipka.de
*/
package de.hendriklipka.pdfconv;

import org.jpedal.PdfDecoder;
import org.jpedal.io.ObjectStore;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;

public class PDFConv
{
    private String _pdfName;
    private String _baseName;
    private String _targetDir="./output/";

    boolean _usePict=false;

    private DocumentWriter _documentWriter=null;
    private ImageWriter _imageWriter=null;

    public static void main(String[] args)
    {
        try
        {
            PDFConv pc=new PDFConv();
            pc.readParameters(args);
            pc.convert();
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }

    }
    
    public PDFConv()
    {
    }

    private void readParameters(String[] args) throws IOException
    {
        String sourceFile=null;
        String targetBaseName=null;
        String targetDir="";
        String maxWidth="";
        String maxHeigth="";
        String scaleFactor="";
        String quality="";
        String pict="";

        for (int i=0; i<args.length; i++)
        {
            String arg=args[i];
            if (arg.startsWith("-source="))
            {
                sourceFile=arg.substring(8);
            }
            else if (arg.startsWith("-basename="))
            {
                targetBaseName=(arg.substring(10)).trim();
            }
            else if (arg.startsWith("-maxwidth="))
            {
                maxWidth=arg.substring(10);
            }
            else if (arg.startsWith("-maxheigth="))
            {
                maxHeigth=arg.substring(11);
            }
            else if (arg.startsWith("-scalefactor="))
            {
                scaleFactor=arg.substring(13);
            }
            else if (arg.startsWith("-quality="))
            {
                quality=arg.substring(9);
            }
            else if (arg.startsWith("-targetdir="))
            {
                scaleFactor=arg.substring(11);
            }
            else if (arg.startsWith("-pict"))
            {
                pict="yes";
            }
            else
            {
                System.out.println("unknown parameter: "+arg);
            }
        }
        if (null==sourceFile||null==targetBaseName)
        {
            System.out.println("usage: java de.hendriklipka.pdfconv.PDFConv  [-pict] -source=PDFFILE -basename=IMAGEBASENAME [-targetdir=DIR] [-quality=FLOAT] [-maxwidth=INT] [-maxheigth=INT] [scalefactor=DOUBLE]");
            System.exit(1);
        }

        if (0!=targetDir.length())
        {
            targetDir=targetDir.replace('\\', '/');
            if (!targetDir.endsWith("/"))
                targetDir+="/";
            _targetDir=targetDir;
        }

        _pdfName=sourceFile;
        _baseName=targetBaseName;

        if (0!=pict.length())
        {
            _usePict=true;
            _documentWriter = new BookWriter();
            _imageWriter=new PICTImageWriter();
        }
        else
        {
            _documentWriter = new HTMLWriter();
            _imageWriter=new JPEGImageWriter();

        }

        _imageWriter.setMaxWidht(maxWidth);
        _imageWriter.setMaxHeigth(maxHeigth);
        _imageWriter.setScaleFactor(scaleFactor);
        _imageWriter.setQuality(quality);
        
    }

    private void convert()
    {
        //create a directory if it doesn't exist
        File targetPath=new File(_targetDir);
        if (!targetPath.exists())
            targetPath.mkdirs();


        PdfDecoder pdfDecoder=null;

        System.out.println("converting PDF file: "+_pdfName);
        try
        {
            _documentWriter.open(_targetDir+_baseName);
            _documentWriter.writeHeader(_baseName);

            ObjectStore.init("pdfconv");
            pdfDecoder=new PdfDecoder(true);
            pdfDecoder.openPdfFile(_pdfName);
        }
        catch (Exception ex)
        {
            System.err.println("Error while opening PDF file");
            ex.printStackTrace();
            System.exit(1);
        }

        try
        {
            for (int page=1; page<=pdfDecoder.getPageCount(); page++)
            {
                System.out.println("converting page #"+page);
                pdfDecoder.initDisplay(page);
                pdfDecoder.decodePage(true, page);

                String targetName=_baseName+"_page_"+page;
                if (_usePict)
                    targetName+=".pict";
                else
                    targetName += ".jpg";
                _documentWriter.writeLine(targetName);
                try
                {
                    BufferedImage targetImage=pdfDecoder.getPageAsImage();
                    if (null==targetImage)
                    {
                        System.err.println("can't render page, skipping");
                    }
                    else
                    {
                        _imageWriter.writeImage(targetImage, _targetDir, targetName);

                    }
                }
                catch (Exception ex)
                {
                    System.err.println("Error while writing image");
                    ex.printStackTrace();
                }
                pdfDecoder.clearPdfImageData();
                System.gc();
            }
            _documentWriter.writeFooter();
        }
        catch (Exception ex)
        {
            pdfDecoder.closePdfFile();
            System.err.println("Error while executing ");
            ex.printStackTrace();
            System.exit(1);
        }
        finally
        {
            _documentWriter.close();
            if (null!=pdfDecoder)
            {
                pdfDecoder.markAllPagesAsUnread();
                pdfDecoder.flushObjectValues(false);
                pdfDecoder.closePdfFile();
            }
            ObjectStore.flush();
        }
    }

}
