#! /usr/bin/env python1.5
#############################################################################
#
# Project:     GNUton
#
# File:        $Source: /home/arnold/CVS/gnuton/lib/GnutOS/FileDevice.py,v $
# Version:     $RCSfile: FileDevice.py,v $ $Revision: 1.2 $
# Copyright:   (C) 1998-2000, David Arnold.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#############################################################################
"""
FileDevice

Under GnutOS, files have a special status: unlike a desktop operating
system, files are not a normal part of a GNUton -- they are completely
outside the Gnut way of doing things.

So while you might reasonably expect a file system to be a basic part
of the OS, it's not.

However, on a desktop system, it makes sense to use files, as provided
by the underlying operating system.  They have use in GnutOS as the
primitive container for GnutOS stores.

For this reason, there is a FileDriver -- a device driver that treats
a file as a pseudo-hardware entity, most like a PCMCIA card.  Files
are seen as a string of bytes, able to be used for storage.  GnutOS
has no knowledge of file *systems* however -- they are outside its
context.

So -- the FileDriver allows GnutOS to use FileDevice instances as
storage, in a similar way to how the FlashDriver will allow it to use
FlashDevices.

"""

#############################################################################

import os, sys

from   GnutOS.DeviceDriver               import DeviceDriver
from   GnutOS.Device                     import StorageDevice
from   GnutOS.Exceptions                 import HM_DeviceParameterError, \
                                                HM_DeviceError


#############################################################################
#  Device Identifiers

TYPE = "file"
NAME = "File"


#############################################################################
#  Exceptions

class ParameterError(HM_DeviceParameterError):
    """Bad parameters for FileDevice initialisation."""
    pass

class FileDeviceException(HM_DeviceError):
    """General error with FileDevice."""
    pass

class HostOSFileError(FileDeviceException):
    """A host OS operation on the file failed."""
    pass


#############################################################################

def install(mgr):
    """Load and initialise the driver(s)."""

    mgr.RegisterDriver(FileDriver)

    return


#############################################################################

class FileDriver(DeviceDriver):
    """

    Driver code for file devices.

    """

    def __init__(self, hwMgr):
	self._mgr = hwMgr
	return


    def Type(self):
	"""Return an identifier for the device type."""
	return TYPE


    def Name(self):
	return NAME


    def NewDevice(self, t_param):
	"""Create a new FileDevice instance.

	*t_param* -- parameters for device initialisation (see below).

	FileDevices require three parameters: the file name for the
	host OS, the internal name used by GnutOS, and a size in
	bytes.  Additional parameters may be present but are currently
	ignored."""

	if len(t_param) < 3:
	    raise ParameterError(t_param)

	return FileDevice(self, t_param[0], t_param[1], t_param[2])


#############################################################################

class FileDevice(StorageDevice):
    """A FileDevice is the pseudo-hardware instance of a host OS file."""

    def __init__(self, ref_drv, os_file, name, size):
	"""Initialise a FileDevice.

	*os_file* -- string file name for the host OS
	*name*    -- string internal name for this device
	*size*    -- maximum size of file in bytes.

	FileDevices present a standard interface to host OS files for
	GnutOS.  Note that while a FileDevice tests for the existence
	of the file, it does not attempt to create it if it doesn't
	exist.  This could fail later.

	The *size* parameter specifies the maximum size that the file
	should be allowed.  This cannot be exact, but should be tested
	when writing to the device and if exceeded, further writes
	should fail.

	The readability status of the file is also verified."""

	self._driver = ref_drv
	self._os_name = os_file
	self._gnut_name = name
	self._size = size

	#-- test whether file exists
	self._exists = os.path.exists(self._os_name)

	#-- get readability status of file
	self._ro = 1
	if self._exists:
	    t_stat = os.stat(self._os_name)
	    if t_stat[0] & os.path.stat.S_IWRITE:
		self._ro = 0

	return


    def Name(self):
	"""Return the identifier used by GnutOS for the device."""

	return self._gnut_name


    def HostName(self):
	"""Return the host OS name for the file."""

	return self._os_name


    def Exists(self):
	return os.path.exists(self._os_name)


    def SetReadOnly(self):
	"""Set the file for read-only operation."""

	self._ro = 1
	return


    def SetReadWrite(self):
	"""Set the file for read-write operation."""

	self._ro = 0
	return


    def IsWriteable(self):
	"""Is this file writeable?"""

	return not self._ro


    def Size(self):
	#fixme: use stat() to return current size of file
	pass


    def MaxSize(self):
	return self._size


#############################################################################
