#! /usr/bin/env python1.5
#############################################################################
#
# Project:     GNUton
#
# File:        $Source: /home/arnold/CVS/gnuton/lib/GnutOS/StorageManager.py,v $
# Version:     $RCSfile: StorageManager.py,v $ $Revision: 1.3 $
# Copyright:   (C) 1998, David Arnold.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#############################################################################
"""

The StorageManager is one of the basic components of GnutOS.  It is
responsible for managing the persistent storage architecture, and the
devices and their drivers that support it.

The StorageManager relies on the HardwareManager for access to the
devices on which Store instances are constructed.

"""

#############################################################################

from   random                    import randint
from   GnutOS.Store              import Store, StoreException

#############################################################################

STORE_DRV_TYPE = "store"


#############################################################################

class NoSuchStore(StoreException):
    """The specified store (by index) does not exist."""
    pass

class IncompatibleHostForStore(StoreException):
    """The specified device is not supported as a host by this driver."""
    pass

class NoStoreAvailable(StoreException):
    """The StorageManager has initialised, but could not mount any Stores."""
    pass


#############################################################################

class StorageManager:
    """

    A GNUton instance must manage three sorts of memory: system RAM,
    used for caching blocks off the flash, thread stacks, HWR, driver
    buffers, etc; heap, used to create NewtonScript objects; and
    storage, used to hold soups and packages.

    This class manages the storage domain, comprised of individual
    Stores, loaded (on the MessagePads and eMate) from either the
    internal flash or a PCMCIA card.

    GNUton will support a variety of Store classes, anticipated to
    include disc and network-based stores in addition to PCMCIA cards.

    """

    def __init__(self, ref_gnut):
	"""Initialise the StorageManager.

	*ref_gnut* -- the owning Gnuton instance
	Exceptions -- ?

	The StorageManager is the last thing to be initialised before
	the main event loop is started.  All the other Managers are
	active by this time.

	The StorageManager relies particularly on the HardwareManager
	for devices to support its stores.  As part of its
	initialisation, it scans the set of available devices, looking
	for those that could host stores.  Each of these is then
	checked, and if it does contain a store, it is mounted.  If
	not, a deferred event is registered suggesting that the user
	format the device as a store. """

	#-- who owns me?
	self._gnut = ref_gnut

	#-- tables
	self._d_store = {}            # {int: ref_store} user visible
	self._d_store_impl = {}       # {int: store_impl} implementation

	self._d_soupDef = {}
	self._d_soupChangeCB = {}

	self._default_store = None    # integer index of default store

	self._c = None                #fixme: HACK HACK HACK HACK HACK !!!
	self._d_dialog = {}


	#-- scan all available devices looking for stores
	hm = self._gnut._hm

	self._gnut.log("stores ")

	for store_drv_name in hm.ListDriversForType(STORE_DRV_TYPE):
	    self._gnut.log("driver %s: " % store_drv_name)
	    store_drv_ref = hm.Driver(store_drv_name)

	    for host_drv_name in store_drv_ref.HostDriverNames():
		for host_dev_id in hm.ListDevicesForDriver(host_drv_name):
		    host_dev_ref = hm.Device(host_dev_id)
		    host_dev_name = host_dev_ref.Name()

		    #-- if it's locked, it's already in use
		    if self._gnut._hm.IsDeviceLocked(host_dev_id):
			pass

		    #-- check for existing store on device
		    elif store_drv_ref.ProbeDevice(host_dev_id) > 0:
			#-- lock host device
			key = "foo"  #fixme: needs a real token!
			hm.LockDevice(host_dev_id, key)

			#-- open store
			store_impl = self.OpenStore(store_drv_ref, host_dev_id)

			#-- register device
			self._gnut._hm.RegisterDevice(store_impl)

		        #-- mount
			self.MountStore(store_impl)

			#-- log
			self._gnut.log(" %s" % host_dev_name)

		    else:  
		        #-- lodge format request
			self._gnut._vm.SendMessage(self, 
						   "_ConfirmFormatStoreDialog",
						   (store_drv_ref, host_dev_id))
			#-- log
			self._gnut.log("deferred ")
			self._gnut.log("%s on #%s " % (store_drv_name, host_dev_name))

	#-- register global functions
	self._gnut._am.DefGlobalFn("GetDefaultStore", self.GetDefaultStore)
	self._gnut._am.DefGlobalFn("SetDefaultStore", self.SetDefaultStore)
	self._gnut._am.DefGlobalFn("GetStores",       self.GetStores)
	self._gnut._am.DefGlobalFn("RegSoupChange",   self.RegSoupChange)
	self._gnut._am.DefGlobalFn("UnRegSoupChange", self.UnRegSoupChange)
	self._gnut._am.DefGlobalFn("XmitSoupChange",  self.XmitSoupChange)

	#-- ensure that we have at least one store mounted
	#fixme: this check probably shouldn't be *here* ???
	#if len(self._d_store) < 1:
	#    raise NoStoreAvailable()

	self._default_store = 0


	return


    def FormatStore(self, drv_id, dev_id):
	"""Create a new Store(Impl) on a host Device instance.

	*drv_id*   -- integer driver identifier for specific StoreImpl class
	*dev_id*   -- integer device identifier for host device
	Returns    -- reference to StoreImpl

	Exceptions 
	HM_NoSuchDriver -- specified driver is not available
	HM_NoSuchDevice -- specified device is not available
	IncompatibleHostForStore -- the driver doesn't support the device

	Given a StoreImpl driver, and a device of a type supported by
        the driver, this method creates a new, empty store on that
        device.

	The StoreImpl reference should not be used by applications
        (they should use the associated Store instance)."""

	#-- get specific Store driver
	ref_drv = self._gnut._hm.Driver(drv_id)

	#-- create new StoreImpl instance
	store_impl = ref_drv.NewDevice((dev_id,))

	#-- format device with new store instance
	store_impl.Format()

	return store_impl


    def EraseStore(self, device):
	"""Erase the Store on the specified device.

	*device*   -- device identifier
	Returns    -- nothing
	Exceptions -- ?

	Erase the contents of the Store."""

	#fixme: hmmm ...
	return


    def OpenStore(self, drv_ref, dev_id):
	"""Open an existing StoreImpl on a host Device instance.

        *drv_ref*  -- Store driver instance
        *dev_id*   -- store's host device identifier (normally a file)
        Returns    -- StoreImpl instance
        Exceptions --
        """

	store_impl = drv_ref.NewDevice((dev_id,))
	store_impl.Open()

	return store_impl


    def MountStore(self, store_impl, idx_store=None, *flags):
	"""Mount an existing StoreImpl instance.

	*store_impl* -- reference to a StoreImpl object.
	*idx_store*  -- optional integer index for this store
	*flags*      -- optional flags for internal, rom and write-protection 
	Returns      -- assigned index
	Exceptions   -- ?

	This method has the same effect as inserting a card into an
        Apple MessagePad. """

	#-- create a user-visible Store instance
	store = Store(self, store_impl)

	#-- install the user-visible Store into the stores array
	if idx_store == None:
	    idx_store = len(self._d_store)

	if self._d_store.has_key(idx_store):
	    raise DuplicateStoreIndex(idx_store)

	#-- add to tables
	self._d_store[idx_store] = store
	self._d_store_impl[idx_store] = store_impl

	#-- check default store
	if self._default_store == None:
	    self._default_store = idx_store

	#fixme: notifications?
	#fixme: is the announcement a notification client?

	#-- announcement dialog
	self._gnut._vm.SendMessage(self, 
				   "_AnnounceStore",
				   (idx_store,))
	return idx_store


    def UnMountStore(self, idx_store):
	"""Un-mount a store.

	*idx_store* -- integer index of store to un-mount
	Returns     -- (none)
	Exceptions  -- NoSuchStore

	This method will (indirectly) generate the "Newton still needs
        the card" popup if the requested card is still needed.  The
        only thing to do at that point is re-mount it. """

	#-- check store
	if not self._d_store.has_key(idx_store):
	    raise NoSuchStore(idx_store)

	#-- close store cleanly
	self._d_store_impl[idx_store].Close()

	#-- remove from tables
	del self._d_store[idx_store]
	del self._d_store_impl[idx_store]

	#fixme: notifications?
	return


    def ListStores(self):
	"""Return list of currently mounted stores.

	Note that the first store is that nominated as the "internal"
	store, of which each GNUton must have one."""

	#fixme: sort such that "internal" is first!
	return self._d_store.values()


    def Store(self, idx_store):
	"""Return the specified store object."""

	if not self._d_store.has_key(idx_store):
	    raise NoSuchStore(idx_store)

	return self._d_store[idx_store]


    #######################################################################
    #   
    #   public API methods
    #

    def GetStores(self):
	"""Return an array of references to all existing stores."""

	return self.ListStores()


    def GetDefaultStore(self):
	"""Return a reference to the default store."""

	return self.Store(self._default_store)


    def SetDefaultStore(self, newDefaultStore):
	"""Sets the default store and returns a reference to it."""

	#fixme: what a hack!

	return



    def RegSoupChange(self, soupName, callbackID, callbackFn):
	"""Request notification of soup changes."""

	d_cb = self._d_soupChangeCB
	if d_cb.has_key(soupName):
	    d_cb[soupName][callbackID] = callbackFn

	else:
	    d_cb[soupName] = {callbackID: callbackFn}

	return


    def UnRegSoupChange(self, soupName, callbackID):
	"""Request removal from notification of soup changes."""

	d_cb = self._d_soupChangeCB
	if not d_cb.has_key(soupName):
	    raise NameError("no registrations for soup anyway dork")

	if not d_cb[soupName].has_key(callbackID):
	    raise NameError("no such callbackID for this soup.")

	del d_cb[soupName][callbackID]
	return


    def XmitSoupChange(self, soupName, appSymbol, changeType, changeData):
	""" """

	d_cb = self._d_soupChangeCB
	if not d_cb.has_key(soupName):
	    return

	d_cbFn = d_cb[soupName]
	for cbID in d_cbFn.keys():
	    try:
		d_cbFn[cbID](soupName, apSymbol, changeType, changeData)

	    except:
		pass

	return


    def GetUnioSoupAlways(self, soupName):
	""" """

	#fixme: what about any other store/soup global methods?
	return


    #######################################################################
    #
    #   internal methods
    #

    def _AnnounceStore(self, idx_store):
	print "Announce Store", idx_store
	return


    def _ConfirmFormatStoreDialog(self, drv_id, dev_id):
	"""User dialogue to confirm formatting a store."""

	#fixme: this needs to be written using DisplayDevice, not tkinter!

	# HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK
	# HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK
	# HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK

	import Tkinter
	#-- allocate a view tag
	tag = "view%d" % randint(0, 0xffff)
	self._d_dialog[tag] = (drv_id, dev_id)
	self._gnut.log("Confirming %s (%s) ... %s\n" % (drv_id, dev_id, tag))

	#-- get file name
	str_fname = self._gnut._hm.Device(dev_id).Name()
	print "File name", str_fname

	#-- get reference to Display device
	lst_drv = self._gnut._hm.ListDriversForType("display")
	drv_id = lst_drv[0]
	drv_ref = self._gnut._hm.Driver(drv_id)
	drv_nam = drv_ref.Name()

	lst_dev = self._gnut._hm.ListDevicesForDriver(drv_nam)
	dev_id = lst_dev[0]
	dev_ref = self._gnut._hm.Device(dev_id)

	#-- get canvas
	c = dev_ref.Canvas()

	cx = int(c.configure("width")[4])
	cy = int(c.configure("height")[4])

	#-- draw view
	x1 = cx/2 - 150
	x2 = x1 + 300
	y1 = cy/2 - 100
	y2 = y1 + 200

	border  = c.create_rectangle(x1-5,y1-5,x2+5,y2+5,
				     outline="gray50",
				     fill="gray70",
				     width=10,
				     tag=tag)

	outline = c.create_line     (cx/2,  y1-10,
				     x2,    y1-10,
				     x2,    y1-10,
				     x2+10, y1-10,
				     x2+10, y1,
				     x2+10, y1,
				     x2+10, y2,
				     x2+10, y2,
				     x2+10, y2+10,
				     x2,    y2+10,
				     x2,    y2+10,
				     x1,    y2+10,
				     x1,    y2+10,
				     x1-10, y2+10,
				     x1-10, y2,
				     x1-10, y2,
				     x1-10, y1,
				     x1-10, y1,
				     x1-10, y1-10,
				     x1,    y1-10,
				     x1,    y1-10,
				     cx/2,  y1-10,
				     fill="black", 
				     smooth=1,
				     splinesteps=4,
				     width=4,
				     tag=tag)

	handle  = c.create_polygon  (cx/2,    y1-10,
				     cx/2-8,  y1-10,
				     cx/2,    y1,
				     cx/2+8,  y1-10,
				     cx/2,    y1-10,
				     fill="gray70",
				     outline="black",
				     smooth=1,
				     splinesteps=6,
				     width=4,
				     tag=("dagger_"+tag, tag))

	if not self._c:
	    self._c = Tkinter.Image("bitmap", 
				    file="close18x18.xbm",
				    maskfile="close18x18m.xbm")

	t1 = c.create_text(x1+80, y1+20, 
			   text=str_fname,
			   fill="black",
			   font="10x20",
			   anchor="w",
			   tag=tag)

	t2 = c.create_text(x1+80, y1+80, 
			   text="This file is unformatted.  It must be formatted before it can store data.\n\nWould you like to format the file?",
			   width="200",
			   fill="black",
			   anchor="w",
			   tag=tag)

	#-- image is 18x18, gaps should be 6 wide, border should be 4.

	bx = x2 - 24
	by = y2 - 24

	border = c.create_polygon (bx,     by-17,

				   bx+13,  by-17,
				   bx+15,  by-15,
				   bx+17,  by-13,

				   bx+17,  by+13,
				   bx+15,  by+15,
				   bx+13,  by+17,

				   bx-13,  by+17,
				   bx-15,  by+15,
				   bx-17,  by+13,

				   bx-17,  by-13,
				   bx-15,  by-15,
				   bx-13,  by-17,

				   bx,     by-17,
				   fill="gray70",
				   outline="black",
				   smooth=1,
				   splinesteps=20,
				   width=4,
				   tag=(tag, "close_"+tag))

        cross = c.create_image(bx, by, image=self._c, 
			       anchor="center",
			       tag=(tag, "close_"+tag))

	bx = x2 - 86
	by = y2 - 24

	format = c.create_polygon (bx,     by-17,

				   bx+30,  by-17,
				   bx+32,  by-15,
				   bx+34,  by-13,

				   bx+34,  by+13,
				   bx+32,  by+15,
				   bx+30,  by+17,

				   bx-30,  by+17,
				   bx-32,  by+15,
				   bx-34,  by+13,

				   bx-34,  by-13,
				   bx-32,  by-15,
				   bx-30,  by-17,

				   bx,     by-17,
				   fill="gray70",
				   outline="black",
				   smooth=1,
				   splinesteps=20,
				   width=4,
				   tag=(tag, "format_"+tag))

	f_text = c.create_text    (bx, by, 
				   font="10x20",
				   text="Format",
				   tag=(tag, "format_"+tag))

	c.tag_bind("format_"+tag, "<ButtonRelease-1>",self._ConfirmFormatStore)
	c.tag_bind("close_"+tag, "<ButtonRelease-1>", self._CloseDialog)
	c.tag_bind("dagger_"+tag, "<Button-1>", self.drag)

	c.tag_bind(tag, "<Button-2>", self.dump)
	dev_ref.Update()

	return


    def dump(self, e):
	print "dump", e.widget.gettags("current")
    

    def drag(self, e):
	#-- get dialog
	c = e.widget
	tags = c.gettags("current")

	for tag in tags:
	    if tag[0:4] == "view":
		break

	print tags, "->", tag
	
	c.move(tag, 50, 50)
	c.tkraise(tag, "all")
	return


    def _ConfirmFormatStore(self, e):
	"""Handle a confirmed format from the dialog."""

	#-- remove dialog
	c = e.widget
	tags = c.gettags("current")
	print tags
	c.delete(tags[0])

	#-- retrieve store info
	drv_id, dev_id = self._d_dialog[tags[0]]

	#-- format store
	ref_store = self.FormatStore(drv_id, dev_id)

	#-- register store device
	self._gnut._hm.RegisterDevice(ref_store)

	#-- mount store
	self.MountStore(ref_store)

	return


    def _CloseDialog(self, e):
	"""Close the ConfirmFormatStore dialog."""

	#-- remove dialog
	c = e.widget
	tags = c.gettags("current")
	c.delete(tags[0])

	return


	# HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK
	# HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK
	# HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK



#############################################################################

if __name__ == "__main__":

    sm = StorageManager(None)
    print sm.ListStores()


#############################################################################

