#############################################################################
#
# Project:     GNUton
#
# File:        $Source: /home/arnold/play/gnuton/lib/RCS/AppManager.py,v $
# Version:     $RCSfile: AppManager.py,v $ $Revision: 1.2 $
# Copyright:   (C) 1998, David Arnold.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#############################################################################
"""
AppManager.py

The AppManager controls the execution of applications under GnutOS.
As such, it is comprised principly of the NewtonScript Virtual
Machine, together with the global variable and global function tables.

"""

#############################################################################

import sys, types

from   Classes            import Frame, Symbol
from   Constants          import NIL, TRUE
from   Exceptions         import GnutOSException
from   GnutonVM           import GnutonVM


#############################################################################
#  exception classes
#

class AM_BadGlobalNameType(GnutOSException):
    """Names of global vars/funcs must be symbol or string."""
    pass


#############################################################################

class AppManager:
    """ """

    def __init__(self, ref_gnut):
	"""Initialise the Application Manager."""

	self._gnut = ref_gnut
	self._gnut.log("globals ")

	#-- hidden globals tables
	self._global_vars = Frame()
	self._global_funcs = Frame()

	#-- initialise globals
	self.DefGlobalVar("vars",           self._global_vars)
	self.DefGlobalVar("functions",      self._global_funcs)

	self.DefGlobalFn("DefGlobalFn",     self.DefGlobalFn)
	self.DefGlobalFn("UnDefGlobalFn",   self.UnDefGlobalFn)
	self.DefGlobalFn("GetGlobalFn",     self.GetGlobalFn)
	self.DefGlobalFn("GlobalFnExists",  self.GlobalFnExists)
	self.DefGlobalFn("DefGlobalVar",    self.DefGlobalVar)
	self.DefGlobalFn("DefGlobalVar",    self.UnDefGlobalVar)
	self.DefGlobalFn("GetGlobalVar",    self.GetGlobalVar)
	self.DefGlobalFn("GlobalVarExists", self.GlobalVarExists)

	#-- virtual machine
	self._gnut.log("vm ")
	self._vm = GnutonVM(self)

	return



    def DefGlobalFn(self, symbol, function):
	"""Defines a global function; returns symbol."""

	if type(symbol) == types.StringType:
	    symbol = Symbol(symbol)
	    self._global_funcs[symbol] = function

	elif type(symbol) == types.InstanceType and symbol.__class__ == Symbol:
	    self._global_funcs[symbol] = function

	else:
	    raise AM_BadGlobalNameType(symbol)

	return symbol


    def UnDefGlobalFn(self, symbol):
	"""Remove the definition of a global function.  Returns nil."""

	if type(symbol) == types.StringType:
	    symbol = Symbol(symbol)
	    if self._global_funcs.has_key(symbol):
		del self._global_funcs[symbol]

	elif type(symbol) == InstanceType and symbol.__class__ == Symbol:
	    if self._global_funcs.has_key(symbol):
		del self._global_funcs[symbol]

	else:
	    raise AM_BadGlobalNameType(symbol)

	return NIL


    def GetGlobalFn(self, symbol):
	"""Return a global function or nil."""

	if type(symbol) == types.StringType:
	    s = Symbol(symbol)
	    if self._global_funcs.has_key(s):
		return self._global_funcs[s]

	elif type(symbol) == types.InstanceType and symbol.__class__ == Symbol:
	    if self._global_funcs.has_key(symbol):
		return self._global_funcs[symbol]

	else:
	    raise AM_BadGlobalNameType(symbol)

	return NIL


    def GlobalFnExists(self, symbol):
	"""Returns non-nil if a function exists named symbol, or nil."""

	if type(symbol) == types.StringType:
	    s = Symbol(symbol)
	    if self._global_funcs.has_key(s):
		return TRUE

	elif type(symbol) == types.InstanceType and symbol.__class__ == Symbol:
	    if self._global_funcs.has_key(symbol):
		return TRUE

	else:
	    raise AM_BadGlobalNameType(symbol)

	return NIL


    def DefGlobalVar(self, symbol, value):
	"""Defines a global variable; returns value."""

	if type(symbol) == types.StringType:
	    symbol = Symbol(symbol)
	    self._global_vars[symbol] = value

	elif type(symbol) == types.InstanceType and symbol.__class__ == Symbol:
	    self._global_vars[symbol] = value

	else:
	    raise AM_BadGlobalNameType(symbol)

	return value


    def UnDefGlobalVar(self, symbol):
	"""Remove the definition of a global variable.  Returns nil."""

	if type(symbol) == types.StringType:
	    symbol = Symbol(symbol)
	    if self._global_vars.has_key(symbol):
		del self._global_vars[symbol]

	elif type(symbol) == InstanceType and symbol.__class__ == Symbol:
	    if self._global_vars.has_key(symbol):
		del self._global_vars[symbol]

	else:
	    raise AM_BadGlobalNameType(symbol)

	return NIL


    def GetGlobalVar(self, symbol):
	"""Return the value of a slot in the globals frame, or nil."""

	if type(symbol) == types.StringType:
	    s = Symbol(symbol)
	    if self._global_vars.has_key(s):
		return self._global_vars[s]

	elif type(symbol) == types.InstanceType and symbol.__class__ == Symbol:
	    if self._global_vars.has_key(symbol):
		return self._global_vars[symbol]

	else:
	    raise AM_BadGlobalNameType(symbol)

	return NIL


    def GlobalVarExists(self, symbol):
	"""Returns non-nil if a variable exists named symbol, or nil."""

	if type(symbol) == types.StringType:
	    s = Symbol(symbol)
	    if self._global_vars.has_key(s):
		return TRUE

	elif type(symbol) == types.InstanceType and symbol.__class__ == Symbol:
	    if self._global_vars.has_key(symbol):
		return TRUE

	else:
	    raise AM_BadGlobalNameType(symbol)

	return NIL



#############################################################################

if __name__ == "__main__":
    pass


#############################################################################
