#! /usr/bin/env python
#############################################################################
#
# Project:     GNUton
#
# File:        $Source: /home/arnold/play/gnuton/lib/RCS/Classes.py,v $
# Version:     $RCSfile: Classes.py,v $ $Revision: 1.11 $
# Copyright:   (C) 1997-1998, David Arnold.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#############################################################################
"""
Classes

The bytecode interpreter operates on instances of a set of Python
classes representing the basic datatypes of NewtonScript.  These
classes are implemented in Python, with equivalent semantics to the
NewtonScript implementation, but no represenational similarity.

Each class has the ability to marshall to and from a byte stream
representations of itself that match those used by the Newton,
allowing reading and writing Newton-format packages, for example.


Note that the views of the NewtonScript data types presented by the
Apple Formats document, and the NewtonScript Language document are
slightly different:

Language:			Formats:

all objects			refs
-- primitive classes		-- immediate
   -- immediate			   -- integer
      -- integer		   -- char
      -- char			   -- special
      -- boolean		      -- true
   -- binary			      -- nil
      -- symbol			      -- kSymbolClass
      -- string			      -- others ... ?
      -- real			-- pointer
   -- array			   -- pointer
   -- frame			   -- magic pointer
				objects
				-- binary
				-- array
				-- frame

The overlap here is fairly obvious, and the view taken by the language
book is that of a NewtonScript programmer.  This module, however, is
designed to support the virtual machine, and so must take a bias
towards the categorisations of the Formats doc.

"""

#############################################################################

import struct

from   Constants      import kSymbolClass
from   Exceptions     import BadSymbolString


#############################################################################

class GnutData:
    """Base class for all NewtonScript data items."""

    def __init__(self):
	self._primitive = None
	self._class = None

    def PrimClassOf(self):
	return self._primitive

    def ClassOf(self):
	return self._class

    def IsSubclass(self, klass):
	pass

    def IsInstance(self, klass):
	pass


class Ref(GnutData):
    """Base class for primitive objects in NewtonScript."""

    def __init__(self):
	self._primitive = None
	self._class = None

	self._value = None

	return


    def type(self):
	return self._class


    def pack(self, s, o):
	return o


    def unpack(self, s, o):
	"""Unpack a reference."""
	ref_fmt = ">L"
	ref_len = struct.calcsize(ref_fmt)

	ref = struct.unpack(ref_fmt, s[o:o+ref_len])
	self._value = int(ref[0])

	#print "Unpacking a ref:", hex(self._value) 

	if self._value & 0x3 == 0x3:
	    #print "  Magic Pointer"
	    self._class = SYM_MAGIC

	elif self._value & 0x3 == 0xa:
	    #print "  Character"
	    self._class = SYM_CHAR

	elif self._value & 0x3 == 0x2:
	    #print "  Special immediate"
	    self._class = SYM_SPECIAL

	elif self._value & 0x3 == 0x1:
	    #print "  Pointer"
	    self._class = SYM_POINTER

	else:
	    #print "  Integer"
	    self._class = SYM_INTEGER

	return o + ref_len


#############################################################################
#  ref sub-classes

class ImmediateRef(Ref):
    def __init__(self):
	self._primitive = "immediate ref"


class PointerRef(Ref):
    def __init__(self):
	self._primitive = "pointer ref"


#############################################################################
#  immediate sub-classes

class Integer(ImmediateRef):
    def __init__(self):
	ImmediateRef.__init__(self)
	self._class = SYM_INTEGER
	self._value = 0


    def unpack(self, s, o):
	"""Set the value of an Integer from the stream supplied."""

	int_fmt = ">l"
	int_len = struct.calcsize(int_fmt)

	v = struct.unpack(int_fmt, s[o:o+int_len])

	self._value = v[0] >> 2

	return o + int_len


    def pack(self, s, o):
	"""Write the value of Integer to the supplied stream."""

	#-- pack int into 32, with low 2 bits == 00
	str_v = struct.pack("L", (self._value << 2))
	len_v = struct.calcsize("L")

	#-- write to stream
	s[o:o+len_v] = str_v

	#-- return new offset
	return o+len_v
    

class Char(ImmediateRef):
    def __init__(self):
	ImmediateRef.__init__(self)
	self._class = SYM_CHAR
	self._value = ""

    def pack(self, s, o):
	"""Write the value of Char to the supplied stream."""

	pass

    def unpack(self, s, o):
	char_fmt = ">L"
	char_len = struct.calcsize(char_fmt)

	v = struct.unpack(char_fmt, s[o:o+char_len])

	self._value = int(v[0]) & 0x000ffff0  # unicode mask

	return o + char_len
	


class Special(ImmediateRef):
    def __init__(self, value=0):
	ImmediateRef.__init__(self)
	self._class = "special"
	self._value = value
	return

    def unpack(self, s, o):
	spec_fmt = ">L"
	spec_len = struct.calcsize(spec_fmt)

	v = struct.unpack(spec_fmt, s[o:o+spec_len])

	self._value = int(v[0])

	return o + spec_len


    
#############################################################################
#  pointer sub-classes

class Pointer(PointerRef):
    def __init__(self):
	PointerRef.__init__(self)
	self._class = SYM_POINTER
	self._value = 0            # offset in package of target


    def unpack(self, s, o):
	ptr_fmt = ">L"
	ptr_len = struct.calcsize(ptr_fmt)

	v = struct.unpack(ptr_fmt, s[o:o+ptr_len])
	self._value = int(v[0] - 1)  # remove the lower two bits (0x1)

	#-- add pointer to package-wide table, if not already there
	if not pkg._d_ptr.has_key(self._value):
	    pkg._d_ptr[self._value] = None  #fixme: recurse here!

	return o + ptr_len


    def pack(self):
	return struct.pack(">L", ((self._value & 0xffffff00) + 1))
    

    def index(self):
	return self._value


class Magic(PointerRef):
    def __init__(self):
	PointerRef.__init__(self)
	self._class = SYM_MAGIC
	self._table = 0
	self._index = 0

    def unpack(self, s, o):
	magic_fmt = ">H H"
	magic_len = struct.calcsize(magic_fmt)

	table, idx = struct.unpack(magic_fmt, s[o:o+magic_len])

	self._table = table
	self._index = idx >> 2

	return o + magic_len


#############################################################################
#  object & sub-classes

class Object(GnutData):
    """Base class for classed objects in NewtonScript."""

    def __init__(self):
	self._primitive = None
	self._class = None
	self._size = 0
	return


    def pack(self):
	pass


    def unpack(self, s, o):

	hdr = s[o:o+4]

	size = hdr[:3]
	flags = hdr[4]

	#-- dispatch approriate unpacker
	if flags & 0x1:
	    if flags & 0x02:
		Frame.unpack(self, s, o+8)
	    else:
		Array. unpack(self, s, o+8)
	else:
	    Binary.unpack(self, s, o+8)
	
	return
    

class Binary(Object):
    def __init__(self, classSym=None, data=None):
	"""Create a new Binary object."""

	Object.__init__(self)
	#self._primitive = SYM_BINARY  #fixme: chicken&egg with symbol

	self._class = classSym
	self._data = data

	return

    def unpack(self, s, o):
	bin_fmt = ">L L"
	bin_len = struct.calcsize(bin_fmt)

	flgs, junk = struct.unpack(bin_fmt, s[o:o+bin_len])

	#-- unpack class reference
	c = Ref()
	c.unpack(s, o+bin_len)

	if c.type() == SYM_SPECIAL:
	    pass

	elif c.type() == SYM_POINTER:
	    pass

	else:
	    pass

	#-- unpack data
	size = flgs >> 8
	self._data = s[o: o+size]

	return o + size


    def setClass(self, classSym):
	"""Set the class of this binary object."""

	#fixme: check classSym is a symbol?
	self._class = classSym
	return


    def getClass(self):
	return self._class


class Array(Object):
    """
    """

    def __init__(self):
	Object.__init__(self)

	self._primitive = SYM_ARRAY
	self._len = 0       # number of slots
	self._items = {}    # use a dictionary to allow sparse indexing

	return

    def pack(self, s, o):
	pass


    def unpack(self, s, o):
	"""Unpack an array object from the supplied string."""

	arr_fmt = ">L L L"
	arr_len = struct.calcsize(arr_fmt)

	flgs, junk, klass = struct.unpack(arr_fmt, s[o:o+arr_len])
	
	if flgs & 0xff != 0x41:
	    raise "FuckOff!", "unpacking error"

	self._size  = flgs >> 8
	self._len   = int((self._size - (4 * 3)) / 4)
	self._class = klass

	o = o + arr_len

	for idx in range(self._len):
	    i = Ref()
	    i.unpack(s, o)
	    klass = i.type()

	    if klass == SYM_INTEGER:
		item = Integer()
		o = item.unpack(s, o)

	    elif klass == SYM_POINTER:
		item = Pointer()
		o = item.unpack(s, o)

	    elif klass == SYM_CHAR:
		item = Character()
		o = item.unpack(s, o)

	    elif klass == SYM_SPECIAL:
		item = Special()
		o = item.unpack(s, o)

	    elif klass == SYM_MAGIC:
		item = Magic()
		o = item.unpack(s, o)

	    else:
		print "fuck! what's this?", klass
		raise "FUCK", "bad ref"

	    self._items[idx] = item

	return o


    def __getitem__(self, n):
	"""Return the nth item from the array."""

	return self._items[n]


    def __setitem__(self, n, i):
	"""Set the nth item to i."""

	self._items[n] = i
	return


    def __delitem__(self, n):
	"""Remove the nth item from the array."""

	self._items[n] = None
	return


    def __len__(self):
	return len(self._items)


    def __str__(self):
	return str(self._items)


class Frame(Object):
    """newtonscript frames are LISP-like, slotted collections"""

    def __init__(self, *vargs, **kargs):
	Object.__init__(self)

	self._primitive = SYM_FRAME
	self._items = {}
	self._len = 0

	for key in kargs.keys():
	    self._items[key] = kargs[key]
	return

    def __getitem__(self, name):
	return self._items[name]

    def __setitem__(self, name, value):
	self._items[name] = value

    def __len__(self):
	return len(self._items)

    def __delitem__(self, name):
	del self._items[name]

    def has_key(self, name):
	return self._items.has_key(name)

    def keys(self):
	return self._items.keys()


    def __str__(self):
	return str(self._items)


    def unpack(self, s, o):
	"""Unpack a frame from the supplied string."""

	frm_fmt = ">L L"
	frm_len = struct.calcsize(frm_fmt)

	flgs, junk = struct.unpack(frm_fmt, s[o:o+frm_len])

	if flgs & 0xff != 0x43:
	    raise "FuckOff!", "unpacking error"

	self._size = flgs >> 8
	self._len = (self._size - (3 * 4)) / 4

	o = o + frm_len

	#-- unpack frame map pointer
	map_ptr = Pointer()
	o = map_ptr.unpack(s, o)

	#-- unpack slots
	tmp_items = {}
	for idx in range(self._len):
	    i = Ref()
	    i.unpack(s, o)
	    klass = i.type()

	    if klass == SYM_INTEGER:
		item = Integer()
		o = item.unpack(s, o)

	    elif klass == SYM_POINTER:
		item = Pointer()
		o = item.unpack(s, o)

	    elif klass == SYM_CHAR:
		item = Character()
		o = item.unpack(s, o)

	    elif klass == SYM_SPECIAL:
		item = Special()
		o = item.unpack(s, o)

	    elif klass == SYM_MAGIC:
		item = Magic()
		o = item.unpack(s, o)

	    else:
		print "fuck! what's this?", klass
		raise "FUCK", "bad ref"

	    tmp_items[idx] = item

	#-- unpack the map
	m = Array()
	map_offset = m.unpack(s, map_ptr.index())

	for idx in range(1, len(m) - 1):
	    sym = Symbol()
	    sym.unpack(s, m[idx].index())
	    self._items[sym.name()] = tmp_items[idx]

	print str(self._items)

	return o



#############################################################################
#  binary subclasses

class Symbol(Binary):
    """

    Symbols are one of the very basic classes of objects in
    NewtonScript.  They are used as names for pretty much everything.
    This is because they have a quick hash value which makes lookup
    fast.

    """

    def __init__(self, name=None):
	"""Create a new symbol."""

	Binary.__init__(self)
	self._class = kSymbolClass     # we're special, not a symbol!
	self._hash = 0

	if name:
	    self.setName(name)
	else:
	    self._name = name

	return

    def unpack(self, s, o):
	sym_fmt = ">L L L L"
	sym_len = struct.calcsize(sym_fmt)

	flgs, junk, klass, hash = struct.unpack(sym_fmt, s[o:o+sym_len])

	size = int(flgs >> 8)
	name_o = o + sym_len

	while s[name_o] != "\0":
	    self._name = self._name + s[name_o]
	    name_o = name_o + 1
	
	return o + size


    def name(self):
	"""Return the symbol's string value."""
	return self._name


    def hash(self):
	"""Return the symbol's hash value."""
	return self._hash


    def setName(self, name):
	"""Set the string name for the symbol.

	*name*     -- string value to set as symbol contents
	Returns    -- nothing
	Exceptions -- none

	The algorithm to calculate the hash value is specified on page
        1-15 of Apple's Newton Formats document, version 1.1 """

	if hasattr(self, "_name") and self._name:
	    raise SymbolNameIsImmutable(name, self._name)

	self._name = name

	#-- calculate the hash value
	res = 0L
	lc_offset = ord("a") - ord("A")

	for c in name:
	    if ord(c) < 32 or ord(c) > 127:
		raise BadSymbolString(name, c)

	    if c >= "a" and c <= "z":
		res = res + ord(c) - lc_offset
	    else:
		res = res + ord(c)

	self._hash = (res * 2654435769L) & 0xffffffffL  # 3 force to 32 bit

	return


    def __hash__(self):
	"""Returns value to be used in Python dictionary indexes."""
	return hash(self._name)

    def __str__(self):
	return "Symbol (%x) |%s|" % (self._hash, self._name)


class String(Binary):
    def __init__(self):
	Binary.__init__(self)
	self._class = SYM_SYMBOL
	self._value = ""


class Real(Binary):
    def __init__(self):
	Binary.__init__(self)
	self._class = SYM_SYMBOL
	self._value = 0.0


#############################################################################

SYM_IMMEDIATE = Symbol("Immediate")
SYM_BINARY    = Symbol("Binary")
SYM_FRAME     = Symbol("Frame")
SYM_ARRAY     = Symbol("Array")
SYM_INTEGER   = Symbol("Integer")
SYM_BOOLEAN   = Symbol("Boolean")
SYM_CHAR      = Symbol("Char")
SYM_POINTER   = Symbol("Pointer")
SYM_MAGIC     = Symbol("Magic")
SYM_SPECIAL   = Symbol("Special")
SYM_STRING    = Symbol("string")
SYM_REAL      = Symbol("real")


#############################################################################

if __name__ == "__main__":
    pass


#############################################################################
