#! /usr/bin/env python1.5
#############################################################################
#
# Project:     GNUton
#
# File:        $Source: /home/arnold/play/gnuton/lib/RCS/HardwareManager.py,v $
# Version:     $RCSfile: HardwareManager.py,v $ $Revision: 1.8 $
# Copyright:   (C) 1998, David Arnold.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#############################################################################
"""

"""

#############################################################################

import random

from   Exceptions     import HM_NoDriverForDeviceType, \
                             HM_DriverLoadError, \
			     HM_DriverInstallError, \
			     HM_NoSuchDriver, \
			     HM_NoSuchDevice, \
			     HM_DriverNotLocked


#############################################################################

class HardwareManager:
    """

    A Gnuton instance is hosted by a physical device.  While there are
    minimal configuration limits, this hardware can vary wildly in
    capabilities.

    The HardwareManager provides an abstract interface to various
    types of hardware used by GnutOS, and registers the presence of
    instances of these types.

    For example, a removable persistent storage device, such as a
    flash card or disc drive, is registered with the HardwareManager
    as a class of device.  When booting, drivers are loaded for
    configured classes of hardware.  At runtime, if an instance of a
    registered device is detected, it is registered in the
    HardwareManager's registry, enabling its use by other GnutOS
    components.

    """

    def __init__(self, ref_gnut):
	"""Initialise the HardwareManager for the specified Gnuton."""

	self._gnut = ref_gnut

	#-- registry
	self._dev_id = 0

	self._drv_type = {}   # drivers, by type -> [ref_drv]
	self._drv_name = {}   # drivers, by name -> ref_drv
	self._drv_lock = {}   # drivers, by name -> [token]

	self._dev_ids  = {}   # devices, by id   -> ref_dev
	self._dev_lock = {}   # devices, by id   -> [token]

	self._devices  = {}   # devices, by drv_name -> [dev_id]

	return


    def Gnuton(self):
	"""Return a reference to the Gnuton controlling this Manager."""

	return self._gnut


    def LoadDriver(self, driver):
	"""Load a driver.

	*driver*   -- string class name for the device driver.
	Returns    -- none
	Exceptions -- HM_DriverInstallError

	Loads a driver class into the Gnuton instance.  This driver is
	then available to be registered with the HardwareManager.  The
	Driver must be registered before using devices of this type.

	Drivers can be loaded at any time (and can also be unloaded if
	unused)."""

	#fixme:  drivers should use a separate namespace?  and rexec?

	#-- import the class file
	try:
	    exec("import %s" % driver)
	except:
	    raise HM_DriverLoadError(driver)

	#-- call driver's installation function
	try:
	    exec("%s.install(self)" % driver)
	except:
	    raise HM_DriverInstallError(driver)

	return


    def RegisterDriver(self, deviceDriver):
	"""Register a driver for a class of devices.

	*deviceDriver* -- reference to driver **class**
	Returns        -- nothing
	Exceptions     -- none

	Once a driver class is registered, devices which require it
	can also be loaded.  Devices use the driver **name** to locate
	a suitable driver (driver types are used by the system, for
	example, any "store" driver's device can be mounted, while the
	driver names could be "FileStore" or "FlashStore", for
	example."""

	#-- create driver instance
	d = deviceDriver(self)         # instantiate driver from class
	t = d.Type()                   # get the device type string
	n = d.Name()                   # get the device name string

	#-- enter in driver registry
	self._drv_name[n] = d          # names must be unique

	if self._drv_type.has_key(t):  #fixme: what about duplicates?
	    self._drv_type.append(d)
	else:
	    self._drv_type[t] = [d]

	#-- create registry for devices
	self._devices[n] = []
	self._gnut.log(" %s" % str(t))

	return


    def UnRegisterDriver(self, driverName):
	"""Remove a driver from the system."""

	pass


    def LockDriver(self, driverName, token=None):
	"""Prevent the driver from being unloaded."""

	#-- assign token if required
	if token == None:
	    token = random.randint(0, 0xffff)

	#-- check that specified driver exists
	if not self._drv_name.has_key(driverName):
	    raise HM_NoSuchDriver(driverName)

	#-- add token to list for driver
	if self._drv_lock.has_key(driverName):
	    self._drv_lock[driverName].append(token)
	else:
	    self._drv_lock[driverName] = [token]

	#-- return token (in case we allocated it)
	return token


    def ReleaseDriver(self, driverName, token):
	"""Remove a hold on the driver, allowing it to be unloaded."""

	#-- check driver exists
	if not self._drv_name.has_key(driverName):
	    raise HM_NoSuchDriver(driverName)

	#-- check it was locked using this token
	if not self._drv_lock.has_key(driverName):
	    raise HM_DriverNotLocked(driverName, token)
	
	#-- remove token from list
	self._drv_lock[driverName].remove(token)
	return


    def Driver(self, name):
	"""Return a reference to the named driver."""

	if not self._drv_name.has_key(name):
	    raise HM_NoSuchDriver(name)

	return self._drv_name[name]


    def ListDrivers(self):
	"""Return a list of names of registered drivers."""
	return self._drv_name.keys()


    def ListDriversForType(self, driverType):
	"""Return a list of names of registered drivers of specified type."""

	if not self._drv_type.has_key(driverType):
	    raise HM_NoSuchDriver(driverType)

	return map(lambda d:d.Name(), self._drv_type[driverType])


    def ListDriverTypes(self):
	"""Return a list of types of registered drivers."""
	return self._drv_type.keys()




    def CreateDevice(self, driverName, deviceInfo):
	"""Request the specified driver to create a new device.

	*driverName* -- string name of driver to handle request
	*deviceInfo* -- tuple of parameters for driver
	Returns      -- reference to the device instance

	This method gives a safe way of creating a new device.  Once
	created, the device must be registered before it can be
	properly used, although direct calls to the device instance
	are allowed prior to registration.

	This is normally used to create a device for some hardware,
	verify that it is actually working, and then register it for
	other parts of the system to use."""

        #-- ensure we have a driver loaded for this device
	if not self._drv_name.has_key(driverName):
	    raise HM_NoDriverForDeviceType(driverName)

	driver = self._drv_name[driverName]
	type = driver.Type()

	#-- create new device instance, using deviceInfo
	dev = driver.NewDevice(deviceInfo)

	return dev


    def RegisterDevice(self, device):
	"""Register the availability of a device instance.

	*device*   -- reference to device instance
	Returns    -- device identifier
	Exceptions -- ???

	Each initialised device is given a unique identifier,
	allocated by the HardwareManager._getDeviceId() method.  This
	is used as a key into the device tables for this device
	instance."""

	#-- get driver for this device
	drv_name = device.DriverName()

	#-- generate device id
	dev_id = self._getDeviceId()

	#-- register device
	self.LockDriver(drv_name, dev_id)
	self._dev_ids[dev_id] = device
	self._devices[drv_name].append(dev_id)

	#-- log
	self._gnut.log(" %s#%d" % (str(drv_name), dev_id))

	return dev_id


    def UnRegisterDevice(self, deviceID):
	"""Register the loss of a device instance."""

	dev = self._dev_ids[deviceID]
	drv_name = dev.DriverName()

	self.ReleaseDriver(drv_name, deviceID)

	return


    def Device(self, deviceID):
	"""Return a reference to the specified device."""

	if not self._dev_ids.has_key(deviceID):
	    raise HM_NoSuchDevice(deviceID)

	return self._dev_ids[deviceID]


    def ListDevicesForDriver(self, driverName):
	"""Return a list of devices controlled by the specified driver."""

	if not self._devices.has_key(driverName):
	    raise HM_NoSuchDriver(driverName)

	return self._devices[driverName]



    def LockDevice(self, deviceID, token):
	"""Obtain a lock on a device, preventing it's removal."""
	return

    def ReleaseDevice(self, deviceID, token):
	"""Release a lock on a device, enabling it's removal."""
	return

    def IsDeviceLocked(self, deviceID):
	"""Is the specified device locked by anything?"""
	return 0


    #-- private methods

    def _getDeviceId(self):
	"""Allocate a new device identifier."""

	self._dev_id = self._dev_id + 1
	return self._dev_id


#############################################################################

if __name__ == "__main__":
    pass


#############################################################################

