#! /usr/bin/env python1.5
#############################################################################
#
# Project:     GNUton
#
# File:        $Source: /home/arnold/play/gnuton/lib/RCS/Store.py,v $
# Version:     $RCSfile: Store.py,v $ $Revision: 1.9 $
# Copyright:   (C) 1998, David Arnold.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#############################################################################
"""
Store.py

User-visible Store implementation.  Instances of the Store class are
backed by an instance of StoreImpl.  StoreImpl is specialised for
different types of host device (like disc files, pcmcia cards, etc),
but Store remains the same.

Stores are created by the StorageManager when mounting a StoreImpl.

"""

#############################################################################

import random

from   Exceptions   import EvtExFrStore, HM_DeviceError
from   Platform     import kSError_WriteProtected, kFramesErrStoreIsROM
from   Primitive    import true, nil
from   Soup         import Soup
from   VBO          import VBO
from   Classes      import Frame


#############################################################################

class StoreException(HM_DeviceError):
    """Store related exceptions."""
    pass


#############################################################################

class StoreImpl:
    """

    Base class for Store implementations, where different
    implementations might be used for different device types (for
    example, a DiscStore might use a different implementation than a
    FlashStore).

    """

    def __init__(self):
	"""Initilise a Store object for the specified device."""
	pass

    def GCreate(self, size):
	"""Create a new Store instance, on the specified device."""
	pass

    def GOpen(self):
	"""Open an existing Store instance."""
	pass

    def GClose(self):
	"""Close the Store, ready for unmounting."""
	pass

    def GVerify(self):
	"""Verify that the device specified contains a Store."""
	pass

    def GSetROM(self, flag):
	pass

    def GSetWriteProtect(self, flag):
	pass

    def GSoup(self):
	"""Return the Soup class for this type of Store."""
	return None

    def GIndex(self):
	"""Return the Index class for this type of Store."""
	return None

    def GEntry(self):
	"""Return the Entry class for this type of Store."""
	return None


#############################################################################

class Store:
    """

    Visible Store class.  Store implementations are hidden by the
    StorageManager, and user programs refer to instances of this
    class.

    """

    def __init__(self, storageManager, storeImpl=None):
	"""Create a new user-visible Store object.

	*storageManager* -- reference to the Gnuton's StorageManager
	*storeImpl*      -- reference to the device-specific Store
	Returns          -- new, user-visible Store instance

	This method is called by the StorageManager when a formatted
	device is made available to the Gnuton (in the case of PCMCIA
	cards, this is when the card is inserted; in the case of
	disc-based stores, it's when the user tells the system to
	mount a particular store file).

	The HardwareManager determines that the device has been added
	to the active configuration, and that it contains a StoreImpl
	(device-specific store) instance.  It passes that instance to
	the StorageManager, where it is loaded and initialised.  Then
	an instance of this class is created referring to the
	StoreImpl instance, and notified to the user/system."""

	self._ref_sm = storageManager
	self._ref_impl = storeImpl
	return


    def AtomicAction(self, myAction):
	"""Execute *myAction* as a transaction."""
	return


    def BusyAction(self, appSymbol, appName, myAction):
	"""Call *myAction* with the store marked busy."""
	return


    def CheckWriteProtect(self):
	"""Raise exception if store is locked or in ROM.

	Returns  --  unspecified

	If locked, raises *kSError_WriteProtected* (-10605) or if ROM,
	*kFramesErrStoreIsROM* (-48020). Note that current versions of
	GnutOS return *nil* is the store is writeable."""

	if not (self._wp or self._rom):
	    return nil

	elif not self._wp:
	    raise EvtExFrStore(kSError_WriteProtected)

	else:
	    raise ExtExFrStore(kFramesErrStoreIsROM)



    def CreateSoupXmit(self, soupName, indexArray, changeSym):
	"""Create a soup and transmit a change notification."""

	if len(soupName) > 39:
	    #fixme: do this and determine error!
	    raise UnknownError("soup name too long, max 39 chars")

	#-- create and initialise soup impl
	soup = self._ref_impl.GSoup(soupName)

	s = Soup(self)
	for f in indexArray:
	    s.AddIndex(f)

	#-- add soup to store list
	self.d_soup[soupName] = s

	#-- notify
	#fixme: appName???
	#fixme: soup change symbols
	self._ref_sm.XmitSoupChange(soupName, "????", "soupCreated", s)

	return s


    def Erase(self):
	self.lst_soup = []
	return


    def GetAllInfo(self):
	"""Return (a clone of) the Store's information frame."""

	return Frame(self._info)


    def GetInfo(self, slotSymbol):
	"""Returns contents of specified slot in store's information frame.

	*slotSymbol* -- symbol, slot name
	Returns      -- contents of the specified slot, or nil
	Exceptions   -- ?

	If the specified slot does not exist, this method returns
        *nil*."""

	#fixme: convert symbol to string before use!

	if not self._info.has_key(slotSymbol):
	    return nil

	else:
	    return self._info[slotSymbol]


    def GetKind(self):
	"""Returns string description of storage type used by the store."""

	#fixme: this method needs to be a little more useful ...

	if self._internal:
	    return "Internal"

	else:
	    return "Storage card"


    def GetName(self):
	"""Return store name."""

	return self._name


    def GetSignature(self):
	"""Return store signature."""

	return self._sig


    def GetSoup(self, soupNameString):
	if not self.d_soup.has_key(soupNameString):
	    return nil

	return self.d_soup[soupNameString]


    def GetSoupNames(self):
	return self.d_soup.keys()


    def HasSoup(self, soupName):
	r = self.d_soup.has_key(soupName)
	if r:
	    return true
	else:
	    return nil


    def IsReadOnly(self):
	"""Return non-nil value if store cannot be written.

	Returns  -- nil if store can be written, otherwise non-nil (true).

	Note that the Store could either be in ROM or be
	write-protected when read only."""

	if self._wp or self._rom:
	    return true

	else:
	    return nil


    def IsValid(self):
	"""Returns true if the store can be used."""

	#fixme: before open/create and after close ?

	return true


    def SetAllInfo(self, frame):
	#fixme: convert frame to dict before storage
	self._info = frame
	return


    def SetInfo(self, slotSymbol, value):
	#fixme: convert symbol to string before use?
	self._info[slotSymbol] = value
	return


    def SetName(self, storeNameString):
	self._name = storeStringName
	return


    def SetSignature(self, signature):
	self._sig = signature
	return


    def SuckPackageFromBinary(self, binary, paramFrame):
	print "sucking package"

	cb = None
	freq = 0

	if paramFrame.has_key("callback"):
	    cb = paramFrame["callback"]

	if paramFrame.has_key("callbackFrequency"):
	    freq = paramFrame["callbackFrequency"]


	#-- load package! ;-)

	return


    def SuckPackageFromEndpoint(self, endPoint, paramFrame):
	return


    def TotalSize(self):
	return self._size


    def UsedSize(self):
	return self._used


    def NewVBO(self, klass, size):
	"""Create a new VBO in this store."""

	return VBO(klass, size)


    def NewCompressedVBO(self, klass, size, companderName, companderData):
	"""Create anew compressed VBO in this store."""

	return VBO(klass, size, companderName, companderData)


    #-- obsolete methods

    def CreateSoup(self, soupName, indexArray):
	""" """
	raise ObsoleteMethod("use CreateSoupFromSoupDef()")

    def RestorePackage(self, packageObject):
	"""Installs the specified package onto the store."""
	raise ObsoleteMethod("use SuckPackageFromBinary().")


#############################################################################

if __name__ == "__main__":
    pass


#############################################################################
