#############################################################################
#
# Project:     GNUton
#
# File:        $Source: /home/arnold/play/gnuton/lib/RCS/ViewManager.py,v $
# Version:     $RCSfile: ViewManager.py,v $ $Revision: 1.4 $
# Copyright:   (C) 1998, David Arnold.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#############################################################################
"""
ViewManager.py

Views are the core of the GnutOS.  All the user-visible work that goes
on is accomplished by sending messages to View instances.

The ViewManager maintains the set of View classes currently created.
The most important of these is the "root" view, from which all other
views are descended.  


  """

#############################################################################

import struct, sys, types

from   Classes            import Frame
from   Constants          import NIL

#############################################################################

class ViewManager:
    """the View management system"""

    def __init__(self, ref_gnut):
	"""Initialise the View system."""

	self._gnut = ref_gnut
	self.lst_v = []

	self._display = self._getDisplay()

	#-- view lookup table
	self._d_pview = {}
	self._vid = 0

	#-- create root view frame(s)
	self._gnut.log("rootview ")
	self._rom_root = Frame()   #fixme: ROM ??
	self._rom_root_wrapper = Frame(_proto=self._rom_root)  #fixme: ROM ??

	vb = {"left": 0,
	      "top": 0,
	      "right": self._display.getWidth(),
	      "bottom": self._display.getHeight()}

	self._root = Frame(_proto=self._rom_root_wrapper,
			   _parent=NIL,
			   viewBounds=vb)

	self._evt_q = []

	#-- initialise globals
	self._gnut.log("globals ")
	am = self._gnut._am

	am.DefGlobalFn("GetRoot", self.GetRoot)
	am.DefGlobalFn("BuildContext", self.BuildContext)
	am.DefGlobalFn("GetView", self.GetView)
	am.DefGlobalFn("RefreshViews", self.RefreshViews)
	am.DefGlobalFn("SetValue", self.SetValue)
	am.DefGlobalFn("AddStepView", self.AddStepView)
	am.DefGlobalFn("RemoveStepView", self.RemoveStepView)
	am.DefGlobalFn("AddView", self.AddView)
	am.DefGlobalFn("RemoveView", self.RemoveView)
	am.DefGlobalFn("AsyncConfirm", self.AsyncConfirm)
	am.DefGlobalFn("ModalConfirm", self.ModalConfirm)
	am.DefGlobalFn("RelBounds", self.RelBounds)
	am.DefGlobalFn("SetBounds", self.SetBounds)
	am.DefGlobalFn("ButonBounds", self.ButtonBounds)
	am.DefGlobalFn("PictBounds", self.PictBounds)
	am.DefGlobalFn("StdButtonWidth", self.StdButtonWidth)
	am.DefGlobalFn("HiliteOwner", self.HiliteOwner)
	am.DefGlobalFn("GetHiliteOffsets", self.GetHiliteOffsets)
	am.DefGlobalFn("TieViews", self.TieViews)
	am.DefGlobalFn("GetViewFlags", self.GetViewFlags)
	am.DefGlobalFn("Visible", self.Visible)

	return


    def register(self, view):
	"""Register a new view object with the manager."""

	self.lst_v.append(view)
	return


    def NewPView(self, frm_view):
	"""Create a new primitive view object.

	*frm_view* -- view frame object
	Returns    -- ???
	Exceptions -- ???

	PView instances are the things that actually call the screen
	device to draw themselves on the display of the Gnuton.  So
	they are generic, in that they use the abstract ScreenDevice
	drawing API.

	Initially there might be some Tkinter-isms here, but they'll
	need to be weeded out."""

	#-- validate the view frame
	if not frm_view.has_key("viewClass"):
	    raise VM_BadViewFrame(frm_view)

	#-- create the PView
	pv = PView(self._gnut, frm_view)

	#-- store PView in table
	vid = 1
	self._d_pview[vid] = pv

	#-- update view frame
	frm_view["viewCObject"] = vid

	return


    def SendMessage(self, ref_frame, str_msg, t_param):
	"""Send a message to a view.

	*ref_frame* -- destination view frame
	*str_msg*   -- string name of method slot
	*t_param*   -- tuple of parameters
	Returns     -- (none)
	Exceptions  -- ???

	Queues a message for a view frame."""

	self._evt_q.append((ref_frame, str_msg, t_param))
	return


    def DoOneEvent(self):
	""" """

	if len(self._evt_q) > 0:
	    evt = self._evt_q[0]
	    self._evt_q = self._evt_q[1:]

	    ref_frame, str_msg, t_param = evt

	    #-- check frame
	    if type(ref_frame) == types.DictType:
		print "SendMessage to frame", str_msg

	    elif type(ref_frame) == types.InstanceType:
		print "SendMessage to PClass (%s):%s()" % \
		      (ref_frame.__class__.__name__, str_msg)
		params = tuple([ref_frame] + list(t_param))
		apply(ref_frame.__class__.__dict__[str_msg], params)

	    else:
		print "SendMessage broke ...", str_msg

	return


#############################################################################

    #
    #   internal methods
    #

    def _getDisplay(self):
	"""Lookup the unit's display device in the HardwareManager."""

	#-- lookup display drivers
	lst_drv = self._gnut._hm.ListDriversForType("display")
	drv_id = lst_drv[0]
	drv_ref = self._gnut._hm.Driver(drv_id)
	drv_nam = drv_ref.Name()

	#-- lookup devices from display driver
	lst_dev = self._gnut._hm.ListDevicesForDriver(drv_nam)
	dev_id = lst_dev[0]

	return self._gnut._hm.Device(dev_id)


#############################################################################

    #
    #   functions for globals
    #

    def GetRoot(self):
	"""Return a reference to the root view."""

	return self._root


    def BuildContext(self, template):
	"""Create a view from the supplied template."""

	#-- validate template
	if not template.has_key("viewClass"):
	    raise VM_FrameNotTemplate("No viewClass", template)

	if not template.has_key("viewBounds"):
	    raise VM_FrameNotTemplate("No viewBounds", template)

	if template.has_key("viewCObject"):
	    raise VM_FrameIsView(template)

	#-- lookup view class
	#fixme: needs to use real lookup function/method
	view_class = self._display.getViewClass(template["viewClass"])

	#-- create view
	ref_view = view_class(self, template, self._root)

	return


    def GetView(view):
	pass

    def RefreshViews():
	pass

    def SetValue(vuew, slotSymbol, value):
	pass

    def AddStepView(parentView, childTemplate):
	pass

    def RemoveStepView(parentView, childView):
	pass

    def AddView(parentView, childTemplate):
	pass

    def RemoveView(parentView, childView):
	pass

    def AsyncConfirm(confirmMesage, buttonList, fn):
	pass

    def ModalConfirm(confirmMessage, buttonList):
	pass

    def RelBounds(left, top, width, height):
	pass

    def SetBounds(left, top, right, bottom):
	pass

    def ButtonBounds(width):
	pass

    def StdButtonWidth(str):
	pass

    def PictBounds(name, left, top):
	pass

    def HiliteOwner():
	pass

    def GetHiliteOffsets():
	pass

    def TieViews(mainView, dependentView, methodName):
	pass

    def GetViewFlags(template):
	pass

    def Visible(view):
	pass


#############################################################################

if __name__ == "__main__":
    pass


#############################################################################
