/*
 * AppController.m
 * Implementation for the application delegate.
 *
 * Copyright (c) 1997 21st Century Software, New York City.
 *
 * See the file Reference/License.rtf for information on usage and
 * redistribution of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by David Young <dwy@ace.net>, 1997.
 */

#import <foundation/NSData.h>
#import <foundation/NSException.h>
#import <foundation/NSString.h>
#import <foundation/NSUtilities.h>

#import <TCKit/TCKit.h>

#import "Newton.h"
#import "Preferences.h"

@implementation AppController

AppController *appController = nil;

+ (AppController *)controller
	/*" Returns the single instance of the application controller. "*/
{
	return appController;
}

- init
{
	[super init];

	appController = self;

	return self;
}

- (NSArray *)arrayOfAcceptedExtensions
	/*" Returns an autoreleased array containing the extensions the
		application will open. "*/
{
	return [LOCALIZED_STRING(@"acceptedExtensions", @"PKG, newtpkg, pkg", @"")
		componentsSeparatedByString:@", "];
}

- (BOOL)extensionIsAccepted:(NSString *)extension
	/*" Returns YES if extension is an allowed extension. "*/
{
	return [[self arrayOfAcceptedExtensions] containsObject:extension];
}

- (void)openPackage:(NSString *)packagePath
	/*" Opens the package specified by packagePath. "*/
{
	[[PackageController alloc] initWithContentsOfFile:packagePath];
}

- openNewtonPackage:sender
	/*" IB target method. Presents an OpenPanel to permit the user to 
		open one or more packages to install. "*/
{
	OpenPanel *openPanel = [[OpenPanel new] allowMultipleFiles:YES];
	const char *fileTypes[] = {"pkg", "PKG", "newtpkg", NULL};
	char **fileNames;
	unsigned i;

	if ([openPanel runModalForTypes: fileTypes]) {
		fileNames = (char **)[openPanel filenames];
		i = 0;
		while (fileNames[i] != NULL)
			[self openPackage:[NSString stringWithFormat:@"%s/%s",
					[openPanel directory], fileNames[i++]]];
	}

    return self;
}

- clearDisplay:sender
	/*" IB target method. Clears the display on the Newton console. "*/
{
	[consoleView setText:""];
	return self;
}

- sendPasteboard:sender
	/*" IB target method. Transmits the pasteboard, in raw mode. "*/
{
	[newton sendPasteboard];

	return self;
}

- sendFile:sender binary:(BOOL)binaryFlag
	/*" Prompts for a file or files to open and transmits them raw
		across the serial link. If binaryFlag is NO then UNIX to
		Macintosh linefeed to carriage return substitution is performed. 
	"*/
{
	OpenPanel *openPanel = [OpenPanel new];
	const char *fileTypes[] = { NULL };
	char **fileNames;
	unsigned i;

	if ([openPanel runModalForTypes: fileTypes]) {
		fileNames = (char **)[openPanel filenames];
		i = 0;
		while (fileNames[i] != NULL) {
			NSString *fileName = [NSString stringWithFormat:@"%s/%s",
						[openPanel directory], fileNames[i]];

		NS_DURING
			[newton sendContentsOfFile:fileName binary:binaryFlag];
		NS_HANDLER
			NXRunAlertPanel(LOCALIZED_CSTRING(@"errFileSend",
					@"File Send Exception", @""),
				[[exception exceptionReason] cString], 
				LOCALIZED_CSTRING(@"cancel", @"Cancel", @""), NULL, NULL);
		NS_ENDHANDLER

			++i;
		}
	}

    return self;
}

- sendTextFile:sender
	/*" An IB target method which is a cover for #sendFile:binary:NO. "*/
{
	return [self sendFile:self binary:NO];
}

- sendBinaryFile:sender
	/*" An IB target method which is a cover for #sendFile:binary:YES. "*/
{
	return [self sendFile:self binary:YES];
}

/* The _instantiate* methods are private stubs for instantiating various
 * kinds of serial connections. Only _instantiateSerialPort is fleshed out;
 * in future versions we hope to be able to provide Ethernet and ADSL.
 */

- _instantiateSerialPort:(unsigned)portSpec
	/*" The serial port instantiator. Creates a serial link object and
		applies the app's preferences to that object's communications
		parameters.
	"*/
{
	TCSerialPort *portObject = nil;

	switch (portSpec) {
	case HP_PORTA:
		portObject = [[TCSerialPort alloc] initWithDevice:TC_FASTPORTA];
		break;
	case HP_PORTB:
		portObject = [[TCSerialPort alloc] initWithDevice:TC_FASTPORTB];
		break;
	}

	[portObject setBaud:[[Preferences preferences] portSpeed]];
	[portObject connect];

	return portObject;
}

- _instantiateAppleTalk
	/*" The ADSL instantiator is not yet finished. "*/
{
	return nil;
}

- _instantiateEthernet
	/*" The Ethernet instantiator is not yet finished. "*/
{
	return nil;
}

- startNewtonConnection:sender
	/*" An IB target method which "resets" the Newton connection; calls
		the appropriate instantiator for the link type as described in
		Preferences, and applies parameters.
	"*/
{
	NSObject *commObject;

	if (newton)
		[newton release];

	commObject = nil;

	[statusView setStringValue:"Opening connection..."];

NS_DURING
	switch ([[Preferences preferences] connectionPort]) {
		case HP_UNSET:
			NXRunAlertPanel(LOCALIZED_CSTRING(@"errCommUnsetTitle", 
					@"Communications Error", @""),
				LOCALIZED_CSTRING(@"errCommUnsetDetail", 
					@"No communications port set. Please set one in the Preferences panel", @""), 
				LOCALIZED_CSTRING(@"cancel", @"Cancel", @""), NULL, NULL);
			[[Preferences preferences] makeKeyAndOrderFront:self];
			break;

		case HP_PORTA:
		case HP_PORTB:
			commObject = [self _instantiateSerialPort:
				[[Preferences preferences] connectionPort]];
			break;

		case HP_APPLETALK:
			commObject = [self _instantiateAppleTalk];
			break;
		case HP_ETHERNET:
			commObject = [self _instantiateEthernet];
			break;

		default:
			break;
	}

	if (commObject) {
		newton = [[NewtonController alloc] initWithObject:commObject];
		[(NewtonController *)newton setDelegate:self];

		[commObject release];
		[statusView setStringValue:LOCALIZED_CSTRING(@"statusOpen",
			@"Newton connection open.", @"")];
	}
NS_HANDLER
	NXRunAlertPanel(LOCALIZED_CSTRING(@"errSerial", @"Serial Error", @""),
		[[exception exceptionReason] cString], 
		LOCALIZED_CSTRING(@"cancel", @"Cancel", @""), NULL, NULL);

		[statusView setStringValue:LOCALIZED_CSTRING(@"statusFailed",
			@"Newton connection failed!", @"")];
NS_ENDHANDLER

    return self;
}

- (NewtonController *)newton
	/*" Returns the currently connected Newton, if available. "*/
{
	return newton;
}

/*
 * Newton delegate methods
 */

- (void)newton:(NewtonController *)aNewton didNotProcessData:(NSData *)data
	/*" Appends unrecognized ASCII stream data to the console. "*/
{
	NSString *string = [[[NSString alloc] initWithData:data
		encoding:NSASCIIStringEncoding] autorelease];

	[(Text *)consoleView appendString:string];
}

@end
