/*
 * NewtonController.m
 * Model a Newton MessagePad device as an object.
 *
 * Copyright (c) 1997 21st Century Software, New York City.
 *
 * See the file Reference/License.rtf for information on usage and
 * redistribution of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by David Young <dwy@ace.net>, 1997.
 */

#import <foundation/NSString.h>
#import <foundation/NSData.h>

#import <TCKit/TCKit.h>

#import "Newton.h"
#import "Preferences.h"

@interface NSObject (CommunicationsObject)
- (void)connect;
- (int)descriptor;
- (void)disconnect;
@end

@implementation NewtonController

- initWithObject:(NSObject *)anObject;
{
	[super init];

	commObject = [anObject retain];
	[TCStreamMonitor notifyObject:self 
		descriptor:[commObject descriptor]
		selector:@selector(serialPortActivity:)];

	stream = [[TCStream alloc] initWithDescriptor:[commObject descriptor]];
	[stream setCloseOnDealloc:NO];

	isDocked = 0;

	return self;
}

- delegate
{
	return delegate;
}

- (void)setDelegate:_delegate
{
	delegate = _delegate;
}

- (BOOL)isDocked
{
	return isDocked;
}

- (BOOL)isAwaitingAsynchronousEvent
{
	return isAwaitingAsynchronousEvent;
}

+ (NewtonEvent *)dataContainsNewtonEvent:(NSData *)data
{
	NewtonEvent *nePtr = (NewtonEvent *)[data bytes];

	if (strncmp(nePtr->eventHeader, kDNEHeader, sizeof(kDNEHeader)) != 0)
		return NULL;
	if (strncmp(nePtr->eventHeaderExtended, kDNEHeader, 
		sizeof(kDNEHeaderExtended)) != 0)
		return NULL;

	return nePtr;
}

- (NSData *)getOrRejectNextNewtonEvent
{
	NSData *streamData = [stream readAvailable];
	NewtonEvent *nePtr;

	if ((nePtr = [NewtonController dataContainsNewtonEvent:streamData])) {
		printf("woohoo! got event %s\n", nePtr->eventCommand);
		return nil;
	}

	return streamData;
}

- (void)serialPortActivity:(int)descriptor
{
	NSData *streamData;

	if ([delegate respondsTo:@selector(newtonWillProcessInput:)])
		[delegate newtonWillProcessInput:self];

	/* check to see if this is a NewtonEvent. */
	streamData = [self getOrRejectNextNewtonEvent];
	if (streamData != nil) 
		if ([delegate respondsTo:@selector(newtonDidProcessInput:)])
			[delegate newtonDidProcessInput:self];
	else
		if ([delegate respondsTo:@selector(newton:didNotProcessData:)])
			[delegate newton:self didNotProcessData:streamData];
}

- (TCStream *)stream
{
	TCStream *streamCopy = [[stream copy] autorelease];
	[streamCopy setCloseOnDealloc:NO];

	return streamCopy;
}

- (int)descriptor
{
	return [commObject descriptor];
}

- (void)sendPasteboard
{
	Pasteboard *pasteboard = [Pasteboard newName:NXGeneralPboard];
	char *data;
	int length;

	if ([pasteboard readType:NXAsciiPboardType data:&data length:&length]) {
		[[stream writeString:[NSString stringWithCString:data]] flush];
		[pasteboard deallocatePasteboardData:data length:length];
	}
}

- (void)sendContentsOfFile:(NSString *)fileName binary:(BOOL)binaryFlag
{
	NSData *fileData = [NSData dataWithContentsOfFile:fileName];

	if (binaryFlag)
		[[stream writeData:fileData] flush];
	else {
		NSString *fileString = [[NSString alloc]
			initWithData:fileData encoding:NSASCIIStringEncoding];
		NSArray *fileLines = [fileString componentsSeparatedByString:@"\n"];
		NSEnumerator *e = [fileLines objectEnumerator];
		NSString *line;

		while (line = [e nextObject]) {
			[stream writeString:line];
			[stream writeString:@"\r\n"];
		}
		[stream flush];
	}
}

- (void)dealloc
{
	[stream release];
	if (commObject != nil) {
		[TCStreamMonitor ignoreDescriptor:[commObject descriptor]];
		[commObject release];
	}

	[super dealloc];
}

@end
