/*
 * Preferences.m
 * Implementation for a generic Preferences manager.
 *
 * Copyright (c) 1997 21st Century Software, New York City.
 *
 * See the file Reference/License.rtf for information on usage and
 * redistribution of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by David Young <dwy@Eng.ACE.NET>, 1997.
 */

#import "Preferences.h"

#import <foundation/NSString.h>
#import <foundation/NSArray.h>
#import <foundation/NSUtilities.h>

#import "SwappablePreferences.h"
#import "NoPreferences.h"

@implementation Preferences
	/**" 
	 ** The Preferences class is an attempt to abstract application
	 ** preference management. The class is generally built within an
	 ** Application subproject and needs the supplied .nib files
	 ** #Preferences.nib and #NoPreferences.nib. These provide the
	 ** interfaces for the Panel and the No Preferences view.
	 **
	 ** The preferences class forwards messages to each of its submodules
	 ** in the order they are loaded. Failing that, it will try to 
	 ** forward to the Panel it owns. 
	 **
	 ** Construction of this object's nib file is very important. The
	 ** example should be copied and modified. The modules' classes are
	 ** instantiated by name from the table 
	"**/

Preferences *_preferences_sharedInstance;

- init
{
	[super init];

	_preferences_sharedInstance = self;

	if (panel == nil)
		[NXApp loadNibSection:"Preferences.nib" owner:self];

	emptySet = [[NoPreferences alloc] initWithParent:self];

	[self reloadModules];
	[self reloadButtonTitles];

	[self changeViewToTag:1];
	[[panel contentView] addSubview:[[self currentViewSet] enclosingView]];

	return self;
}

- (void)reloadModules
	/*" Reloads the module list from the string table in the nib file. 
	"*/
{
	int i;
	id	classProxy;
	const char *className;

	if (viewSetArray)
		[viewSetArray release];

	viewSetArray = [[NSMutableArray alloc] init];

	// load modules based on the string table
	i = 1;
	while ((className = [moduleTable valueForStringKey:
		[[NSString stringWithFormat:@"Node%d", i] cString]])) {
		SwappablePreferences *tempSet;
		classProxy = [(NXBundle *)[NXBundle mainBundle] classNamed:className];
		tempSet = [[classProxy alloc] initWithParent:self];
		if (tempSet != nil)
			[viewSetArray addObject:tempSet];
		else
			NXRunAlertPanel ("Preferences Error", 
				"Can't load preferences module %s\n", 
				"Cancel", NULL, NULL, className);

		++i;
	}

	currentSet = [viewSetArray lastObject];
}

- (void)reloadButtonTitles
	/*" Resets the titles of the buttons in the view selection Matrix to
		the titles supplied by the modules. The titles only take effect
		if the button does not have an icon. "*/
{
	NSEnumerator *e = [viewSetArray objectEnumerator];
	SwappablePreferences *obj;
	unsigned i = 0;

	while (obj = [e nextObject]) {
		if ([[buttons cellAt:0 :i] icon] == NULL)
			[[buttons cellAt:0 :i] setTitle:[[obj buttonTitle] cString]];
		++i;
	}
}

- (SwappablePreferences *)currentViewSet
	/*" Returns the currently visible view set. "*/
{
	return currentSet;
}

- (SwappablePreferences *)emptyViewSet
	/*" Returns an empty view set which contains a TextField that says
		"No Preferences." 
	"*/
{
	return emptySet;
}

- changeViewToViewSet:(SwappablePreferences *)aSet
	/*" Switches to the SwappablePreferences aSet. 
	"*/
{
	SwappablePreferences *oldSet;

	oldSet = [self currentViewSet];
	[panel setTitle:[[NSString stringWithFormat:@"%@ Preferences",
		[aSet windowTitle]] cString]];

	if ([oldSet respondsTo:@selector(willResignCurrentView:)])
		[oldSet willResignCurrentView:self];

	if ([aSet respondsTo:@selector(willBecomeCurrentView:)])
		[aSet willBecomeCurrentView:self];

	[[[oldSet enclosingView] superview] 
		replaceSubview:[oldSet enclosingView] 
		with:[aSet enclosingView]];

	currentSet = aSet;
	[[currentSet enclosingView] update];

	return self;
}

- changeViewToTag:(int)tag
	/*" Switches to the SwappablePreferences specified by tag. 
		The tag index must be equivalent to the Node number specified
		in the moduleTable string table. Calls #changeViewToViewSet:.
	"*/
{
	tag--;
	if (tag < [viewSetArray count])
		return [self changeViewToViewSet:[viewSetArray objectAtIndex:tag]];
	else
		return [self changeViewToViewSet:emptySet];
}

- changeView:sender
	/*" Switches to the SwappablePreferences specified by the sender's tag,
		or if the sender responds to the #selectedCell message, the sender's
		selected cell's tag. Calls #changeViewToTag:. 
	"*/
{
	if ([sender respondsTo:@selector(selectedCell)])
		return [self changeViewToTag:[[sender selectedCell] tag]];
	else
		return [self changeViewToTag:[sender tag]];
}

// anything else goes to the prefsViews; if not, to the window.

- (BOOL)respondsTo:(SEL)aSelector
{
	NSEnumerator *e = [viewSetArray objectEnumerator];
	id obj;

	while (obj = [e nextObject])
		if ([obj respondsTo:aSelector])
			return YES;

    if ([panel respondsTo:aSelector])
        return YES;

	return [super respondsTo:aSelector];
}

- forward:(SEL)aSelector :(marg_list)argFrame
{
	NSEnumerator *e = [viewSetArray objectEnumerator];
	id obj;

    if ([panel respondsTo:aSelector])
        return [panel performv:aSelector :argFrame];

	while (obj = [e nextObject])
		if ([obj respondsTo:aSelector])
			return [obj performv:aSelector :argFrame];

	return nil;
}

@end
