/*
 * SlurpeeController.m
 * Implementation for the application's interface to Steve Weyer's Slurpee.
 *
 * Copyright (c) 1997 21st Century Software, New York City.
 *
 * See the file Reference/License.rtf for information on usage and
 * redistribution of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by David Young <dwy@ace.net>, 1997.
 */

#import <foundation/NSString.h>
#import <foundation/NSData.h>
#import <foundation/NSException.h>

#import <TCKit/TCKit.h>

#import "Newton.h"
#import "Preferences.h"

@implementation SlurpeeController

- init
{
	[super init];

	[NXApp loadNibSection:"SlurpeeController.nib" owner:self];
	[SlurpeeAdaptor setUnfiledFolderName:[[[Preferences preferences]
		standardFolderList] 
		objectAtIndex:0]];

	return self;
}

- (SlurpeeAdaptor *)adaptorFromNewton:(NewtonController *)controller
{
	SlurpeeAdaptor *adaptor;

	adaptor = [SlurpeeAdaptor adaptorWithStream:[controller stream]];
	[adaptor setDelegate:self];

	return adaptor;
}

- (SlurpeeAdaptor *)adaptorFromCurrentNewton
{
	return [self adaptorFromNewton:[appController newton]];
}

- (BOOL)newtonIsConnected
{
	if ([appController newton] == nil) {
		NXRunAlertPanel(LOCALIZED_CSTRING(@"errNoNewtonTitle", 
			@"Newton Error", @""), 
			LOCALIZED_CSTRING(@"errNoNewtonDetail",
				@"No MessagePad device is connected.", @""), 
			LOCALIZED_CSTRING(@"cancel", @"Cancel", @""), NULL, NULL);
		return NO;
	}

	return YES;
}

- orderFrontCommandPanel:sender
{
	[commandPanel makeKeyAndOrderFront:self];
	[commandParameterForm selectTextAt:0];

	return self;
}

- orderFrontKeyboardPanel:sender
{
	[keyboardPanel makeKeyAndOrderFront:self];
	[keyboardPanel makeFirstResponder:keyboardResponderView];

	return self;
}

- deliverCommand:sender
{
	SlurpeeAdaptor *adaptor;
	if (![self newtonIsConnected])
		return self;

	[(Window *)commandPanel close];

	adaptor = [self adaptorFromCurrentNewton];
	[adaptor dumpSoupNamed:[NSString stringWithCString:
			[commandParameterForm stringValueAt:0]]
		options:[NSString stringWithCString:
			[commandParameterForm stringValueAt:1]]
		formatting:[NSString stringWithCString:
			[commandParameterForm stringValueAt:2]]];

	return self;
}

- listPackages:sender
{
	SlurpeeAdaptor *adaptor;

	if (![self newtonIsConnected])
		return self;

	[packageListWindow setTitle:LOCALIZED_CSTRING(@"packageWindowLoadingTitle",
		@"Loading Packages...", @"")];
	[packageListWindow makeKeyAndOrderFront:self];

NS_DURING
	adaptor = [self adaptorFromCurrentNewton];
	packageListArray = [adaptor arrayOfSoupNames];

	[packageListBrowser reloadColumn:0];
	[packageListWindow setTitle:
		LOCALIZED_CSTRING(@"packageWindowTitle", @"Packages", @"")];
NS_HANDLER
	NXRunAlertPanel(LOCALIZED_CSTRING(@"errSerial", @"Serial Error", @""),
		[[exception exceptionReason] cString], 
		LOCALIZED_CSTRING(@"cancel", @"Cancel", @""), NULL, NULL);
	[(Window *)packageListWindow close];
NS_ENDHANDLER

	return self;
}

- (unsigned)shouldConvertFileNamed:(NSString *)name
{
	if ([[Preferences preferences] performsConversions]) {
		NSEnumerator *alwaysEnumerator = [[[Preferences preferences]
			alwaysConvertExtensions] objectEnumerator];
		NSEnumerator *neverEnumerator = [[[Preferences preferences]
			neverConvertExtensions] objectEnumerator];
		NSString *extension;

		while (extension = [alwaysEnumerator nextObject])
			if ([name matchesWildcard:[NSString stringWithFormat:
				@"*.%@", extension]])
				return TCExportWillConvert;

		while (extension = [neverEnumerator nextObject])
			if ([name matchesWildcard:[NSString stringWithFormat:
				@"*.%@", extension]])
				return TCExportWillNotConvert;
	}

	return TCExportWillPrompt;
}

- notesExportFile:sender
{
	SlurpeeAdaptor *adaptor;
	OpenPanel *openPanel = [[OpenPanel new] allowMultipleFiles:YES];
	const char *fileTypes[] = { NULL };
	char **fileNames;
	unsigned i, alertSpec;

	if (![self newtonIsConnected])
		return self;

	adaptor = [self adaptorFromCurrentNewton];
	if ([openPanel runModalForTypes: fileTypes]) {
		fileNames = (char **)[openPanel filenames];
		i = 0;
		while (fileNames[i] != NULL) {
		NS_DURING
			NSString *pathString = [NSString stringWithFormat:@"%s/%s",
					[openPanel directory], fileNames[i]];

			if ([self shouldConvertFileNamed:pathString] == TCExportWillPrompt)
				alertSpec = (NXRunAlertPanel(
					LOCALIZED_CSTRING(@"askConvert", @"Convert?", @""),
					LOCALIZED_CSTRING(@"alertPromptForTextConversion",
						@"Perform UNIX to Mac text conversion on %s?",
						@""),
					LOCALIZED_CSTRING(@"NO", @"No", @""), 
					LOCALIZED_CSTRING(@"YES", @"Yes?", @""), 
					NULL, fileNames[i]) == 0);
			else
				alertSpec = ([self shouldConvertFileNamed:
					pathString] == TCExportWillConvert);


			[adaptor createNoteWithContentsOfFile:pathString
				convertLineFeed:alertSpec
			font:[SlurpeeAdaptor newtonFontWithFamily:
				[[Preferences preferences] defaultFontFamily]
			face:0 
			size:[[Preferences preferences] defaultFontSize]]
			folder:
				[[Preferences preferences] defaultFolderForNotes]];
		NS_HANDLER
			NXRunAlertPanel(LOCALIZED_CSTRING(@"errSerial", 
				@"Serial Error", @""),
				[[exception exceptionReason] cString], 
				LOCALIZED_CSTRING(@"cancel", @"Cancel", @""), NULL, NULL);
		NS_ENDHANDLER
			i++;
		}
	}

    return self;
}

- notesImportFile:sender
{
	return self;
}

- notesNew:sender
{
	[[NoteController alloc] initWithParent:self];
	return self;
}

- transferPackage:sender
{
	SlurpeeAdaptor *adaptor;

	if (![self newtonIsConnected])
		return self;

NS_DURING
	adaptor = [self adaptorFromCurrentNewton];

	[[PackageController alloc] initWithData:
		[adaptor soupNamed:[NSString stringWithCString:
			[[packageListBrowser selectedCell] stringValue]]]
		title:@"Downloaded Package"];
NS_HANDLER
	NXRunAlertPanel(LOCALIZED_CSTRING(@"errSerial", 
		@"Serial Error", @""),
		[[exception exceptionReason] cString], 
		LOCALIZED_CSTRING(@"cancel", @"Cancel", @""), NULL, NULL);
NS_ENDHANDLER

	return self;
}

- (void)adaptorWillBeginTransfer:(SlurpeeAdaptor *)adaptor
{
	[packageListStatus setStringValue:""];
}

- (void)adaptor:(SlurpeeAdaptor *)adaptor transferInProgress:(NSData *)data
{
	char buf[32];

	sprintf(buf, "%d", [data length]);
	[packageListStatus setStringValue:buf];
}

- (void)adaptor:(SlurpeeAdaptor *)adaptor finishedTransfer:(NSString *)data
{
	[packageListStatus setStringValue:""];
}

- browserAction:sender
{
	[packageListTransferButton setEnabled:([sender selectedCell] != nil)];
	return self;
}

- (int) browser:sender getNumRowsInColumn:(int)column
{
	if (column == 0) 
		return [packageListArray count];
	else
		return 0;
}

- browser:sender loadCell:cell atRow:(int)row inColumn:(int)column
{
	// we've only got one column
	if (column == 0) {
		[[[cell setStringValue:[[packageListArray objectAtIndex:row] cString]]
			setLeaf:YES]
			setLoaded:YES];
	}

	return self;
}

@end
