/*
 * TCNewtonPackage.m
 * Implementation loading, parsing, etc of a Newton package object.
 *
 * Copyright (c) 1997 21st Century Software, New York City.
 *
 * See the file doc/LICENSE for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by David Young <dwy@ace.net>, 1997.
 *         for NEXTSTEP 3.3, OpenStep ready.
 */

#import "TCNewtonPackage.h"
#import <foundation/NSUtilities.h>

#import <libc.h>

@implementation TCNewtonPackage

static NSTimeInterval NEpochDeltaInterval;
static NSBundle *bundle;

+ initialize
{
	[super initialize];

	NEpochDeltaInterval = NEpochYearDelta;
	NEpochDeltaInterval *= 365.25;
	NEpochDeltaInterval *= 24;
	NEpochDeltaInterval *= 3600;

	bundle = [NSBundle bundleForClass:[self class]];

	return self;
}

+ packageWithContentsOfFile:(NSString *)fileName
{
	return [[[self alloc] initWithContentsOfFile:fileName] autorelease];
}

+ packageWithData:(NSData *)someData
{
	return [[[self alloc] initWithData:someData] autorelease];
}

- initWithContentsOfFile:(NSString *)fileName
{
	NSData *pData;

	pData = [[NSData alloc] initWithContentsOfFile:fileName];
	if (pData == nil)
		[NSException raise:@"TCNewtonPackageException"
			format:[bundle localizedStringForKey:@"errorPkgOpen"
				value:@"Cannot open %@" 
				comment:@"" table:@"TCNewtonPackage"],
			fileName];

	return [self initWithDataNoCopy:pData];
}

- initWithData:(NSData *)someData
{
	return [self initWithDataNoCopy:[someData copy]];
}

- initWithDataNoCopy:(NSData *)someData
{
	unsigned i;
	char *stringTemp;

	[super init];

	packageData = someData;
	packageHeader = (NPHeader *) [packageData bytes];

	/* check package signature */
	if (strncmp(packageHeader->signature, NEWTON_PACKAGE_MAGIC, 7) != 0)
		[NSException raise:@"TCNewtonPackageException"
			format:[bundle localizedStringForKey:@"errorPkgParse"
				value:@"Bad or unknown package signature: \"%@\""
				comment:@"" table:@"TCNewtonPackage"],
			[self packageSignature]];

	packagePartHeaders = [[[NSMutableData alloc] initWithLength:
		(([self partCount]+1)*sizeof(NPPartHeader *))]
		mutableBytes];
	for (i = 0; i < [self partCount]; ++i) {
		packagePartHeaders[i] = 
			(NPPartHeader *)((char *)packageHeader+sizeof(NPHeader)+(i * sizeof(NPPartHeader)));
	}

	/* I wouldn't have this problem if I wrote in m68k assembly */
	stringTemp = 
		((char *)(packagePartHeaders[i-1]))+sizeof(NPPartHeader);
	stringTable = (unichar *)stringTemp;

	return self;
}

- (void)dealloc
{
	[packageData release];
}

- (NPHeader *)headerStructure
{
	return packageHeader;
}

- (const unichar *)stringTable
{
	return (const unichar *)stringTable;
}

- (NSString *)packageSignature
{
	return [NSString stringWithCString:packageHeader->signature length:8];
}

- (BOOL)isDispatchOnly
{
	return NXSwapBigIntToHost(packageHeader->flags) &
		TCNewtonPackageIsDispatchOnly;
}

- (BOOL)isCopyProtected
{
	return NXSwapBigIntToHost(packageHeader->flags) &
		TCNewtonPackageIsCopyProtected;
}

- (BOOL)isCompressed
{
	return NXSwapBigIntToHost(packageHeader->flags) &
		TCNewtonPackageIsUncompressed;
}

- (NSString *)stringWithInfoRef:(NPInfoRef)aRef type:(unsigned)refType
{
	NSData *stringData;
	NPInfoRef swappedRef;
#ifdef __LITTLE_ENDIAN__
	unsigned short tmp;
#endif

	swappedRef.value = NXSwapBigIntToHost(aRef.value);
	if (swappedRef.value == 0)
		return @"";

#ifdef __LITTLE_ENDIAN__
	tmp = swappedRef.ref.length;
	swappedRef.ref.length = swappedRef.ref.offset;
	swappedRef.ref.offset = tmp;
#endif

	/* refType constants are byte sizes. */
	stringData = [NSData dataWithBytes:
		(char *)stringTable+swappedRef.ref.offset
		length:swappedRef.ref.length];

	switch(refType) {
	case NPInfoRefIsASCII:
		return [[[NSString alloc] initWithData:stringData
			encoding:NSASCIIStringEncoding] autorelease];

	case NPInfoRefIsUnicode:
		return [[[NSString alloc] initWithData:stringData
			encoding:NSUnicodeStringEncoding] autorelease];

	default:
		return nil;
	}
}

- (NSString *)copyrightInfo
{
	return [self stringWithInfoRef:packageHeader->copyrightRef
		type:NPInfoRefIsUnicode];
}

- (NSString *)packageName
{
	return [self stringWithInfoRef:packageHeader->nameRef
		type:NPInfoRefIsUnicode];
}

- (unsigned)packageVersion
{
	return NXSwapBigIntToHost(packageHeader->version);
}

- (NSDate *)creationDate
{
	if (NXSwapBigIntToHost(packageHeader->creationDateRep) > 0) 
		return [[[NSCalendarDate alloc] initWithTimeIntervalSinceReferenceDate:
			NXSwapBigIntToHost(packageHeader->creationDateRep)
				+ NEpochDeltaInterval]
			autorelease];
	else
		return nil;
}

- (unsigned)headerSize
{
	return NXSwapBigIntToHost(packageHeader->headerSize);
}

- (unsigned)partCount
{
	return NXSwapBigIntToHost(packageHeader->numParts);
}

- (NSData *)packageData
{
	return [[packageData copy] autorelease];
}

- (NPPartHeader *)partHeaderAtIndex:(unsigned)anIndex
{
	if (anIndex > ([self partCount]-1))
		[NSException raise:NSRangeException
			format:[bundle localizedStringForKey:@"errorPkgPartIndex"
				value:@"Part index %d is out of bounds (max: %d)"
				comment:@"" table:@"TCNewtonPackage"],
			anIndex, ([self partCount]-1)];

	return packagePartHeaders[anIndex];
}

@end
