//
//  Newton Connection Library - Newton/UNIX Connectivity
//  Copyright (C) 1995-1997  A. E. Hall
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Library General Public
//  License as published by the Free Software Foundation; either
//  version 2 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Library General Public License for more details.
//
//  You should have received a copy of the GNU Library General Public
//  License along with this library; if not, write to the Free
//  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

char Newton_C_rcsid[] = "$Id: Newton.C,v 1.9 1997/10/29 22:26:53 spoonman Exp $";

#include <termios.h>
#include <fcntl.h>
#include <unistd.h>
#include <ctype.h>
#include <regex.h>
#include "Newton.H"

#define RCSID "$Id: " // this is so RCS doesn't replace it...
char Newton_C_copyright[] = RCSID NewtonTITLE " " NewtonCOPYRIGHT " $";

// Newton object constructor
Newton::Newton()
  {
  NewtonDeviceName = (const char *)NULL;    // reset the device name
  };

// Newton object destructor
Newton::~Newton()
  {
  Close();              // Close the NewtonDevice
  };

// Open the NewtonDevice
int Newton::Open(const String Device, int speed, int databits, char parity,
                 int stopbits)
  {
  return(Open(Device.chars(), speed, databits, parity, stopbits));
  };

int Newton::Open(const char *Device, int speed, int databits, char parity,
                 int stopbits)
  {
  if (Device != NULL)
    {
    NewtonDeviceName = (const char *)Device;
    }
  else
    {
    NewtonDeviceName = NewtonDEVICE;
    };
  NewtonDevice.open(NewtonDeviceName, 2048, ios::in|ios::out, 0600);
  if (!NewtonDevice.is_open())
    {
    fprintf(stderr, "Newton: Cannot open device: %s\n",
            NewtonDeviceName.chars());
    return(NewtonERROR);
    };
  NewtonFD = NewtonDevice.filedesc();  // set the File Descriptor
  setSerial(speed, databits, parity, stopbits);

  // kludge to work with lpkg integration in Package.C
  NewtonFDS = fdopen(NewtonFD,"r+");

  return(NewtonOK);
  };

// Setup the serial port
int Newton::setSerial(int speed, int databits, char parity, int stopbits)
  {
  struct termios ntty, otty;

  tcgetattr(NewtonFD, &otty);
  ntty = otty;
//  ntty.c_iflag = INPCK | IXON | IXOFF;
//  ntty.c_oflag = 0;
//  ntty.c_cflag = CS8 | PARENB | CREAD;
//  ntty.c_lflag = 0;
  ntty.c_cc[VMIN] = 1;
  ntty.c_cc[VTIME] = 0;

  // the following is equivalent to cfmakeraw(&ntty) which isn't
  // always supported...
  ntty.c_iflag &= ~(IGNBRK|BRKINT|PARMRK|ISTRIP|INLCR|IGNCR|ICRNL|IXON);
  ntty.c_oflag &= ~OPOST;
  ntty.c_lflag &= ~(ECHO|ECHONL|ICANON|ISIG|IEXTEN);
  ntty.c_cflag &= ~(CSIZE|PARENB);
  ntty.c_cflag |= CS8;

  // set the baud rate
  switch(speed)
    { 
#ifdef B2400
    case 2400:
        cfsetospeed(&ntty, B2400);
        cfsetispeed(&ntty, B2400);
      break;
#endif
#ifdef B9600
    case 9600:
        cfsetospeed(&ntty, B9600);
        cfsetispeed(&ntty, B9600);
      break;
#endif
#ifdef B19200
    case 19200:
        cfsetospeed(&ntty, B19200);
        cfsetispeed(&ntty, B19200);
      break;
#endif
#ifdef B38400
    case 38400:
        cfsetospeed(&ntty, B38400);
        cfsetispeed(&ntty, B38400);
      break;
#endif
#ifdef B57600
    case 57600:
        cfsetospeed(&ntty, B57600);
        cfsetispeed(&ntty, B57600);
      break;
#endif
    default:
      fprintf(stderr,"Newton: invalid serial port speed\n");
      return(NewtonERROR); 
      break;
    };

  // set the databits
  switch(databits)
    {
    case 5:
      ntty.c_cflag |= CS5;
      break;
    case 6:
      ntty.c_cflag |= CS6;
      break;
    case 7:
      ntty.c_cflag |= CS7;
      break;
    case 8:
      ntty.c_cflag |= CS8;
      break;
    default:
      fprintf(stderr,"Newton: invalid serial port data bits\n");
      return(NewtonERROR); 
    };

  // set the parity
  switch(toupper(parity))
    {
    case 'N':
      ntty.c_cflag &= ~(PARENB | PARODD);
      break;
    case 'O':
      ntty.c_cflag &= ~(PARENB | PARODD);
      ntty.c_cflag |= (PARENB | PARODD);
      break;
    case 'E':
      ntty.c_cflag &= ~(PARENB | PARODD);
      ntty.c_cflag |= (PARENB);
      break;
    default:
      fprintf(stderr,"Newton: invalid serial port parity\n");
      return(NewtonERROR); 
    };

  // set stop bits
  switch(stopbits)
    {
    case 1:
      ntty.c_cflag &= ~CSTOPB;
      break;
    case 2:
      ntty.c_cflag |= CSTOPB;
      break;
    default:
      fprintf(stderr,"Newton: invalid serial port stop bits\n");
      return(NewtonERROR); 
    };

  tcdrain(NewtonFD);
  tcflow(NewtonFD, TCOON);
  tcflow(NewtonFD, TCION);

  tcsetattr(NewtonFD, TCSANOW, &ntty);


  return(NewtonOK);
  };

// Close the NewtonDevice
int Newton::Close()
  {
  Drain();                                 // wait for all output to complete
  Flush();                                 // flush anything left
  NewtonDevice.close();                    // close the device
  NewtonDeviceName = (const char *)NULL;   // reset the device name
  NewtonFD = 0;                            // reset the file descriptor
  return(NewtonOK);
  };

// Flush the NewtonDevice
int Newton::Flush()
  {
  tcflush(NewtonFD, TCIOFLUSH);
  return(NewtonOK);
  };

// Wait for all output to go through NewtonDevice
int Newton::Drain()
  {
  tcdrain(NewtonFD);
  return(NewtonOK);
  };

// Return TRUE if NewtonDevice is open, FALSE if closed
int Newton::isOpen()
  {
  if (NewtonDevice.is_open())
    {
    return(TRUE);
    };
  return(FALSE);
  };

// Return TRUE if NewtonDevice is connected to Slurpee, FALSE if not
int Newton::isConnected()
  {
  int Connected=FALSE;
  String tmpString="";

  if (isOpen())
    {
/*
    WriteString("system\n");
    WriteString("{ country: \"string\" }\n");
    WriteString(SlurpeeDUMP);

    ReadLine(tmpString);

    // if we received something, Slurpee is connected,
    // flush the rest...
    Flush();

    if (tmpString.contains(RXalpha))
      {
      Connected = TRUE;
      printf("sleeping...");
      fflush(stdout);
      sleep(10);
      printf("done\n");
      fflush(stdout);
      };
*/
Connected = TRUE;
    };

  return(Connected);
  };

// Wake up the Newton if it's asleep!
int Newton::wakeUp()
  {
  // send NULL as the wakeup call!
  WriteChar('\000');
  return(NewtonOK);
  };

