#!/bin/sh

usage () {
  echo "Usage: build [option ...]" 1>&2
  echo "Options:" 1>&2
  echo "  -a <apxs_path>" 1>&2
  echo "  -p <install_prefix>" 1>&2
  echo "  -s <openssl_prefix>" 1>&2
  exit 1
}

prefix=/usr/local/shibboleth
apxs_path=`which apxs 2>/dev/null`
OPENSSL=

umask 022

case `uname` in
SunOS)
  platform=solaris
  if [ -f "/var/ssl/lib/pkgconfig/openssl.pc" ]; then
    default_openssl=/var/ssl
  elif [ -f "/var/local/ssl/lib/pkgconfig/openssl.pc" ]; then
    default_openssl=/var/local/ssl
  elif [ -f "/usr/local/ssl/lib/pkgconfig/openssl.pc" ]; then
    default_openssl=/usr/local/ssl
  elif [ -f "/usr/athena/lib/pkgconfig/openssl.pc" ]; then
    default_openssl=/usr/athena
  elif [ -f "/usr/sfw/lib/libssl.so" -a \
         -f "/usr/lib/pkgconfig/openssl.pc" ]; then
    default_openssl=/usr/sfw
    default_openssl_pkgconfig=/usr/lib/pkgconfig
  fi
  CC=cc
  CXX=CC
  make=gmake
  ;;
Linux)
  platform=linux
  if [ -f "/usr/athena/lib/pkgconfig/openssl.pc" ]; then
    default_openssl=/usr/athena
  elif [ -f "/usr/lib/pkgconfig/openssl.pc" ]; then
    default_openssl=/
    default_openssl_pkgconfig=/usr/lib/pkgconfig
  fi
  CC=gcc
  CXX=g++
  make=make
  ;;
*)
  echo "Unsupported platform" 1>&2
  exit 1
  ;;
esac
export CC CXX

# The xerces-c build takes a 32/64-bit configure option.
# (This likely doesn't matter on platforms we care about).
case `uname -m` in
x86_64)
  bits=64
  ;;
*)
  bits=32
  ;;
esac

while getopts a:p:s: opt; do
  case "$opt" in
  a)
    apxs_path=$OPTARG
    if [ ! -x "$apxs_path" ]; then
      echo "Cannot execute $apxs_path" 1>&2
      usage
    fi
    ;;
  p)
    prefix=$OPTARG
    ;;
  s)
    OPENSSL=$OPTARG
    if [ ! -d "$OPENSSL" ]; then
      echo "$OPENSSL is not a directory" 1>&2
      usage
    fi
    if [ ! -f "$OPENSSL/lib/pkgconfig/openssl.pc" ]; then
      echo "$OPENSSL does not contain required OpenSSL development tools" 1>&2
      usage
    fi
    ;;
  \?)
    usage
    ;;
  esac
done

if [ -z "$apxs_path" ]; then
  echo "Please provide the apxs executable path" 1>&2
  usage
fi

# Set the OPENSSL prefix if not yet set.  Also set PKG_CONFIG_PATH
# to point at the openssl config file, as the Shibboleth build system
# uses pkg-config to find openssl even when --with-openssl is given.
if [ -z "$OPENSSL" ]; then
  if [ -z "$default_openssl" ]; then
    echo "Please provide the OpenSSL prefix (-s <prefix>)" 1>&2
    usage
  fi
  OPENSSL=$default_openssl
  if [ -n "$default_openssl_pkgconfig" ]; then
    PKG_CONFIG_PATH="$default_openssl_pkgconfig"
  fi
fi
if [ -z "$PKG_CONFIG_PATH" ]; then
  PKG_CONFIG_PATH="$OPENSSL/lib/pkgconfig"
fi
export OPENSSL PKG_CONFIG_PATH

# Work around link run path not being set properly on Solaris for the
# OpenSSL libraries.
if [ "`uname`" = SunOS ]; then
  LD_RUN_PATH="$OPENSSL/lib"
  export LD_RUN_PATH
fi

apache_sbindir=`$apxs_path -q SBINDIR 2>/dev/null`
apache_bindir=`$apxs_path -q BINDIR 2>/dev/null`
apache_target=`$apxs_path -q TARGET 2>/dev/null`
apache_version=
if [ -n "$apache_sbindir" -a -n "$apache_target" ]; then
  apache_version=`$apache_sbindir/$apache_target -v | \
    awk '/^Server version: Apache\// { print substr($3, 8, length($3) - 7); }'`
fi
if [ -z "$apache_version" ]; then
  echo "Cannot determine Apache version" 1>&2
  usage
fi
case $apache_version in
1.3|1.3.*)
  apache_module=13
  apxs_opt=apxs
  with_apr=
  ;;
2.0|2.0.*)
  apache_module=20
  apxs_opt=apxs2
  with_apr="--with-apr=$apache_bindir/apr-config"
  ;;
2.2|2.2.*)
  apache_module=22
  apxs_opt=apxs22
  with_apr="--with-apr1=$apache_bindir/apr-1-config"
  ;;
*)
  echo "Unrecognized or unsupported Apache version $apache_version" 1>&2
  usage
  ;;
esac

echo "Building curl..."
(cd curl* && \
  ./configure --disable-static --without-ca-bundle --enable-thread \
    --prefix="$prefix" && \
  $make clean && \
  $make && \
  $make install) || exit 1

echo ""
echo "Building log4shib..."
(cd log4shib* && \
  ./configure --disable-static --disable-doxygen --prefix="$prefix" && \
  $make clean && \
  $make && \
  $make install) || exit 1

echo ""
echo "Building xerces-c..."
XERCESCROOT=`(cd xerces-c* && /bin/pwd) 2>/dev/null`
export XERCESCROOT
(cd $XERCESCROOT/src/xercesc && \
  ./runConfigure -p $platform -c $CC -x $CXX -r pthread -b $bits -P "$prefix" && \
  $make clean && \
  $make && \
  $make install) || exit 1

echo ""
echo "Building xml-security-c"
(cd xml-security-c* && \
  ./configure --without-xalan --prefix="$prefix" --with-xerces="$prefix" && \
  $make clean && \
  $make && \
  $make install) || exit 1

echo ""
echo "Building opensaml..."
(cd opensaml* && \
  ./configure --prefix="$prefix" --with-curl="$prefix" \
    --with-log4shib="$prefix" --with-openssl="$OPENSSL" && \
  $make clean && \
  $make && \
  $make install) || exit 1

echo ""
echo "Building shibboleth..."
(cd shibboleth* && \
  case $CXX in \
  CC) \
    CXXFLAGS='-library=Cstd,Crun' LDFLAGS=-lz \
      ./configure --with-saml="$prefix" --with-log4shib="$prefix" \
      --enable-apache-$apache_module --with-$apxs_opt=$apxs_path \
      --with-openssl="$OPENSSL" --prefix="$prefix" $with_apr \
    ;; \
  *) \
    ./configure --with-saml="$prefix" --with-log4shib="$prefix" \
      --enable-apache-$apache_module --with-$apxs_opt=$apxs_path \
      --with-openssl="$OPENSSL" --prefix="$prefix" $with_apr \
    ;; \
  esac && \
  $make clean && \
  $make && \
  $make install) || exit 1
