;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;             NOTICE OF COMPUTER PROGRAM USE RESTRICTIONS             ;;
;;                                                                     ;;
;;  The program was developed by the Navy Center for Applied           ;;
;;  Research in Artificial Intelligence.  Its distribution and         ;;
;;  use are governed by a Software Use Agreement.                      ;;
;;                                                                     ;;
;; This will certify that all authors of this software are or were     ;;
;; employees or under contract of the U.S. Government and performed    ;;
;; this work as part of their employment and that the software is      ;;
;; therefore not subject to U.S. Copyright protection.                 ;;
;;                                                                     ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package cl-user)

(eval-when (:compile-toplevel :load-toplevel :execute)
  (if (not (fboundp 'confirm-modules))
      (load "general/modules"))
  (confirm-modules "packages"
		   "config"
		   "record"
		   ))

(in-package :NLP)

(defun ipc-publish (mesg value)
  (save-action (list 'ipc-publish mesg value))
  (ipc:ipc_publishdata mesg value))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  actuator messages for the challenge
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmessage ChangeFloor (mean bearing destination)
  (flet ((format-destination ()
	    (or (and destination 
		     (unhyphenate destination)) "")))
  (cond ((eql mean 'elevator)
	 (ipc-publish MMI_Use_Elevator
		       (make-ChangeFloor 
			:bearing (coerce (or bearing 999.0) 'double-float)
			:destination (format-destination))))
	((eql mean 'stairs)
	  (ipc-publish MMI_Use_Stairs
		       (make-ChangeFloor 
			:bearing (coerce (or bearing 999.0) 'double-float)
			:destination (format-destination))))
	((eql mean 'escalator)
	 (ipc-publish MMI_Use_Escalator
		       (make-ChangeFloor 
			:bearing (coerce (or bearing 999.0) 'double-float)
			:destination (format-destination))))
	(t (UnImplemented)))))

(defmessage ChangeMode (mode)
  (ipc-publish MMI_Mode (make-Mode
				 :autonomy mode)))

(defmessage WakeUp ()
  (ipc-publish MMI_WakeUp (make-Generic)))

(defmessage ShutDown ()
  (ipc-publish MMI_Shutdown (make-Generic)))



(defmessage Greeting ()
  (let ((greeting (random-pick '("Hello." "Greetings." "Howdy." "What's up?" "Good to go."))))
    (talk-and-echo greeting)
    (ipc-publish MMI_Greeting (make-Generic))))


;;Move in the relative direction <int arg> degrees.  If <str arg>
;;is the word "obstacle" then go until you reach an obstacle,
;;otherwise treat <str arg> as a distance in inches to move.
(defmessage GoGesture (degrees distance)
  (let ((point (make-Point2DPolar
		:range (coerce (or distance 0.0) 'double-float)
		:radians (coerce (or degrees 0.0) 'double-float))))
    (ipc-publish MMI_Go_Polar (make-GoPolar :point point)))
;  (send-token (format nil "4 ~D ~D" degrees distance))
  )

(defmessage GoObstacle (direction)
  (ipc-publish MMI_Go_Obstacle (make-direction
					:dir (coerce direction 'double-float)))
;  (send-token (format nil "4 ~D obstacle" direction))
  )

;;add a lookup in the location.dat file
(defmessage FindObject (object-name)
  (ipc-publish MMI_Find (make-FindObj
			 :name (if (consp object-name) ;SCT
				   (format nil "~s" object-name)
				 (unhyphenate object-name)))))

;;Spawn the 'Follow' process to follow the object in front of
;;the robot (not currently implemented well).
(defmessage FollowMe ()
;  (ipc-publish MMI_Follow_Me nil)
  (ipc-publish MMI_Mode (make-Mode
				 :autonomy :FOLLOW_ME))
;  (send-token "5 0 0")
  )

;; not implemented
;(defmessage FaceObject (x y)
;  (send-token (format nil "7 ~D ~D" x y))
;  )

;;turn to face position where human was when last gesture was
;;observed
(defmessage FaceMe ()
  (ipc-publish MMI_Face_Me (set-timestamp))
;  (send-token "8 0 0")
  )

;(defmessage FaceHier (name)
;  (send-token (format nil "8 0 ~A" name)))

;;Move to position where robot was when last gesture was observed.
;(defmessage MovetoMe ()
;  (send-token "9 0 0"))


;(defmessage TurnToObject (object dir)
;  (format t "Turning to ~A and ~A" object dir)  )

;;Move the robot in a straight line for a distance of <int arg> inches,
;;WITHOUT collision avoidance.
;;If <str arg> is "l" or "left", the robot moves left.
;;If <str arg> is "r" or "right", the robot moves right.
;;If <str arg> is "f" or "forward", the robot moves forward.
;;If <str arg> is "b" or "back", the robot moves backward.
; direction translated to degrees
(defmessage MoveForward (distance)
  (let ((point (make-Point2DPolar :range (coerce distance 'double-float)
				  :radians 0.0d0)))
    (ipc-publish MMI_Go_Polar (make-GoPolar :point point)))
;  (send-token (format nil "13 ~D f" distance))
  )

(defmessage MoveLeft (distance)
  (let ((point (make-Point2DPolar :range (coerce distance 'double-float)
				  :radians *90-degrees*)))
    (ipc-publish  MMI_Go_Polar (make-GoPolar :point point) ))
  ;(send-token (format nil "13 ~D l" distance))
  )

(defmessage MovePolar (distance radians)
  (let ((point (make-Point2DPolar :range (coerce distance 'double-float)
				  :radians radians)))
    (ipc-publish  MMI_Go_Polar (make-GoPolar :point point) )))

(defmessage MoveRight (distance)
  (let ((point (make-Point2DPolar :range (coerce distance 'double-float)
				  :radians *-90-degrees*)))
    (ipc-publish  MMI_Go_Polar (make-GoPolar :point point) ))
;  (send-token (format nil "13 ~D r" distance))
  )

(defmessage MoveBack (distance)
  (let ((point (make-Point2DPolar :range (coerce distance 'double-float)
				  :radians PI)))
   (ipc-publish  MMI_Go_Polar (make-GoPolar :point point)))
;  (send-token (format nil "13 ~D b" distance))
  )

(defmessage TurnToObject (object direction)
  (format t "turning to ~A in the ~A direction" object direction))

;;Turn the robot by <int arg> degrees (+ CCW, - CW).
(defmessage Turn (degrees)
  (ipc-publish MMI_Turn
		       (make-Direction :dir (coerce degrees 'double-float)))
;  (send-token (format nil "15 ~D 0" degrees))
  )


;;OBSOLETE
;;Go to a specific coordinate location, BUT, coordinates are
;;of the form "A 29", or "G 2".  letter=<str arg>, number=<int arg>.
;;Mapping from these coords to x,y is done via a fixed transform.
;;(defmessage GoToCoord (x y)
;;  (send-token (format nil "18 ~A ~D" x y)))


;;Stop the robot immediately, terminate any concurrent
;;motion processes.
(defmessage Quit ()
  (ipc-publish MMI_Stop (make-Generic))
;  (send-token (format nil "19 0 0"))
  )


;;not implemented
;(defmessage GoThroughDoor (coords)
;  (send-token (format nil "20 0 0")))


;;Go to x,y location x=<int arg>, y=<str arg>
(defmessage GoToLocation (x y)
  (ipc-publish MMI_Go_XY
		       (make-Point2D :x (coerce x 'double-float)
				     :y (coerce y 'double-float) ))
			
;  (send-token (format nil "21 ~D ~D" x y))
  )

;Pick a quadrant of the map in the direction <int arg> and
;explore randomly in that quadrant.  Continue
;until timeout or interrupted.
(defmessage ExploreAreaVector (quadrant)
   (ipc-publish MMI_Go_Explore_Direction
			(make-direction :dir quadrant))
;  (send-token (format nil "22 ~D 0" quadrant))
  )

;;Go to x,y location x=<int arg>, y=<str arg>
(defmessage GoToPalmXY (x y)
    (ipc-publish MMI_Go_XY
		       (make-Point2D :x (coerce x 'double-float)
				     :y (coerce y 'double-float)))
;  (send-token (format nil "23 ~D ~D" x y))
  )

;;Follow a path defined by a set of x,y points.  The 
;;number of x,y points is in <int arg>, and the points are all in
;;<long arg> in the format "<x1> <y1> <x2> <y2> ...".  Stop as
;;soon as the end is reached.
(defmessage FollowPathPalm (numPoints points)
    (ipc-publish MMI_Go_Explore
			(make-Point2DList
			 :num numPoints
			 :points (make-point-array points)))
;  (send-token (format nil "24 ~D ~A" numPoints points))
  )

;;Randomly explore an area defined by a set of x,y points.  The 
;;number of x,y points is in <int arg>, and the points are all in
;;<long arg> in the format "<x1> <y1> <x2> <y2> ...".  Continue
;;until timeout or interrupted.
(defmessage ExploreAreaPoints (numPoints points)
   (ipc-publish MMI_Go_Explore
			(make-Point2DList
			 :num numPoints
			 :points (make-point-array points)))
;  (send-token (format nil "25 ~D ~A" numPoints points))
  )

;;Pick a quadrant of the map in the direction <int arg> and
;;follow a fixed course within in that quadrant. Continue
;;until timeout or interrupted.
(defmessage ExplorePerimVector (quadrant)
    (ipc-publish MMI_Go_Explore_Direction
			(make-direction :dir quadrant))
;  (send-token  (format nil "27 ~D 0" quadrant))
  )

(defmessage ExplorePerimPoints (numPoints points)
   (ipc-publish MMI_Go_Explore
			(make-Point2DList
			 :num numPoints
			 :points (make-point-array points)))
;  (send-token  (format nil "28 ~D ~A" num points))
  )


;;Go to waypoint number <in arg>.  Use the path planner and
;;collision avoidance to do it.
;;add a lookup to the waypoint.dat file
(defmessage GoWayPoint (pointNumber)
  (ipc-publish MMI_Go_Waypoint
		       (make-Waypoint :waypoint pointNumber))
;  (send-token (format nil "17 ~D 0" pointNumber))
  )

(defmessage DestinationReached (destination)
  (flet ((format-destination ()
			     (aif (get destination :direction)
				  (format nil "~A OF THE ~A" it (unhyphenate destination))
				  (format nil "~A" (unhyphenate destination)))))
    (ipc-publish MMI_Destination
			 (make-Destination
			  :destination-name (format-destination)))))

(defmessage Confirm ()
  (ipc-publish MMI_YesNo
		       (make-YesNo :answer :yes)))


(defmessage Deny ()
  (ipc-publish MMI_YesNo
		       (make-YesNo :answer :no)))

(defmessage Done ()
  (ipc-publish MMI_EOS (make-Generic)))

(defun isa (x)
  (and (symbolp x)
       (listp (get x 'isa))
       (car (get x 'isa))))
	      
(defun publish-is-located (theme position referent)
  (ipc-publish MMI_Is_Located (make-is-located
			       :theme (format nil "~s" theme)
			       :position (format nil "~s" position)
			       :referent (format nil "~s" referent)
			       :referent-isa 
			       (format nil "~s" (isa referent)))))









  
  
  


  


