;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;             NOTICE OF COMPUTER PROGRAM USE RESTRICTIONS             ;;
;;                                                                     ;;
;;  The program was developed by the Navy Center for Applied           ;;
;;  Research in Artificial Intelligence.  Its distribution and         ;;
;;  use are governed by a Software Use Agreement.                      ;;
;;                                                                     ;;
;; This will certify that all authors of this software are or were     ;;
;; employees or under contract of the U.S. Government and performed    ;;
;; this work as part of their employment and that the software is      ;;
;; therefore not subject to U.S. Copyright protection.                 ;;
;;                                                                     ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package cl-user)

(eval-when (:compile-toplevel :load-toplevel :execute)
  (if (not (fboundp 'confirm-modules))
      (load "general/modules"))
  (confirm-modules  "packages"
		    "config"
		   ))

(in-package :NLP)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; move
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsubcommand move-robot-without-gesture (goal distance to-loc)
  ((and (has-type goal 'goal-direction)
	(has-type to-loc 'thing))
   (talk-and-echo (format nil "Please point me in the direction of the ~A" to-loc)))
  ((floor-direction-p to-loc)
   (change-floor-without-gesture 'elevator goal to-loc NIL NIL))
  ((has-type to-loc 'thing)		; should check the position of the thing
   (FindObject to-loc)
;   (talk-and-echo (format nil "Where is the ~A?" to-loc))
   )
  ((floor-goal-p distance)
   (change-floor-without-gesture 'elevator goal to-loc NIL NIL))
  ((obj-typep to-loc 'waypoint)
   (move-to-waypoint (get to-loc :id)))
  ((and (null distance)
	(eq to-loc 'my-right))
   (GoObstacle *90-degrees*)
;   (talk-and-echo "You didn't tell me how far to move to your right.")
   )
  ((and (null distance) 
	(eq to-loc 'my-left))
   (GoObstacle *-90-degrees*)
;   (talk-and-echo "You didn't tell me how far to move to your left.")
   )
  ((and (null distance)
	(eq to-loc 'your-right))
   (GoObstacle *-90-degrees*)
;   (talk-and-echo "You didn't tell me how far right to move.")
   )
  ((and (null distance)
	(eq to-loc 'your-left))
   (GoObstacle *90-degrees*)
;   (talk-and-echo "You didn't tell me how far left to move.")
   )
  ((and (null to-loc)
	(null goal)
	(distance-goal-p distance))
   (talk-and-echo "You didn't tell me which direction to move."))
  ((and (or (has-type goal 'way)
	    (has-type to-loc 'over))
	(distance-goal-p distance))
   (multiple-value-bind (dist scale pretty-distance)
       (format-distance-goal distance)
     (declare (ignore dist scale))
     (talk-and-echo (format nil "Which way do you want me to move ~A." pretty-distance))))
  ((and (has-type goal 'back)
	(null distance))
   (GoObstacle PI)
;   (talk-and-echo "How far back do you want me to move?")
   )
  ((and (null distance)
	(has-type goal 'forward))
   ;(talk-and-echo "How far?")
   (GoObstacle 0.0d0)
   )
  ((and (physobj-p to-loc)
	(get to-loc :to-loc))		; direction to place relative to object
   (let ((point (get-point to-loc)))
     (if (null point)
	 (talk-and-echo "Where?")
       (GoToLocation (point-x point) (point-y point)))))
  ((has-type (obj-typeof to-loc) 'thing)
   (GoToLocation (get to-loc :x-coord) (get to-loc :y-coord)))
  ((and (distance-goal-p distance)
	(loc-direction-p to-loc))
   (move-robot (format-distance-goal distance) to-loc))
  ((and (distance-goal-p distance)
	(loc-direction-p goal))
   (move-robot (format-distance-goal distance) goal))
  ((null to-loc)
   (talk-and-echo "Where?"))
  ((null goal)
   (talk-and-echo "Where?"))) 


(defsubcommand move-robot-with-gesture (goal distance to-loc)
  ((or (and (right-loc-p to-loc)
	    (not (right-gesture-p)))
       (and (left-loc-p to-loc)
	    (not (left-gesture-p)))
       (and (back-p to-loc)
	    (not (back-gesture-p)))
       (and (forward-p to-loc)
	    (not (forward-gesture-p))))
     (talk-and-echo "You're saying one thing, pointing in another direction. I'm confused"))
  ((or (and (right-loc-p goal)
	    (not (right-gesture-p)))
       (and (left-loc-p goal)
	    (not (left-gesture-p)))
       (and (back-p goal)
	    (not (back-gesture-p)))
       (and (forward-p goal)
	    (not (forward-gesture-p))))
     (talk-and-echo "You're saying one thing, pointing in another direction. I'm confused"))
  ((or (has-type to-loc 'door-gesture)
       (has-type to-loc 'door))
    (GoToDoor (location-coords)))
  ((has-type to-loc 'waypoint-gesture)
        (move-to-waypoint (get-waypoint)))
  ((and (obj-typep to-loc 'waypoint) ; a waypoint was specified
	(= (get to-loc :id)
	   (get-waypoint)))
   (move-to-waypoint (get to-loc :id)))
  ((and (obj-typep to-loc 'waypoint)
	(not (= (get to-loc :id)
		(get-waypoint))))
   (talk-and-echo (format nil "Waypoint #~D is not on the ~:[left~;right~]" (get to-loc :id) (right-gesture-p))))
   ((and (physobj-p to-loc)
	(get to-loc :to-loc))		; direction some place relative to the object
   (let ((point (get-point to-loc)))
     (if (null point)
	 (talk-and-echo "Where?")
       (GoToLocation (point-x point) (point-y point)))))
  ((has-type (obj-typeof to-loc) 'thing)
   (GoToLocation (get to-loc :x-coord) (get to-loc :y-coord)))
  ((not (distance-goal-p distance))
   (GoObstacle (get-direction))
;   (talk-and-echo "How far?")
   )
  ;; should provide for other gesture
  (t (move-robot (format-distance-goal distance) to-loc)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; for go-there type of command 
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsubcommand move-robot-with-direction (to-loc)
  ((has-type to-loc 'door-gesture)
   (GoToDoor (location-coords)))
  
  (t
    (talk-and-echo "going in that direction");
;   (talk-and-echo (format nil "Going ~A" (format-gesture)))
     (GoObstacle (get-direction))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; got a distance gesture
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defsubcommand move-robot-to-distance (goal distance to-loc)
  ((and (distance-goal-p goal)
	;; arbitrary way of disambiguating gesture and speech
	;; will need to be developed based on some perception theory
	(not (close-enough-p (format-distance-goal goal)
		      (get-distance))));; check command and gesture
   (multiple-value-bind (dist scale pretty-goal)
       (format-distance-goal goal)
     (declare (ignore dist scale))
     (talk-and-echo (format nil "I'm confused. That didn't look like ~A to me."
	   pretty-goal))))
  ((or (has-type to-loc 'door-gesture)
       (obj-typep to-loc 'waypoint))
   (talk-and-echo  "Sorry, I don't understand that gesture with that command"))
  ((and (loc-direction-p to-loc)
	(not (distance-goal-p goal)));pick up the distance gesture w/o hesitation
     (move-robot (get-distance) to-loc))
  ((null to-loc)
     (talk-and-echo "You need to point to some direction.")))

(defsubcommand move-robot-with-palm-gesture (goal distance to-loc)
  ((or (has-type to-loc 'door-gesture)
       (has-type to-loc 'door))
   ;;it does not matter here whether it's a point or a range
   ;;pick the first point
   (multiple-value-bind (x y)
       (point-palm-xy) 
     (GoToDoor (point-to-location-coords x y))))
  ((or (distance-goal-p distance)
       (loc-direction-p to-loc))
     (talk-and-echo "I don't understand that gesture with that command."))
  ;; if it's not a location (door) check for point or range
  ((or (range-palm-gesture-p) (point-palm-gesture-p))
   (multiple-value-bind (x y)
	 (point-palm-xy)
       (GoToLocation x y)))
  (t 
   (talk-and-echo "I don't understand that gesture with that command.")))

(defsubcommand GoTodoor (coords)
  ((string-equal coords "NONE")
     (talk-and-echo "There is no door over there."))
  (t (talk-and-echo "Going to the door.")
     (apply #'GoToLocation (listify coords))))

(defsubcommand move-to-waypoint (waypoint)
  ((< waypoint 0)
     (talk-and-echo "There is no waypoint in that direction."))
  (t (GoWayPoint waypoint)))

  ;; we have a distance (in inches) and a location to go to
(defsubcommand move-robot (distance to-loc)
  ((not (within-range-p distance))
   (talk-and-echo "Sorry, that's beyond my range"))
  ((point-p to-loc)
   (GoToLocation (point-x to-loc) (point-y to-loc)))
  ((vague-gesture-p)
   (MovePolar distance (get-radians)))
  ((left-loc-p to-loc)
     (MoveLeft distance))
  ((right-loc-p to-loc)
     (MoveRight distance))
  ((forward-p to-loc)
     (MoveForward distance))
  ((back-p to-loc)
     (MoveBack distance))
  ;; maybe should go to another object representation
  ((has-type (obj-typeof to-loc) 'waypoint)
     (GoWayPoint (get to-loc :id))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; backup
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsubcommand backup-without-gesture (goal distance to-loc)
  ((distance-goal-p distance)
   (move-robot (format-distance-goal distance) 'BACK))
  (t (GoObstacle PI))
;  (t (talk-and-echo "How far?"))
  )


(defsubcommand backup-with-distance (goal distance to-loc)
  ;;arbitrary way of disambiguiting gesture and speech
  ((and (distance-goal-p distance)
	(not (close-enough-p (format-distance-goal distance)
			     (get-distance))))
   (multiple-value-bind (dist scale pretty-goal)
       (format-distance-goal distance)
     (declare (ignore dist scale))
     (talk-and-echo (format nil "I'm confused. That didn't look like ~A to me"
			    pretty-goal))))
  ((distance-goal-p distance) (MoveBack (format-distance-goal distance)))
  (t (MoveBack (get-distance))))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; turn
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsubcommand turn-without-gesture (distance to-loc manner)
  ((and (null distance)
	(or (has-type to-loc 'around)
	    (back-p to-loc))
	(not (has-type manner 'complete))) ; assume turning to the left
    (Turn PI))
  ((and (null distance)
	(has-type to-loc 'around)
	(has-type manner 'complete))
    (Turn (* 2 PI))) ; assume turning to the left
  ((and (null distance)
	(right-loc-p to-loc))
     (Turn *-90-degrees*))
  ((and (null distance)
	(left-loc-p to-loc))
     (Turn *90-degrees*))
  ((and (vector-goal-p distance)
	(right-loc-p to-loc))
    (Turn (- (format-distance-goal distance))))
  ((and (vector-goal-p distance)
	(left-loc-p to-loc))
    (Turn (format-distance-goal distance)))

  (t (talk-and-echo "Which way?")))

;;; we have to disambiguate here with a directional gesture
(defsubcommand turn-with-gesture (distance to-loc manner)
  ((or (and (right-loc-p to-loc)
	    (not (right-gesture-p)))
       (and (left-loc-p to-loc)
	    (not (left-gesture-p)))
       (and (or (back-p to-loc)
		(has-type to-loc 'around))
	    (not (back-gesture-p))))
   (talk-and-echo "I'm sorry. You told me to turn one way but pointed in another direction. What do you want me to do?"))
  ((forward-gesture-p)
   (talk-and-echo
    "I'm sorry. I don't understand that gesture with that command."))
  ((null distance)
   (Turn (get-radians)))
  ((left-gesture-p)
   (Turn (format-distance-goal distance)))
  ((right-gesture-p)
   (Turn (- (format-distance-goal distance)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; explore
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defsubcommand explore-with-gesture (goal to-loc relative-to)
  ((has-type to-loc 'perimeter)
     (talk-and-echo "I'm on my way.")
     (ExplorePerimVector (get-quadrant)))
  ((and (has-type to-loc 'over)
	(has-type goal 'goal-direction))
   (talk-and-echo "I'm on my way.")
   (ExplorePerimVector (get-quadrant)))
  ((or (has-type to-loc 'goal-direction)
       (has-type goal 'goal-direction))
     (talk-and-echo "I'm on my way.")
     (ExploreAreaVector (get-quadrant)))
  ((or (has-type to-loc 'region)
       (has-type to-loc 'area)
       (has-type relative-to 'area)
       (has-type relative-to 'region))
     (talk-and-echo "I'm on my way.")
     (ExploreAreaVector (get-quadrant))))

(defsubcommand explore-with-palm-gesture (goal to-loc relative-to)
  ((has-type to-loc 'perimeter)
     (multiple-value-bind (num points)
	 (range-palm-xys )
       (talk-and-echo "I'm on my way.")
       (ExplorePerimPoints num points)))
  ((and (has-type to-loc 'over)
	(has-type goal 'goal-direction))
   (multiple-value-bind (num points)
       (range-palm-xys )
     (talk-and-echo "I'm on my way.")
     (ExplorePerimPoints num points)))
  ((has-type goal 'goal-direction)
   (multiple-value-bind (num points)
       (range-palm-xys)
     (talk-and-echo "I'm on my way.")
     (ExploreAreaPoints num points)))
    ((or (has-type to-loc 'region)
	 (has-type to-loc 'area)
	 (has-type relative-to 'area)
	 (has-type relative-to 'region))
     (multiple-value-bind (num points)
	 (range-palm-xys)
       (talk-and-echo "I'm on my way.")
       (ExploreAreaPoints num points))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; follow
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsubcommand follow-with-palm-gesture ()
  (t (multiple-value-bind (num points)
	 (range-palm-xys)
       (FollowPathPalm num points))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; face
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsubcommand face-with-no-gesture (to-loc)
  ((member to-loc *others*)
     (FaceHier to-loc))
  ((eql to-loc *self*)
     (talk-and-echo "Sorry, I can't turn myself inside out.  Or maybe you have a mirror?"))
  ((left-loc-p to-loc)
     (Turn *90-degrees*))
  ((right-loc-p to-loc)
     (Turn *-90-degrees*))
  ((physobj-p to-loc)
   (FaceObject (get to-loc :x-coord) (get to-loc :y-coord)))
  ((has-type to-loc 'way)(talk-and-echo "Which way?")))

(defsubcommand face-with-gesture (to-loc)
  ((and (right-loc-p to-loc)
	(left-gesture-p))
   (talk-and-echo "You told me to face right, but pointed left.  I'm confused."))
  ((and (left-loc-p to-loc)
	(right-gesture-p))
   (talk-and-echo "You told me to face left, but pointed right. I'm confused."))
  (t (Turn (get-radians))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; challenge
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;



(defsubcommand change-floor-without-gesture (theme goal to-loc up-distance down-distance)
  ((has-type goal 'goal-direction)	;there, over there
   (if theme
       (ask-where (format nil "Sorry, which ~(~s~)?" theme)
		  (list 'change-floor-without-gesture theme goal to-loc
			up-distance down-distance))
     (ask-where "Where?"
		(list 'change-floor-without-gesture theme goal to-loc
		      up-distance down-distance))))
  ((and (null to-loc)
	(or (eql goal 'up)
	    up-distance))
;will have to be changed if keeping track of robot's state to the actual floor
   (talk-and-echo "which floor upstairs?"))
  ((and (null to-loc)
	(or (eql goal 'down)
	    down-distance))
; will have to be changed if keeping track of robot's state to the actual floor
   (talk-and-echo "which floor downstairs?"))
  ((eql to-loc 'upstairs)
   (talk-and-echo "which floor upstairs?"))
  ((eql to-loc 'downstairs)
   (talk-and-echo "which floor downstairs?"))
  ((and (null to-loc)
	(null goal)
	)			
   ;; SCT: try to add 
   ;; (null up-distance)		
   ;; (null down-distance))		
   (talk-and-echo "which floor?"))
  ((has-type to-loc 'floor)	;SCT was floor-location
   (ChangeFloor theme (format-direction (get theme :direction)) to-loc)))

(defsubcommand change-floor-with-gesture (theme goal to-loc up-distance down-distance)
  ((and (left-gesture-p )
	(right-loc-p (get theme :direction)))
   (talk-and-echo (format nil "You indicated the ~A was on the right but pointed left. I am confused." theme)))
  ((and (right-gesture-p)
	(left-loc-p (get theme :direction)))
   (talk-and-echo (format nil "You indicated the ~A was on the left but pointed right. I am confused." theme)))
  ((null to-loc)
   (talk-and-echo "Which floor?"))
  ((and (vague-gesture-p)
	to-loc)
   (ChangeFloor theme (get-radians) to-loc)))

   








