;;; Release: CMUCL_standalone_port (1.1)
;;; File: util.lisp,v
;;; File date: 2003/06/20 13:58:59 (UTC)
;;; Author: SCT

;;; General utility routines (and macros), in (mostly) alphabetical 
;;; order---macros first

(in-package cl-user)

;; TO DO
;; Rewrite PREFIX, SUFFIX to use SEQ? (Changes functionality)

;; Not really needed here:

(eval-when (:compile-toplevel :load-toplevel :execute)
  (if (not (fboundp 'confirm-modules))
      (load "modules")))

(defparameter *count-diagonally-array-size* 16000)
(defparameter *buffered-count-array-size* 2000)

;;; Macros

(defmacro and-it (&rest conds)
  "Variant of AND: in each form IT has the value of the previous form"
  `(let (it) (and ,@(mapcar #'(lambda (form) (list 'setq 'it form)) conds))))

(defmacro comp (func . more-funcs)
  "(COMP 'A 'B 'C ...) returns function for (A (B (C ...)))"
  (if (null more-funcs)
      func
    (let ((a (gensym)))
      `(function (lambda (&rest ,a)
		   (funcall ,func
			    (apply (comp ,@more-funcs) ,a)))))))

(defmacro cond-it (&rest conds)
  "Variant of COND: in each subform, binds a local variable IT to test value"
  (if conds `(if-it ,(caar conds) (progn ,@(cdar conds))
		    (cond-it ,@(cdr conds)))))

(defmacro dofile (&rest body)		;Deprecated---renamed
  `(dostream ,@body))

(defmacro doseq ((var sequence &optional r) . body)
  (let ((v (gensym)))
    `(let (,var)
       (map nil #'(lambda (,v) (setq ,var ,v) ,@body) ,sequence)
       ,r)))
       
(defmacro dostream ((var stream &optional ret) . body)
  "Macro variant of MAPS"
  `(maps #'(lambda (,var) ,@body) ,stream 
	 :on-eof #'(lambda (list last values) 
		     (declare (ignore list)
			      (ignore last)
			      (ignore values))
		     ,ret)))

(defmacro for (&rest conds)
  "Variant of COND that allows (WITH VAR TEST) in conditionals"
  ;; FOR is a bad name ...
  (multiple-value-bind
      (var test)
      (if (and (consp (caar conds))
	       (eq 'with (first (caar conds))))
	  (values (second (caar conds))
		  (third (caar conds)))
	(values	(gensym)
		(caar conds)))
    `(let ((,var ,test))
       (if ,var ,(if (cdar conds)
		     (cons 'progn (cdar conds))
		   var)
	 ,(if (cdr conds)
	      (cons 'for (cdr conds))
	    nil)))))

(defmacro if-it (test then &optional (else nil))
  "Variant of IF: inside form, binds a local variable IT to test value"
  `(let ((it ,test))
     (if it ,then ,else)))

(defmacro mvb (&rest body)
  "Short for MULTIPLE-VALUE-BIND"
  `(multiple-value-bind ,@body))

(defmacro mvl (&rest body)
  "Short for MULTIPLE-VALUE-LIST"
  `(multiple-value-list ,@body))

(defmacro mvs (&rest body)
  "Short for MULTIPLE-VALUE-SETQ"
  `(multiple-value-setq ,@body))

(defmacro or-it (&rest conds)
  "Variant of OR: in each form IT has the value of the previous form"
  `(let (it) (or ,@(mapcar #'(lambda (form) (list 'setq 'it form)) conds))))

(defmacro safe-incf (place &optional (inc 1))
  "Variant of INCF that sets PLACE to INC if not numerical"
  (let ((value (gensym)))
    `(let ((,value ,place))
       (if (numberp ,value)
	   (incf ,place ,inc)
	 (setf ,place ,inc)))))

(defmacro read-file ((var name) . body)
  `(with-open-file (,var ,name) ,@body))

; (defmacro with-file ((var name) . body)
;   (let ((time-var (gensym)))
;     `(with-open-file (,var ,name :direction :io :if-exists :overwrite) 
;        (let ((,time-var (file-write-date ,var)))
; 	 (unwind-protect (progn ,@body)
; 	   (if (> (file-write-date ,var) ,time-var)
; 	       (setf (file-length ,var) (file-position ,var)))))))) ;can't do!

(defmacro trac (&rest names)
  "An implementation of TRACE"
  `(trac-to *trace-output* ',names))

(defmacro untrac (&rest names)
  "An implementation of UNTRACE"
  `(untrac-them ,names))

(defmacro write-file ((var name) . body)
  `(with-open-file (,var ,name :direction :output :if-exists :supersede)
     ,@body))
		    
;;; Functions

(defun alist-to-hash (alist &key (test #'eql)) ;Goes with HASH-TO-ALIST
  "Returns hash table for key/value pairs ALIST (mapping with TEST)"
  (let ((table (make-hash-table :test test)))
    (dolist (pair alist table)
	    (setf (gethash (car pair) table) (cdr pair)))))

(defun combinations (list)
  "Lists all combinations from list of possibilies"
  (cond ((null (cdr list)) (mapcar 'list (car list)))
	(t (mapcan #'(lambda (tail)
		       (mapcar #'(lambda (head)
				   (cons head tail))
			       (car list)))
		   (combinations (cdr list))))))

(defun copy-file (from to)
  "Copies lines from file named FROM to stream TO"
  (mapf #'(lambda (line)
	    (write-line line to))
	from
	:reader 'read-line))
   		  
(defstruct whole-number (base 2) (digits (make-stack)))

(defun inc-vector (&optional (base 2)
			     (vector (make-stack))
			     (offset 0))
  "Treats VECTOR as digits of a number in given BASE, and increments it by 1" 
  ;; Digit K is at position K
  (cond ((= 0 (length vector))
	 (vector-push-extend 1 vector)
	 vector)
	((= (elt vector offset)
	    (1- base))
	 (setf (elt vector offset) 0)
	 (cond ((> (length vector)
		   (1+ offset))
		(inc-vector base vector (1+ offset)))
	       (t 
		(vector-push-extend 1 vector)
		vector)))
	(t (incf (elt vector offset))
	   vector)))

;; Alternately:
;; (make-whole-number :base (whole-number-base w)
;;                    :number (1+ (vector-number w)))
;; Or
;; (setf (whole-number-digits w)
;;       (whole-number-digits (number-vector (1+ (vector-number w))))) ;!

(defun inc-whole-number (whole-number)
  ;; INCF?
  (inc-vector (whole-number-base whole-number)
	      (whole-number-digits whole-number))
  whole-number)

(defun vector-number (whole-number)
  "Returns a NUMBER for (object of type) WHOLE-NUMBER"
  (let ((return 0) 
	(k 0)
	(mult 1))
    (loop
     (if (= k (length (whole-number-digits whole-number)))
	 (return return))
     (incf return (* mult (elt (whole-number-digits whole-number) k)))
     (if (= (incf k) (length (whole-number-digits whole-number)))
	 (return return)
       (setq mult (* mult (whole-number-base whole-number)))))))
  
(defun number-vector (number &optional (base 2))
  "Returns a vector for NUMBER in given BASE"
  (let ((vector (make-stack))
	div rem)
    (loop
     (mvs (div rem)
	  (truncate number base))
     (vector-push-extend rem vector)
     (if (= 0 div) (return (make-whole-number :digits vector :base base)))
     (setq number div))))

(let ((vector '#())
      (primes nil)
      (integer-array '#())
      (max-integer 0))
  (defun diagonal-primes () primes)

  (defun init-diagonal-count (&optional (init '#())
					(first-primes nil))
    (setq primes first-primes)
    (setq vector init))

  (defun buffered-count (&optional (array-size *buffered-count-array-size* ?))
    "On Kth eval, unique-valued, (roughly) buffered (+ K (RANDOM ARRAY-SIZE))"
    (if ? (setq *buffered-count-array-size* array-size))
    (cond ((not (eql array-size (length integer-array)))
	   (setq integer-array (make-array array-size :initial-element nil))
	   (setq max-integer (1- array-size))))
    (let* ((k (random array-size))
	   (return (or (elt integer-array k) k)))
      (setf (elt integer-array k)
	    (incf max-integer))
      (values return)))

  (defun count-diagonally2 ()
    (inc-array-diagonally)
    (let ((count 1))
      (do ((i 0 (1+ i))
	   (ps primes (cdr ps)))
	  ((null ps) count)
	  (setq count (* count (expt (car ps) (elt vector i)))))))
  (defun inc-array-diagonally (&optional (index (1- (length vector))))
    (cond ((< index 0)
	   (setq vector (make-array (1+ (length vector)) :initial-element 0))
	   (incf (elt vector 0))
	   (setq primes (extend-primes primes)))
	  ((= (length vector)
	      (elt vector index))
	   (setf (elt vector index) 0)
	   (inc-array-diagonally (1- index)))
	  (t
	   (incf (elt vector index))))
    vector))    

(defun all-that(predicate seq &key (key #'identity)) ;Goes with TREE-ALL-THAT
  "Lists all PREDICATE-true elements of SEQ"
  (remove-if (complement predicate) seq :key key))

(defun always (&rest rest)		;Useful for default predicates
  "Always returns non-nil"		;Better in ANSI: (constantly t)
  (or rest t))

;(defun more-of-symbol-p (stream &optional (example-constituents
;					   '(#\a #\: #\| #\\)))
;  "T if stream pointer is not end of an object according to READ"
;  ;; Implementation dependent as written :(
;  ;; Rewrite to return macro-char function if next
;  ;; Check non-terminating flag!
;  (let* ((short-peek (peek-char nil stream nil #1='#:eof))
;	 (short-position (file-position stream))
;	 (long-peek (peek-char t stream nil #1#))
;	 (long-position (file-position stream))
;	 (macro (and (characterp short-peek)
;		     (get-macro-character short-peek))))
;    (and (not (eq short-peek #1#))
;	 (not (eq long-peek #1#))
;	 (eql short-position long-position) ;=> not at whitespace
;	 (or (null macro)		;=> implementation dependent
;	     (dolist (char example-constituents nil)
;		     (if (eq macro (get-macro-character char))
;			 (return t))))
;	 short-peek)))

(defun by-pairs (func list)
  (rmap-by-pairs func list #'(lambda (x y)
			       (declare (ignore x)
					(ignore y)))))

(defun char-reader (stream char) 	;Useful for readtable funcs
  (declare (ignore stream)) char)

(defun char-symbol-reader (stream char)	;Useful for readtable funcs
  (declare (ignore stream))
  (intern (string char)))

;(defun colon-symbol-reader (stream char)
;  "Returns a non-keyword symbol  that begins with #\:'s"
;  ;; Can't be used to accept #\: in the middle or at end of symbol
;  ;; unless it begins witn #\: 
;  ;; (!) :(
;  (declare (ignore char))
;  (cond ((more-of-symbol-p stream)
;	 (let ((obj (read stream nil #1='#:eof t)))
;	   (cond ((eq obj #1#) (values))
;		 ((eq #\: (peek-char nil stream nil))
;		  (intern (concatenate 'string ":" 
;				       (symbol-name obj)
;				       (symbol-name 
;					(read stream nil nil t)))))
;		 (t				;ASSUMES SYMBOL-P !
;		  (intern (concatenate 'string ":" (symbol-name obj)))))))
;	(t (intern ":"))))
	  
(defun comment (stream format &rest args)
  "Formats output to STREAM, prepending \"; \" to each line"
  (cond ((not (stringp format))
	 (setq args (cons format args))
	 (setq format "~s")))
  (apply #'comment-with "; " stream format args))

(defun comment-with (commenter stream format &rest rest-args)
  "Format output to STREAM, prepending COMMENTER to each line"
  (let ((print-pretty t)
	(args rest-args))
    (cond-it ((member :print-pretty rest-args)
	      (setq args (append (ldiff rest-args it) (cddr it)))
	      (setq print-pretty (cadr it))))
    (let ((*print-pretty* print-pretty)
	  (stack nil))
      (with-input-from-string
       (string-stream (apply #'format nil format args))
       (loop
	(if-it (read-line string-stream nil nil)
	       (if-it (format stream "~%~a~a" commenter it)     ;Slow!
		      (push it stack))
	       (return (if stack
			   (format nil "~{~a~}" (nreverse stack))))))))))

;(defun compact-adjacents (obj seq &key (count 2) ;Or RM-REPEATS?
;			      (test #'eql)
;			      (return-type (type-of seq)))
;  "Replaces adjacent OBJs with one in SEQ"
;  ;; That is, (compact-adjacents #\# "...##...###...") -> "...#...#..."
;(coerce (mapcon #'(lambda (tail)
;		    (if (or (null (cdr tail))
;			    (not (funcall test (car tail) (cadr tail))))
;			(list (car tail))))
;		  (coerce seq 'list))	;too many conses!
;					;for vectors: use DO-SEQUENCE,
;					;PUSH-VECTOR-EXTEND, DOTIMES
;	  return-type))

(defun stringable (x)
  (or (stringp x)
      (characterp x)
      (symbolp x)))

(defun simple-general-vector (x)
  (ctypecase x 
    (simple-vector x)
    (vector (coerce x 'simple-vector))
    (symbol x (coerce (string x) 'simple-vector))
    (atom (vector x))))

(defun true-list (x)
  (cond-it ((vectorp x) (coerce x 'list))
	   ((symbolp x) (coerce (string x) 'list)) ;is this what we want?
	   ((atom x) (list x))
	   ((cdr (last x))
	    (let ((list (copy-list x)))
	      (setf (cdr (last list))
		    (list it))
	      list))
	   (t x)))
	       
(defun simple-general-vector-able (x) (atom x))

; Can't use SYMBOL as the name :(
; (defun symbol (x)
;   (cond ((symbolp x) (values x (nth-value 1 (find-symbol (string x)))))
; 	((stringp x) (intern x))
; 	((characterp x) (intern (string x)))
; 	((and (sequence-p x)		;lists too?
; 	      (every 'characterp x))
; 	 (intern (coerce x 'string)))
; 	(t (error "Can't make symbol from ~s" x))))

(defun counting-number-p (x)
  (or (and (integerp x)
	   (>= x 0))
      (and (floatp x)
	   (= 0 (nth-value 1 (truncate x))))
      (and (complexp x)
	   (= 0 (imagpart x))
	   (>= (realpart x) 0))))

(defun counting-number (x)
  (cond ((and (integerp x)
	      (>= x 0)) 
	 x)
	((and (floatp x)
	      (= 0 (nth-value 1 (truncate x))))
	 (values (truncate x)))
	((and (complexp x)
	      (= 0 (imagpart x))
	      (counting-number-p (realpart x)))
	 (counting-number (realpart x)))
	(t (error "Can't make counting number from ~s" x))))

(defun non-complex-number-p (x)
  (or (and (complexp x)
	   (= 0 (imagpart x)))
      (and (numberp x)
	   (not (complexp x)))))

(defun non-complex-number (x)
  (cond ((and (complexp x)
	      (= 0 (imagpart x)) )
	 (realpart x))
	((and (numberp x)
	      (not (complexp x)))
	 x)
	(t (error "Can't make non-complex number from ~s" x))))
  
(defun concat (&rest vectors) 
  "Variant of CONCATENATE that determines type from args"
  (cond ((every #'(lambda (x) (or (symbolp x)
				  (characterp x)))
		vectors)
	 (intern (apply 'concatenate 'string (mapcar 'string vectors))))
	((every 'stringable vectors)
	 (apply 'concatenate 'string (mapcar 'string vectors)))
	((every 'non-complex-number-p vectors)
	 (let ((n (read-from-string
		   (format nil "~{~s~}" 
			   (mapcar 'non-complex-number vectors)))))
	   (if (numberp n)
	       n
	     (coerce vectors 'simple-vector))))
	((every 'simple-general-vector-able vectors)
	 (apply 'concatenate 'simple-vector (mapcar 'simple-general-vector
						    vectors)))
	(t
	 ;; TO DO? Last list needn't be a true list ...
	 (apply 'concatenate 'list (mapcar 'true-list vectors)))))

(defun cons-ize (object)
  "Returns OBJECT in a list if not already consp"
  (if (consp object) object (list object)))

(defun duplicate (obj seq &key (count 2)
		      (test #'eql)
		      (return-type (type-of seq))) ;WARNING: type-of
  "Duplicate all instances of OBJ in SEQUENCE"
  ;; That is, (duplicate #\# "...#...#...") -> "...##...##..."
  (coerce (mapcan #'(lambda (elt)
		      (if (funcall test elt obj)
			  (make-list count :initial-element elt)
			(list elt)))
		  (coerce seq 'list))	;too many conses!
					;for vectors: use DO-SEQUENCE,
					;PUSH-VECTOR-EXTEND, DOTIMES
	  return-type))

(defun duplicates (list &key (test #'eql) (key #'identity))
  "Returns non-NIL if there are duplicates in LIST"
  ;; Generalize for sequences
  (mapl #'(lambda (tail)
	    (if-it (member (funcall key (car tail))
			   (cdr tail) :test test :key key)
		   (return-from duplicates (cons (car tail) it))))
	list)
  nil)

(defun pad-list (list length &optional padding)
  "Destructively pads LIST starting at LENGTH"
  (cond ((< length 1) list)
	((null list) (make-list length :initial-element padding))
	(t (cons (car list)
		 (pad-list (cdr list) (1- length) padding)))))

(defun parse-first-integer (string)
  "Returns first integer in STRING, else NIL"
  (let ((digit (position-if #'digit-char-p (string string))))
    (if (numberp digit)
	(parse-integer 
	 (string string) 
	 :start (if (and (> digit 0)
			 (eql #\- (elt (string string) (1- digit))))
		    (1- digit) 
		  digit)
	 :junk-allowed t))))

;(defun peek-token-termination (stream)
;  "Non-NULL if stream pointer is at end of token"
;  ;; Returns :EOF, whitespace, or terminating macro-character-function
;  (let* ((short-peek (peek-char nil stream nil nil))
;	 (long-peek (peek-char t stream nil nil))) ;ERROR: This hangs ...
;    (cond ((null short-peek) :eof)
;	  ((not (eql short-peek long-peek)) short-peek) ; at whitespace
;	  (t (mvb (macro non-terminating-p)
;		  (get-macro-character short-peek)
;		  (if non-terminating-p  
;		      nil
;		    macro))))))

(defun equalp-symbol (x y)		;Deprecated--renamed
  (symbol-equalp x y))

(defun extend-primes (primes)
  "Pushes the next prime onto (the list of) PRIMES"
  ;; Largest prime first
  (let ((n (1+ (or (car primes) 1))))
    (loop				;(loops too much ...)
     (dolist (p primes (return-from extend-primes (cons n primes)))
       (if (zerop (mod n p))		;is MOD best?
	   (return)))
     (incf n))))

(defun shuffle (seq)
  "Returns copy of SEQ with elements pseudo-randomly shuffled"
  ;; Inefficient! too many nthcdrs
  (do* ((stack nil)
	(i (length seq) (1- i))
	(list (coerce seq 'list))
	(r )
	(cdr ))
       ((null list) (coerce stack (type-of seq))) ;Warning: is type-of robust?
       (cond ((zerop (setq r (random i)))
	      (push (pop list) stack))
	     (t (setq cdr (nthcdr (1- r) list))
		(push (cadr cdr) stack)
		(setf (cdr cdr)
		      (cddr cdr))))))
  
(defun symbol-equalp (x y)
  "T iff X and Y are EQUALP, or SYMBOL-NAME's are"
  (or (equalp x y)
      (and (symbolp x)
	   (symbolp y)
	   (equalp (string x)
		   (string y)))))

(defun make-stack (&optional (type t) (size 8))
  "Returns adjustable fill-ptr'ed array of (optionally given) TYPE and SIZE"
  (make-array size :fill-pointer 0 :element-type type :adjustable t))

(let ((stack (make-stack 'character))
      macro non-terminating-p)
  (defun faster-simple-token-reader (stream char) ;a readtable func
    "A simple token reader ignoring READTABLE-CASE, etc"
    ;; Warning: special characters like #\\ #\| #\# #\, and #\: must be
    ;; assigned this function if they are to appear in a token
    ;; Warning 2: Assumes (reasonably?) that whitespace
    ;; has no macro function (Actually, a terminating macro is okay)
    ;; Cf. SIMPLE-TOKEN-READER, which doesn't have this problem
    (setf (fill-pointer stack) 0)
    (vector-push-extend char stack)
    (loop
     (setq char (peek-char nil stream nil nil))
     (if (null char) (return))	;eof
     (mvs (macro non-terminating-p)
	  (get-macro-character char))
     (cond (macro
	    (if non-terminating-p
		(vector-push-extend (read-char stream) stack)
	      (return)))
	   ((read-from-string (string char) nil nil)
	    (vector-push-extend (read-char stream) stack))
	   (t (return))))		;whitespace
    (coerce stack 'simple-string)))

(defun faster-simple-symbol-reader (stream char) ;a readtable func
  (intern (faster-simple-token-reader stream char)))

(defun flatten (tree)
  "Flattens TREE into a simple list"
  (if (atom (cdr tree))
      (list (car tree) (cdr tree))
    (mapcan #'(lambda (elt)
		(if (atom elt) (list elt)
		  (flatten elt)))
	    tree)))

(defun free-plist-to-alist (list segmenter)
  "Breaks LIST into an `alist' on elements SEGMENTER returns non-nil on"
  ;; If CAR of LIST isn't a key, first element of alist will be
  ;; missing a key.  The first of adjacent keys, or a final-position
  ;; key, will have null alist values.
  (let ((alist nil) tail)
    (loop
     (if (null list)
	 (return (nreverse alist)))
     (setq tail (member-if segmenter (cdr list)))
     (push (ldiff list tail) alist)
     (setq list tail))))
     
(defun freely-delimited-list-reader (close-chars) ;Deprecated---renamed
  (read-freely-delimited-list close-chars))

(defun hash-to-alist (hashtable)
  "Returns alist of key/value pairs from HASHTABLE"
  (let ((list nil))
    (maphash #'(lambda (key value)
		 (push (cons key value) list))
	     hashtable)
    list))

(defun hash-member-if (func table)
  "Returns first key-value values in TABLE for which FUNC returns non-nil"
  (maphash #'(lambda (key value)
	       (if (funcall func key value)
		   (return-from hash-member-if (values key value))))
	   table))

(defun top ()
  (let (return (return? nil))
    (loop
     (loop
      (unwind-protect
	  (loop
	   (print '>)
	   (setq return (read))
	   (cond ((equal return '(quit))
		  (setq return? t)
		  (return-from top (values))))
	   (format t "~{~&~s~}" (multiple-value-list (eval return))))
	(if (not return?) (return)))))))
   
(let ((tables (make-hash-table :test 'eq)))
  (defun hash-table-pair (table)
    "Returns one arbitrary key/value values pair of TABLE"
    ;; Returns different pair on different invocations. Slow.
    (cond ((null (gethash table tables))
	   (setf (gethash table tables) 0)))
    (let ((count (incf (gethash table tables))))
      (maphash #'(lambda (key value)
		   (if (zerop (decf count))
		       (return-from hash-table-pair (values key value))))
	       table))))

(defun intersective (x y &rest keys)
  "Look for an intersection without really building it"
  (dolist (elt x nil)
	  (if-it (apply #'member elt y keys)
		 (return it))))

(defun left-union (list1 list2 &key (test 'eql)  (key 'identity))
  "Like UNION, but guaranteed to use item from first arg"
  (let ((return (copy-list list1)))
    (dolist (elt list2)
      (if (not (member (funcall key elt) return
		       :key key
		       :test test))
	  (push elt return)))
    return))

(defun safe-length (seq)
  "Like LENGTH but returns 0 if not a sequence"
  (if (sequence-p seq) (length seq) 0))

(defun sequence-p (x)
  (or (vectorp x)
      (listp x)))

(defun list-closer (end-char)
  "Returns a readtable function for READ-DELIMITED-LIST"
  #'(lambda (stream start-char)
      (declare (ignore start-char))
      (read-delimited-list end-char stream t)))

(defun list-ize (object)
  "Returns OBJECT in a list if not already LISTP"
  (if (listp object) object (list object)))

;; See above for MAKE-STACK

(let ((stack (make-stack 'character))
      (this-macro))
  (defun get-clique-symbol-reader ()
    (setq this-macro
      #'(lambda (stream char)		;readtable func
	  "INTERN's a token, w/out looking for package specs, number reps, etc"
	  ;; Warning: 
	  ;; Ignores READTABLE-CASE too
	  ;; chars not assigned CLIQUE-TOKEN-READER will terminate the token
	  ;; Also: is function-eql ansi defined?
    
	  ;; Possible Variant: Check for GRAPHIC-CHAR-P and not #\space
	  ;; (Would allow some characters to continue token, but not start
	  (setf (fill-pointer stack) 0)
	  (vector-push-extend char stack)
	  (loop
	    (setq char (peek-char nil stream nil nil))
	    (if (characterp char)
		(mvb 
		 (macro non-terminating-p)
		 (get-macro-character char)
		 (if (and non-terminating-p (eq macro this-macro))
		     (vector-push-extend (read-char stream) stack)
		   (return (intern stack))))
	      (return (intern stack))))))))

(defun mapf (&rest rest)		;Deprecated---renamed
  (apply #'maps rest))

(defun mapfile (&rest rest)		;Deprecated---renamed
  (apply #'mapstream rest))

(defun maps (func stream &rest keys)
  "Applies FUNC to STREAM's objects, returning NIL"
  (apply #'mapstream #'(lambda (&rest rest)
			 (apply func rest) nil) 
	 stream keys))

(defun mapstream (func stream[s] &rest keys
		       &key (if-does-not-exist :error)
		       &allow-other-keys)
  "Applies FUNC to STREAM's (or file's) objects, as if in MAPCAN"
  (let ((streams (list-ize stream[s]))
	(new-streams nil))
    (unwind-protect
	(apply #'map-open-stream func
	       (mapcar #'(lambda (stream?)
			   (cond ((streamp stream?) 
				  (cond ((open-stream-p stream?) stream?)
					(t (car (push (open stream?)
						      new-streams)))))
				 ((or (equal stream? t)
				      (equal stream? nil)) stream?)
				 (t (car (push 
					  (or (open stream?
						    :if-does-not-exist
						    if-does-not-exist)
					      (make-string-input-stream ""))
					  new-streams)))))
		       streams)
	       keys)
      (dolist (s new-streams)
	      (close s)))))

#|      
  (cond ((streamp stream)
	 (cond ((open-stream-p stream)
		(apply #'map-open-stream func stream keys))
	       (t (with-open-stream (s stream)
				    (apply #'map-open-stream func s keys)))))
	((or (eq stream t)
	     (eq stream nil))
	 (apply #'map-open-stream func stream keys))
	(t (with-open-file (s stream :if-does-not-exist if-does-not-exist)
			   (apply #'map-open-stream func s keys)))))
|#

(defun mapstream-with-comments (&rest rest)
  (apply #'mapfile-with-comments rest))

(defun mapfile-with-comments (func stream &rest rest &key 
				   (comment-char #\;)
				   (reversed-lines nil)
				   &allow-other-keys)
  "Variant of MAPSTREAM that passes a list of comment lines to FUNC"
  ;; FUNC : (comment-list, object) -> ?
  (let ((lines nil))
    (apply #'mapstream #'(lambda (object)
			   (funcall func (if reversed-lines lines
					   (nreverse lines))
				    object)
			   (setq lines nil))
	   stream
	   :reader #'(lambda (stream eof-p eof-val)
		       (loop
			(if (eql comment-char (peek-char t stream nil))
			    (push (read-line stream) lines)
			  (return (read stream eof-p eof-val)))))
	   rest)))

(defun map-open-stream (func stream[s] &key 
			     (case :upcase ?case) (reader #'read)
			     (readtables nil)
			     (eof-value 'whocares ?eof-value)
			     (on-eof #'(lambda (list last values) 
					 (declare (ignore last))
					 (if values (format t
	      ";; WARNING: Unequal input lengths to MAP-OPEN-STREAM~%"))
					 list))
			     &allow-other-keys)
  "Applies FUNC to (open) STREAM's objects, as if in MAPCAN"
  ;; FUNC may return non-LISTP objects. These are ignored.
  (let* ((*readtable* (copy-readtable))
	 (streams (list-ize stream[s]))
	 (list (list nil))
	 (last list)
	 (number-of-streams (length streams))
	 eofs elt fvalue values)
    (if ?case (setf (readtable-case *readtable*) case))
    (loop
     (setq values nil)
     (setq eofs 0)
     (mapc #'(lambda (stream table?)
	       (let ((*readtable* (if (readtablep table?)
				      table?
				    *readtable*)))
		 (setq elt (funcall reader stream nil #1='#:eof))
		 (cond ((eq elt #1#)
			(cond (?eof-value
			       (if (= (incf eofs) number-of-streams)
				   (setq ?eof-value nil
					 values nil)
				 (push eof-value values))))
			(if (null ?eof-value)
			    (return-from 
			     map-open-stream
			     (funcall on-eof (cdr list) last values))))
		       (t (push elt values)))))
	   streams (or readtables streams))
     (setq fvalue (apply func (nreverse values)))
     (cond ((consp fvalue)		;Add to list if CONSP
	    (setf (cdr last) fvalue)
	    (pop last))))))

(defun median (list &optional (order '<) 
			      (typebreaker #'(lambda (x y)
					       (read-from-string ;hm...
						(format nil "~$" 
							(/ (+ x y) 2.0)))))
			      (key 'identity))
  "Returns median value of LIST by ORDER"
  (setq list (sort (copy-list list)
		   order :key key))
  (let* ((length (length list))
	 (trunc (truncate length 2))
	 (*readtable* (copy-readtable nil)))
    (cond ((oddp length)
	   (nth trunc list))
	  (t 
	   (funcall typebreaker 
		    (nth trunc list)
		    (nth (1- trunc) list))))))
  
(defun never (&rest rest)		;Useful for default predicates
  "Always returns NIL"			;Better in ANSI: (constantly nil)
  (declare (ignore rest)) nil)

(defun num-archy (str)
  "Lists parts of a `number' like 1.2.1.A as (\"1\" \"2\" \"1\" \"A\")"
  (let ((*readtable* (copy-readtable)))
    (set-syntax-from-char #\. #\space)
    (read-list-from-string (cond ((symbolp str) (string str))
				 ((stringp str) str)
				 (t (format nil "~a" str)))
			   #'read-raw)))

(defun num-archy< (h1 h2)
  "Used to induce an order on number hierarchies (See NUM-ARCHY)"
  (let ((depth -1))
    (mapl #'(lambda (x1 x2)
	      (incf depth)
	      (let ((n1 (read-from-string (car x1)))
		    (n2 (read-from-string (car x2))))
		(cond ((and (numberp n1)
			    (numberp n2))
		       (cond ((< n1 n2)
			      (return-from num-archy< depth))
			     ((> n1 n2)
			      (return-from num-archy< nil))))
		      ((string< (car x1) (car x2))
		       (return-from num-archy< depth))
		      ((string> (car x1) (car x2))
		       (return-from num-archy< nil))
		      ((and (cdr x2) (null (cdr x1))) ; 1.0 < 1.0.0
		       (return-from num-archy< depth)))))
	  (if (listp h1)
	      h1
	    (num-archy h1))
	  (if (listp h2)
	      h2
	    (num-archy h2)))
    nil))
		  
(defun pair-list (list)
  "On (a b c d ...) returns ((a b) (c d) ...)"
  (nreverse (rmap-by-pairs #'list list)))

(defun plist-equiv (list1 list2 &key (test #'eql))
  "T if PLIST1 and PLIST2 have the same keys and the same values per key"
  (set-equiv (rmap-by-pairs #'cons list1)
	     (rmap-by-pairs #'cons list2)
	     :test #'(lambda (pair1 pair2)
		       (and (eq (car pair1) (car pair2))
			    (funcall test (cdr pair1) (cdr pair2))))))

(defun plist-to-alist (list)
  (pair-list list))

(defun pprint-list (list &optional (stream t))
    (format stream "~{~%~a~}" list))

(defun pprint-plist (plist &optional (stream t))
  (format stream "~{~%~a ~a~}" plist))

(defun prefix (string &optional (delimiters "+"))
  "Returns prefix of STRING up to character in DELIMITERS, else NIL"
  (if (not (stringp delimiters))
      (setq delimiters (format nil "~a" delimiters)))
  (if-it (position-if #'(lambda (c)
			  (find c delimiters))
		      (string string))
	 (subseq (string string) 0 it)))

(defun prefix-else-all (string &optional (delimiters "+"))
  "Returns (PREFIX STRING) else STRING"
  (or (prefix string delimiters)
      string))

(defun proper-subsetp (x y &rest rest)
  "T iff X is a proper subsetp of Y"
  (and (apply #'subsetp x y rest )
       (not (apply #'subsetp y x rest))))

(defun rcar (whatever &optional (func #'car))
  "Gets CAR, or CAAR, or CAAAR, or ..."
  (cond ((atom (car whatever)) (funcall func whatever))
	(t (rcar (car whatever) func))))

(defun readfile (stream)		;Deprecated---renamed
  (readstream stream))

(defun read-freely-delimited-list (end-chars &optional stream
					     (eof-error-p t)
					     recursive-p
					     (reader #'read))
  "Variant of READ-DELIMITED-LIST: takes a list of chars, opt accepts eof"
  ;; Returns, as 2nd value, the char matched (NIL if eof)
  ;; Takes an arbitrary READER
  (let ((list nil) obj)
    (loop
     (cond ((null (setq obj (peek-char t stream eof-error-p nil
				       recursive-p)))
	    (return (values (nreverse list) nil)))
	   ((member obj end-chars)
	    (return (values (nreverse list) (read-char stream))))
	   (t (setq obj (funcall reader stream nil ;eof-error-p ??
				 #1='#:eof recursive-p))
	      (if (not (eq obj #1#))
		  (push obj list)
		(return (values (nreverse list) nil))))))))

;;(defun read-freely-delimited-stuff (end-chars &optional stream
;;					     (eof-error-p t)
;;					     recursive-p
;;					     (reader #'read))
;                                            (filter #'nreverse)
;;                                           (struct-conc nil)
;;                                           (slot-concs nil))

;;(defun read-freely-delimited-struct (struct-conc slot-concs
;;                                     &optional stream
;;					     (eof-error-p t)
;;					     recursive-p
;;					     (reader #'read))
;;                                      
;; (read-freely-delimited-stuff nil stream eof-error-p recursive-p reader
;;                                         #'identity struct-conc slot-cons)

;;(defun read-struct-from-string (string ;?reader
;;                                struct-conc &rest slot-concs)
;; (read-freely-delimited-struct 
;;   struct-conc slot-concs stream(make-string-input-stream string))

(defun read-list-from-string (string &optional (reader #'read))
  "Returns objects read from STRING in a list"
  (read-freely-delimited-list nil (make-string-input-stream string)
			      nil nil reader))

;(defun read-no-intern (&optional (stream nil)
;			   (eof-error t) 
;			   (eof-value nil)
;			   (recursive nil)
;			   (typical-white #\newline)
;			   (typical-constituent #\z))
;  "Reads without interning strings or parsing numbers"
;  ;; How can this be implementation independant without TYPICAL-WHITE, etc?
;  ;; (See READ-RAW hack with PEEK-CHAR)
;  ;; As is, assumes (mvb (func term)
;  ;;                   (get-macro-function from-char)
;  ;;                   (set-macro-function to-char func term)
;  ;;                   ...)
;  ;; has the effect of (set-syntax-from-char from-char to-char)
;(let (white-func white-terminal const-func const-terminal
;		 ch func terminal)
;  (mvs (white-func white-terminal)
;       (get-macro-function typical-white))
;  (mvs (white-func white-terminal)
;       (get-macro-function typical-constituent))
;  (peek-char t stream nil nil recursive) ;skip whitespace
;  (loop 
;   (setq ch (read-char stream nil #1="eof" recursive))
;   (cond ((eq ch #1#)  ))
;   (mvs (func terminal)
;	 (get-macro-function ch))
;   (cond ((or (and (eq func white-func)
;                   (eq terminal white-terminal))
;              (and func terminal))) ; end of token
;	  ((and (eq func const-func)
;	        (eq terminal const-terminal)))
;	 )))

(defun read-raw (&optional (stream nil)
			   (eof-error t) 
			   (eof-value nil)
			   (recursive nil))
  "Ignores all readtable attributes except whitespace, returning a string"
  ;; String is adjustable, has fill pointer. Apply COPY-SEQ to make it simple.
  ;; BUG! Waits for 2nd token or EOF
  (let (ch1 ch2 (token (make-stack 'character)))
    (peek-char t stream nil nil recursive) ;skip whitespace
    (loop
     (setq ch1 (peek-char nil stream nil token recursive)) ;next char
     (setq ch2 (peek-char t stream nil token recursive)) ;skips whitespace
     (cond ((eq ch2 token)
	    (return (cond ((> (length token) 0) token)
			  (eof-error (read stream t))
			  (t eof-value))))
	   ((not (eql ch1 ch2))		;end of token
	    (return token))
	   (t (vector-push-extend 
	       (read-char stream nil #\! recursive) token))))))

(defun readstream (stream)
  "Returns a list of objs in STREAM"
  (mapstream #'(lambda (elt &optional stream)
		 (declare (ignore stream))
		 (list elt))
	     stream))

(defun remove-adjacents (list &key (test #'eql))
  "Remove adjacents elements satisfying :TEST"
  ;; Change to allow any sequence
  (mapcon #'(lambda (tail)
	      (if (or (null (cdr tail))
		      (not (funcall test (car tail) (cadr tail))))
		  (list (car tail))))
	  list))

(defun rlast (tree)
  "Recursively gets LAST of LAST of ..., until CAR is ATOM"
  (let ((last (last tree)))
    (cond ((atom (car last)) last)
	  (t (rlast (car last))))))

(defun rmap-by-pairs (function list &optional (func2 #'cons))
  "Applies FUNCTION to LIST members, two at a time"
  ;; Odd-length lists are NOT in error; last elt is NIL
  (let ((result nil))
    (loop
     (if (null list) 
	 (return result))
     (setq result (funcall func2 (funcall function (pop list) (pop list)) 
			   result)))))

;Alternately, use:
;(&optional (state '#1=(t nil . #1#)))
;  (mapcon #'(lambda (tail)
;	      (if (pop state)
;		  (list (cons (car tail) (cadr tail)))))
;  	  list))

(defun safe> (a b)
  "Non-nil if A is number and (A > B or B is NIL)"
  (cond ((null b) (realp a))
	((null a) nil)
	((realp a)
	 (cond ((realp b) (> a b))
	       (t nil)))
	(t nil)))


(defun safe< (a b)
  "Non-nil if A is number and (A < B or A is NIL)"
  (cond ((null a) (realp b))
	((null b) nil)
	((realp b)
	 (cond ((realp a) (< a b))
	       (t nil)))
	(t nil)))
		
(defun safe-car (object)
  "Return CAR of OBJECT, if possible, else NIL"
  ;; See also SAFE-RCAR
  (if (consp object) (car object)))

(defun safe-coerce (item type)
  (coerce 
   (cond ((and (numberp item)
	       (= 0 (nth-value 1 (truncate item)))
	       (>= item 0)
	       (subtypep type 'bit-vector))
	  ;; There should be a better (concise) way ...
	  (read-from-string (format nil "#*~b" item)))
	 ((and (bit-vector-p item)
	       (subtypep type 'number))
	  (let ((string (format nil "~s" item)))
	    (setf (elt string 1) #\b)
	    (read-from-string string)))
	 (t item))
   type))
	
;(defun coerce-forcibly (item type)
;  "Coerces (many types of) ITEM to given (sequence) TYPE"
;  (coerce (cond ((listp item) item)
;		((vectorp item) item)
;		((symbolp item) (string item))
;		((characterp item) (string item))
;		((streamp item) (readstream item)) ;???
;		;;((arrayp item)		;multidimensional array
;		;;; Make a (vector t) of arrays
;		;;)
;		;;((a structure?) Make a list of slots?)
;		(t (list item)))
;	  type))

(defun copy (obj)
  ;; COPY-SEQ (defined) ...
  (cond ((consp obj) (copy-tree obj))	;or copy-list?
	((subtypep (type-of obj) 'structure-object) (copy-structure obj))
					;is STRUCTURE-OBJECT ansi?
	((symbolp obj) (copy-symbol obj))
	((readtablep obj) (copy-readtable obj))
	(t obj)))		

(defun safe-vref (sequence index)
  "Like AREF, but for vectors, and returns NIL past end of sequence"
  (cond ((>= index (array-total-size sequence)) nil)
	(t (aref sequence index))))

(defun set-vref-extend (sequence index value)
  ;; WARNING: sequence must be ADJUSTABLE
  ;; TO DO: generalize to multidimensional arrays
  (cond ((>= index (array-total-size sequence))
	 (adjust-array sequence (max (1+ index)
				     (+ 128 (array-total-size sequence))))))
  (setf (aref sequence index) value))

(defsetf safe-vref set-vref-extend)
	
(defun safe-first (seq)
  "Like FIRST, but okay with sequences, and NIL otherwise"
  (if (and (vectorp seq) 
	   (> (length seq) 0))
      (elt seq 0)
    (safe-car seq)))

(defun safe-map (type func array) ;; TO DO: &rest r)
  "Like MAP, but applies to arrays of rank > 1 too"
  (cond ((and (arrayp array)
	      (> (array-rank array) 1)
	      (subtypep (list 'array t (array-dimensions array)) type))
	 ;; Bug: should extract element type from TYPE
	 (let ((new-array (make-array (array-dimensions array)))
	       (size (array-total-size array)))
	   (dotimes (k size new-array)
		    (setf (row-major-aref new-array k)
			  (funcall func (row-major-aref array k))))))
	(t (funcall 'map type array))))

(defun safe-mapcan (func &rest lists)
  "Like MAPCAN, but it goes until the end of the longest of the LISTS"
  (apply #'nconc (apply #'safe-mapcar func lists)))

(defun safe-mapcar (func &rest lists)
  "Like MAPCAR, but it goes until the end of the longest of the LISTS"
  (let ((result nil)
	args
	tails)
    (setq lists (reverse lists))
    (loop
     (if (not (member-if (complement #'null) lists))
	 (return (nreverse result)))
     (setq tails nil)
     (setq args nil)
     (dolist (list lists)
	     (push (cdr list) tails)
	     (push (car list) args))
     (push (apply func args)
	   result)
     (setq lists (nreverse tails)))))
		      
(defun safe-max (a &rest r &key (key #'safe-second))
  (cond ((hash-table-p a)
	 (let ((max-value nil)
	       (max-key nil))
	   (maphash #'(lambda (hash-key value)
			(let ((test-value (funcall key hash-key value)))
			  (cond ((safe> test-value max-value)
				 (setq max-value test-value)
				 (setq max-key hash-key)))))
		    a)
	   max-key))
	(t (apply #'max (cons a r)))))

(defun safe-min (x &rest r)
  "Non-REAL-p objects = +Inf"
  (if (realp x)
      (min x (apply #'safe-min r))
    (apply #'safe-min r)))
			
(defun safe-open (path &rest etc)
  "Returns (VALUES NIL error-condition) if OPEN on PATH ETC fails"
  (handler-case (apply #'open path etc)
		(file-error (condition) (values nil condition))))
		 
(defun safe-rcar (whatever &optional (func #'car))
  "Gets CAR, or CAAR, or CAAAR, or ..."
  (cond ((atom whatever) whatever)
	((atom (car whatever)) (funcall func whatever))
	(t (rcar (car whatever) func))))

(defun safe-rlast (tree)
  "Recursively gets LAST of LAST of ..., until CAR is ATOM"
  (if (atom tree) tree
    (let ((last (last tree)))
      (cond ((atom (car last)) last)
	    (t (safe-rlast (car last)))))))

(defun safe-second (&rest rest)
  (cond ((cdr rest) (second rest))
	((consp (car rest)) (second (car rest)))
	(t nil)))

(defun safe-subseq (seq start &optional (end (length seq)))
  ;; Don't run on lists!
  (if (> start (length seq))
      (setq start (length seq)))
  (if (> end (length seq))
      (setq end (length seq)))
  (subseq seq start end))

(defun safe-string (obj)
  "Returns a string for OBJ"
  (cond ((stringp obj) obj)
	((symbolp obj) (symbol-name obj))
	((characterp obj) (string obj))
	((listp obj) (format nil "~{~a~}" obj))	;why not just ~s?
	((vectorp obj) (format nil "~{~a~}" 
			       (coerce obj 'list)))
	(t (format nil "~s" obj))))

(defun safe-butlast (list)
  "Like BUTLAST, but for dotted lists; returns LAST as second value"
  (mvb (reverse last)
       (safe-reversed-butlast list nil)
       (values (nreverse reverse) last)))

(defun safe-reversed-butlast (list reverse)
  "Like SAFE-BUTLAST, but returns list in reverse order"
  (cond ((atom (cdr list)) (values reverse list))
	(t (safe-reversed-butlast (cdr list) (cons (car list) reverse)))))
	 
(defun safe-tree-all-that (predicate tree)
  "Lists all PREDICATE-true elements of TREE (Dotted LAST's are ignored)"
  (cond ((funcall predicate tree)
	 (list tree))
	((consp tree)
	 (mapcan #'(lambda (elt)
		     (tree-all-that predicate elt))
		 (mvb (list last)
		      (safe-butlast tree)
		      (nconc list (list (car last))))))))

(defun safe-zerop (obj)
  "Returns T iff OBJ is 0"
  (and (numberp obj) (zerop obj)))

(defun sdiff (seq1 seq2 &key (test #'eql))
  "Variant of LDIFF for general sequences"
  (if-it (search seq2 seq1 :from-end t :test test)
	 (subseq seq1 0 it) 
	 seq1))

(defun seq (seq start &optional end)
  "Like SUBSEQ, but shares structure unless SEQ is listp and END is non-null"
  (cond ((arrayp seq) (make-array (- (or end (array-total-size seq)) start)
				  :element-type (array-element-type seq)
				  :displaced-to seq
				  :displaced-index-offset start))
	(end (subseq start end))
	(t   (nthcdr start seq))))

(defun set-equiv (x y &rest keys)	;Deprecated--renamed
  (apply #'set-eql x y keys))

(defun set-eql (x y &rest keys)
  "Returns T if X and Y represent the same set, else NIL"
  (and (apply #'subsetp x y keys)
       (apply #'subsetp y x keys)))

(defun set-equal (x y &rest keys)
  "Returns T if X and Y are EQUAL sets, else NIL"
  (and (apply #'subsetp x y :test #'equal keys)
       (apply #'subsetp y x :test #'equal keys)))

(defun sift-tree (tree &key (or :or))
  "Sift OR nodes up to top of tree"
  ;; An empty OR node is not a node at all!
  ;;   (sift-tree '(a b (:or) c)) => (a b c)
  (cons or (cdr (sift-tree-really tree or (gensym) #'cdr))))

(defun sift-tree-really (tree or or2 test)	
  ;; Helper for SIFT-TREE. OR2 only appears at top level.
  ;; If OR2 appears, OR is not allowed to
  (cond ((atom tree) (list or2 tree))	;     atom  -> (or2 atom)
	((eq (car tree) or2) tree)	; (or2 . l) -> (or2 . l)
	((eq (car tree) or)		;  (or . l) -> (or2 . l')
	 (cons or2 (mapcan test
			   (mapcar #'(lambda (elt)
				       (sift-tree-really elt or or2 test))
				   (funcall test tree)))))
	(t				;        l  -> (or2 l' . ls)
	 (let ((new-trees (list nil)))
	   (dolist (elt tree (cons or2 (mapcar #'reverse new-trees)))
		   (setq elt (funcall test (sift-tree-really elt or or2 test)))
		   (setq new-trees
			 (mapcan #'(lambda (tree)
				     (cond ((null elt) (list tree))
					   (t (mapcar #'(lambda (tree2)
							  (cons tree2 tree))
						      elt))))
				 new-trees)))))))

(defun simple-token-reader (stream char) ;a readtable func
  "A simple token reader ignoring READTABLE-CASE, etc"
  ;; Warning: special characters like #\\ #\| #\# #\, and #\: must be
  ;; assigned this function if they are to appear in a token
  (let ((stack (make-stack 'character)))
    (vector-push-extend char stack)
    ;; Alternate implementation would assume (reasonably?) that whitespace
    ;; has no macro function (Actually, a terminating macro would be okay)
    ;; Could spare the call to READ-FROM-STRING on non-terminating macros
    (loop
     (setq char (peek-char nil stream nil nil))
     (if (null char) (return))	;eof
     (cond ((terminating-char-p char) (return))
	   ((read-from-string (string char) nil nil)
	    (vector-push-extend (read-char stream) stack))
	   (t (return))))		;whitespace
    stack))				;adjust to make simple?

(defun split-on (elt list &key (test #'eql))
  "Returns lists of ELT-separated sublists of LIST"
  ;; Note that (SPLIT-ON 'FOO NIL) -> (NIL)
  (if-it (member elt list :test test)
	 (cons (ldiff list it)
	       (split-on elt (cdr it) :test test))
	 (list list)))

(defun stack-less (n stack)
  "Returns Nth value from top of STACK and T, else NIL and NIL"
  (setq n (- (length stack) n))
  (if (> n 0)
      (values (elt stack (1- n)) t)
    (values nil nil)))

(defun stack-top (stack)
  "Returns top value of STACK and T, else NIL and NIL"
  (if (> (length stack) 0)
      (values (elt stack (1- (length stack))) t)
    (values nil nil)))
	    
(defun suffix (string &optional (delimiters "+") &key (from-end t))
  "Returns suffix of STRING after character in DELIMITERS, else NIL"
  (if-it (position-if #'(lambda (c)
			  (find c delimiters))
		      (string string) :from-end from-end)
	 (subseq (string string) (1+ it))))

(defun suffix-else-all (string &optional (delimiters "+"))
  "Returns (SUFFIX STRING) else STRING"
  (or (suffix string delimiters)
      string))

(defun sum-duplicates (list &key (test 'eql))
  "Returns list of uniq elements (x . k) with k the number of x in LIST"
  (let ((table (make-hash-table :test test)))
    (dolist (elt list)
      (safe-incf (gethash elt table)))
    (hash-to-alist table)))

(defun tail-match (seq tail &key (test #'eql))
  "Returns T if SEQ ends in TAIL, else NIL"
  (eql (- (length seq) (length tail))
       (search tail seq :test test :from-end t)))

(defun take (k seq)
  "Returns prefix of SEQ of length K"
  (cond ((listp seq)
	 (let ((tail (nthcdr k seq)))	;2 traversals
	   (ldiff seq tail)))
	((vectorp seq) (subseq seq 0 k))
	(t (copy-seq		
	    (make-array k :displaced-to seq
			  :element-type (array-element-type seq))))))

(defun terminating-char-p (char)
  "Non-null iff CHAR is a terminating macro char"
  (mvb (macro non-terminating-p)
       (get-macro-character char)
     (and macro (not non-terminating-p))))

(defun type-element-type (type)
  (cond ((atom type)
	 (cond ((member type '(array vector sequence list cons)) '*)
	       ((member type '(simple-vector simple-array)) t)
	       ((subtypep type 'string) 'character) ;base-character?
	       (t nil)))
	(t (cond ((member (car type) '(array vector))
		  (second type))
		 (t nil)))))

(let ((aliases (make-hash-table :test 'eq))
      (depths (make-hash-table :test 'eq)))
  (defun trac-to (out names)
    "Underlines TRAC, an implementation of TRACE"
    (if (null names)
	(mapcar 'car (hash-to-alist aliases))
      (dolist (name names names)
	(let ((alias (gensym)))
	  (setf (symbol-function alias)
		(symbol-function name))
	  (setf (gethash name aliases) alias)
	  (setf (symbol-function name)
		#'(lambda (&rest r)
		    (comment out "~s> ~s ~s" (safe-incf (gethash name depths))
			     name r)
		    (unwind-protect
			(let ((values (mvl (apply alias r))))
			  (comment out "~s< ~s ~s" (gethash name depths)
				   name values))
		      (decf (gethash name depths)))))))))
  (defun untrac-them (names)
    "Underlies UNTRAC"
    (dolist (name names)
	    (setf (symbol-function name)
		  (symbol-function (gethash name aliases)))
	    (remhash name aliases))
    (if (null names)
	(maphash #'(lambda (name alias)
		     (setf (symbol-function name)
			   (symbol-function alias))
		     (remhash name aliases))
		 aliases))))
		 
(defun tree-all-that (predicate tree)
  "Lists all PREDICATE-true elements of TREE"
  (cond ((funcall predicate tree)
	 (list tree))
	((consp tree)
	 (mapcan #'(lambda (elt)
		     (tree-all-that predicate elt))
		 tree))))

(defun tree-apply-if (filter test tree &key
			     (key #'identity)
			     (new-values nil)
			     (old-values nil))
  "Like SUBST-IF but applies FILTER to old value to get new"
  ;; Returns new tree, new values, old values
  (values
   (mapcar #'(lambda (elt)		;maplist in 1.8--why?
	       (cond ((funcall test (funcall key elt))
		      (push elt old-values)
		      (car (push (funcall filter elt) new-values)))
		     ((consp elt)
		      (mvs (elt new-values old-values)
			   (tree-apply-if filter test elt :key key
					  :new-values new-values
					  :old-values old-values)))
		     (t elt)))
	   tree)
   new-values old-values))

(defun tree-r-if (test tree)
  "Like TREE-REMOVE-IF but returns TREE"
  (mapc #'(lambda (elt)
	    (cond ((not (funcall test elt))
		   (cond ((atom elt))
			 (t (tree-r-if test elt))))))
	tree))

(defun tree-count-if (func tree &optional (count 0))
  (mapc #'(lambda (elt)
	    (cond ((funcall func elt)
		   (incf count))
		  ((consp elt)
		   (setf count (tree-count-if func elt count)))))
	tree)
  count)

(defun tree-count (elt tree &key (test #'eql))
  (tree-count-if #'(lambda (x) (funcall test x elt)) tree))

(defun tree-member (elt tree &key (test #'eql) (key #'identity))
  "Returns first tail of TREE which matches (or whose car matches) ELT"
  (tree-member-if #'(lambda (x)
		      (funcall test elt (funcall key x)))
		  tree))

(defun tree-member-if (func tree)
  "Recursively applies FUNC to each tail (!) of TREE"
  (mapl #'(lambda (tail)
	    (if (funcall func (car tail))
		(return-from tree-member-if tail))
	    (if (consp (car tail))
		(if-it (tree-member-if func (car tail))
		       (return-from tree-member-if it))))
	tree)
  nil)

(defun tree-remove (item tree)
  "Removes ITEM(s) from TREE"
  (tree-remove-if #'(lambda (x) (eql x item)) tree))

(defun tree-remove-if (test tree)
  "Removes item(s) according to TEST from TREE"
  (mapcan #'(lambda (elt)
	      (cond ((not (funcall test elt))
		     (cond ((atom elt) (list elt))
			   (t (list (tree-remove-if test elt)))))))
	  tree))

(let ((queue nil))
  (defun pop-comments () 
    (prog1 (nreverse queue)
      (setq queue nil)))
  (defun uniq-comment (comment &rest rest)
    (if-it (apply #'uniq-comment-on-queue queue comment rest)
	   (push it queue))))

(defun uniq-comment-on-queue (queue comment &rest rest)
  "Returns a COMMENT not on *QUEUE*, prints to T"
  (setq comment (apply #'comment nil comment rest))
  (if (not (member comment queue :test #'equal))
      (princ comment t)))

(defun uniq-cons (elt table)
  "Used to get EQUAL(P) ELT from TABLE, and store new element if necessary"
  (or (gethash elt table)
      (cond ((consp elt)
	     (setf (gethash elt table)
		   (cons (uniq-cons (car elt) table)
			 (uniq-cons (cdr elt) table))))
	    (t (setf (gethash elt table) elt))))) ;or just return ELT

(defun void-reader (stream char)	;useful for readtable funcs
  "Maps STREAM and CHAR to nothing"
  (declare (ignore stream)
	   (ignore char))
  (values))
