;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;             NOTICE OF COMPUTER PROGRAM USE RESTRICTIONS             ;;
;;                                                                     ;;
;;  The program was developed by the Navy Center for Applied           ;;
;;  Research in Artificial Intelligence.  Its distribution and         ;;
;;  use are governed by a Software Use Agreement.                      ;;
;;                                                                     ;;
;; This will certify that all authors of this software are or were     ;;
;; employees or under contract of the U.S. Government and performed    ;;
;; this work as part of their employment and that the software is      ;;
;; therefore not subject to U.S. Copyright protection.                 ;;
;;                                                                     ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(in-package :NLP)

(defvar *non-gesture-verbs* '(p-show p-exit find-goal p-face-action robot-action p-jump come-direction p-say p-describe p-thank p-stop p-listen p-see p-isa p-wake p-sleep pay-attention p-sleep p-excuse p-pay p-continue p-dummy-vp be-relative-to))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; newrobotresult format (from get_gesture)
;;; could be of length 5 or more
;;; 
;;; gesture type:  G_NONE, G_ERROR, G_DIRECTION, G_DISTANCE, G_PALMGEST
;;; "g-value" returns the actual gesture value, if it makes sense 
;;;           (direction, distance, or number of palm points)
;;; "timestamp" will return an integer representation of the time the 
;;;           gesture was sensed (not implemented yet?)
;;; "g-error" returns a more detailed indication of the type of error:
;;;           G_NO_ERROR, G_NOT_UNDERSTOOD, G_SEE_ONE_HAND, G_SEE_TWO_VECTORS
;;; "palminfo" returns the palm gesture information (XY pairs, etc).
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun gesture-pred-p (pred)
  (not (member pred *non-gesture-verbs*)))

(defun no-gesture-p ()
  (declare (special newrobotresult))
  (= (first newrobotresult) 0))

(defun bad-gesture-p ()
 (declare (special newrobotresult))
  (= (first newrobotresult) 1))

(defun vague-gesture-p ()
 (declare (special newrobotresult))
  (= (first newrobotresult) 2))

(defun distance-gesture-p ()
 (declare (special newrobotresult))
  (= (first newrobotresult) 3))

(defun palm-gesture-p ()
 (declare (special newrobotresult))
  (= (first newrobotresult) 4))

(defun point-palm-gesture-p ()
  (declare (special newrobotresult))
  (and (palm-gesture-p)
       (= (second newrobotresult) 1)))

(defun range-palm-gesture-p ()
  (declare (special newrobotresult))
  (and (palm-gesture-p)
       (> (second newrobotresult) 1)))

(defun gesture-verb-p (verb)
 (declare (special newrobotresult))
  (not (member verb *non-gesture-verbs*)))

;; converted to radians
(defun right-gesture-p ()
 (declare (special newrobotresult))
  (and (vague-gesture-p)
       (< (* -3/4 PI) (second newrobotresult) (* -1/4 PI))))

(defun left-gesture-p ()
 (declare (special newrobotresult))
   (and (vague-gesture-p)
	(< (* 1/4 PI) (second newrobotresult) (* 3/4 PI))))

(defun back-gesture-p ()
  (declare (special newrobotresult))
  (and (vague-gesture-p)
       (or (< (* 3/4 PI) (second newrobotresult) PI)
	   (< (* -3/4 PI) (second newrobotresult) -PI))))

(defun forward-gesture-p ()
  (declare (special newrobotresult))
  (and (vague-gesture-p)
       (< (* -1/4 PI) (second newrobotresult) (* 1/4 PI))))

(defun format-gesture ()
  (declare (special newrobotresult))
  (when (vague-gesture-p)
    (cond ((> PI (second newrobotresult) 0)
	   "left")
	  ((< (second newrobotresult) 0)
	   "right")
	  ((= (second newrobotresult) PI)
	   "back")
	  ((= (second newrobotresult) 0)
	   "forward"))))


(defun get-radians ()
 (declare (special newrobotresult))
  (when (vague-gesture-p)
    (second newrobotresult)))

(defun get-quadrant ()
  (get-radians))

;; returns an x,y location
(defun location-coords ()
 (declare (special newrobotresult))
  (when (vague-gesture-p)
       (get-location-coords (second newrobotresult))))

(defun get-direction ()
  (declare (special newrobotresult))
  (when (vague-gesture-p)
    (second newrobotresult)))

(defun get-waypoint ()
 (declare (special newrobotresult))
  (when (vague-gesture-p)
       (which-waypoint (second newrobotresult))))

(defun get-distance ()
 (declare (special newrobotresult))
  (when (distance-gesture-p)
        (second newrobotresult)))

(defun point-palm-xy ()
 (declare (special newrobotresult))
  (when (palm-gesture-p)
    (values (fifth newrobotresult)
	    (sixth newrobotresult))))

(defun range-palm-xys ()
  (declare (special newrobotresult))
  (when (palm-gesture-p)
    (values (second newrobotresult) ; number of points
	    (stringify (cddddr newrobotresult)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; errors
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun misunderstood-gesture-p ()
  (declare (special newrobotresult))
  (and (bad-gesture-p)
       (= (fourth newrobotresult) 1)))

(defun one-hand-gesture-p ()
  (declare (special newrobotresult))
  (and (bad-gesture-p)
       (= (fourth newrobotresult) 2)))

(defun two-vectors-gesture-p ()
  (declare (special newrobotresult))
  (and (bad-gesture-p)
       (= (fourth newrobotresult) 3)))

(defun gesture-timestamp (gesture)
  (third gesture))

;;Bill's explanation as to why right/left are +-180 degrees:
;;we want the robot to just turn left and hopefully see the sign.  The vision will
;;report it when/if it does see it.  I'd rather have the robot turn a bit 
;;further and be sure to cover (with the camera's field of view) what the 
;;human thinks of as 'left'. Then, if it doesn't see the sign, it's not because 
;;it stopped at 90 instead of 110, from where it would have seen it.
;;We might tweak this value later after the cameras are on and running,
;;but for now we'll just be thorough.

(defun translate-gesture ()
  (declare (special newrobotresult))
  (cond ((right-gesture-p)
	 -1.6)
	((left-gesture-p)
	  1.6)
	((back-gesture-p)
	 PI)
	((forward-gesture-p)
	 0.0)
	(t (warn "Unknown gesture direction ~A " newrobotresult))))





		     
