/******************************************************************************
 * ipc gesture simulation
 ****************************************************************/
#include <time.h>
#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "ipc.h"
/*********************************************************
 * IPC message for simulated Biclops 3-D vector gesture. *
 *********************************************************/

#include "commonInterface.h"   /* "Timestamp" definition */

typedef struct {
  Meters x;
  Meters y;
  Meters z;
} point3d;

#define point3d_FORMAT_NAME "point3d_FORMAT"
#define point3d_FORMAT "{Meters,Meters,Meters}"


typedef struct {   /* 3-D gesture input from Biclops */
  point3d origin;
  point3d direction;  
  Timestamp timestamp;   /* timestamp for gesture's receipt */
} BVS_Sim_3DVector_Gesture_Msg;


#define BVS_Sim_3DVector_Gesture_FORMAT_NAME "BVS_Sim_3DVector_Gesture_FORMAT"
#define BVS_Sim_3DVector_Gesture_FORMAT "{point3d_FORMAT,point3d_FORMAT,Timestamp_FORMAT}"

	/* The IPC message name for 3D vector gesture */
#define BVS_Send_Sim_3DVector_Gesture_Msg "SendSim3DVector"

#define MODULE_NAME "GestureSim"

/* hardcoded examples */
double  origins [4][3] = {{-0.1, -0.2, 2.1} , {-0.2, 0.8, 1.5}, {0.8, 0.5, 0.3}, {0.7, -1.3, 0.7}};

double  destinations [4][3] = {{-0.2, -0.8, 0.1} , {-0.5, 2.1, 0.7}, {0.9, 0.4, 1.2}, {1.2, -0.4, 0.8}};

int numOptions = 4;

static void showOptions() {
  int n;
 
  printf ("         origin                  destination \n");
  for (n=0;n<numOptions;++n) {
    printf ("%d. %.2f %.2f %.2f    %.2f %.2f %.2f\n",n,origins[n][0],origins[n][1],origins[n][2],destinations[n][0],destinations[n][1], destinations[n][2]);
  }
  printf ("enter a gesture number or q to exit\n");
  fflush(stdout);
}

static void stdinHnd (int fd, void *clientData)
{

  char inputLine[81];
  char line[81];
  time_t the_time;
  int gesture;
  BVS_Sim_3DVector_Gesture_Msg simulated_gesture;

  fgets(inputLine, 80, stdin);

  switch (inputLine[0]) {
  case 'q': case 'Q': 
    IPC_disconnect();
    exit(0);
  case 'm': case 'M':
    { 
      printf ("sorry, nothing for now\n");
      fflush(stdout);
      break;
    }
  case 'i':
    {
      showOptions();
      fgets(line,80,stdin);

      if (strncmp(line,"q",1) == 0 || strncmp(line,"Q",1) == 0) {
	IPC_disconnect();
	exit(0);
      }
      gesture = atoi(line);
      simulated_gesture.origin.x = origins[gesture][0];
      simulated_gesture.origin.y = origins[gesture][1];
      simulated_gesture.origin.z = origins[gesture][2];
      simulated_gesture.direction.x = destinations[gesture][0];
      simulated_gesture.direction.y = destinations[gesture][1];
      simulated_gesture.direction.z = destinations[gesture][2];
      simulated_gesture.timestamp = set_timestamp();
      printf ("Publishing gesture %d\n",gesture);
      IPC_publishData(BVS_Send_Sim_3DVector_Gesture_Msg,&simulated_gesture);
      printf("\nType 'i' for menu of gestures; Type 'q' to quit\n");
      break;
    default: 
      printf("stdinHnd [%s]: Received %s", (char *)clientData, inputLine);
      fflush(stdout);
    }
  }
}


static void handlerChangeHnd (const char *msgName, int num, void *clientData)
{
  fprintf(stderr, "HANDLER CHANGE: %s: %d\n", msgName, num);
}


int main (void)

{
  /* Connect to the central server */
  printf("\nIPC_connect(%s)\n", MODULE_NAME);
  IPC_connect(MODULE_NAME);

  /* Define the messages that this module publishes */
  IPC_defineFormat("Meters","double");
  printf ("after meters\n");
  fflush(stdout);
  IPC_defineFormat(Timestamp_FORMAT_NAME,Timestamp_FORMAT);
  printf ("after timestamp \n");
  fflush (stdout);
  IPC_defineFormat(point3d_FORMAT_NAME,point3d_FORMAT);
  printf ("after format\n");
  fflush(stdout);
  IPC_defineFormat(BVS_Sim_3DVector_Gesture_FORMAT_NAME,BVS_Sim_3DVector_Gesture_FORMAT);
  printf ("after gesture\n");
  fflush (stdout);
  IPC_defineMsg (BVS_Send_Sim_3DVector_Gesture_Msg,IPC_VARIABLE_LENGTH,
                 BVS_Sim_3DVector_Gesture_FORMAT);
  printf ("after message \n");
  fflush (stdout);

  IPC_subscribeHandlerChange(BVS_Send_Sim_3DVector_Gesture_Msg, handlerChangeHnd, NULL);

  /* Subscribe a handler for tty input.
     Typing "q" will quit the program; Typing "m" will send MSG1;
     Typing "r" will send QUERY1 ("r" for response) */
  printf("\nIPC_subscribeFD(%d, stdinHnd, %s)\n", fileno(stdin),MODULE_NAME );
  IPC_subscribeFD(fileno(stdin), stdinHnd, (void *) MODULE_NAME);

  printf("\nType 'i' for menu of gestures; Type 'q' to quit\n");

  IPC_dispatch();

}
