;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;             NOTICE OF COMPUTER PROGRAM USE RESTRICTIONS             ;;
;;                                                                     ;;
;;  The program was developed by the Navy Center for Applied           ;;
;;  Research in Artificial Intelligence.  Its distribution and         ;;
;;  use are governed by a Software Use Agreement.                      ;;
;;                                                                     ;;
;; This will certify that all authors of this software are or were     ;;
;; employees or under contract of the U.S. Government and performed    ;;
;; this work as part of their employment and that the software is      ;;
;; therefore not subject to U.S. Copyright protection.                 ;;
;;                                                                     ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(in-package  :NLP)


(defvar *robot-distance-range* 240 "How far we want the robot to navigate")

(defconstant *degrees-to-radians-conversion* (/ PI 180.0))

(defconstant *radians-to-degrees-conversion* (/ 180.0 PI))

;;(defvar *wakup-commands* '(p-pay p-sleep p-wake p-listen p-dummy-vp))
(defvar *wakup-commands* '(p-pay p-start p-wake p-listen p-dummy-vp))

(defvar *non-continuable-actions* '(p-stop p-dummy-vp p-continue))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;            general utilities
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun mappend (fn thelist)
  (apply #'append (mapcar fn thelist)))

(defun find-all (fn thelist)
  (delete nil (mapcar #'(lambda (elem)
			  (when (funcall fn elem)
			    elem)) thelist)))

(defun append1 (lst obj)
   (append lst (list obj)))

(defun random-pick (thelist)
  (nth (random (length thelist)) thelist))

(defun empty-string-p (str)
  (string= str ""))

(defun listify (str)
  (let ((ll nil))
    (with-input-from-string (s  str)
       (do ((token (read s nil 'eof) (read s nil 'eof)))
	   ((eql token 'eof))
         (setf ll (append1 ll token))))
    ll))

(defun restringify (listified-string)
  (if (null listified-string) (format nil "")
    (concatenate 'string (format nil "~A " (first listified-string))
		 (restringify (rest listified-string)))))


(defun string-elt (str index &optional (separator #\,))
  (let ((splitstr (split-string str separator)))
    (elt splitstr index)))


(defun split-string (string &optional (separator #\,))
  "Split STRING at occurences of SEPARATOR, returning a list of
substrings."
  (let ((loc (position separator string :test #'char=)))
    (if loc
	(cons (subseq string 0 loc) 
	      (split-string (subseq string (1+ loc)) separator))
      (list string))))

;;substitute a string into a larger string
(defun splice (new old str)
  (let ((pos (search old str :test #'string-equal)))
    (if pos
      (splice new old (concatenate 'string (subseq str 0 pos)
		   new (subseq str (+ pos (length old)))))
      str)))


(defun make-keyword (symb)
  (intern (symbol-name symb) 'keyword))

(defun parse-float (string)
  (let ((str (read-from-string string)))
    (if (floatp str)
      str
      (error "Invalid float ~A" string))))

(defun unhyphenate (string)
  (substitute #\space #\- (format nil "~A" string) :test #'char=))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;onlisp macros (P. Graham)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro when-bind ((var expr) &body body)
  `(let ((,var ,expr))
     (when ,var
       ,@body)))

(defmacro aif (test-form then-form &optional else-form)
  `(let ((it ,test-form))
     (if it ,then-form ,else-form)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;            queues P. Graham
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defun make-queue ()
  (cons nil nil))

(defun enqueue (obj q)
  (if (null (car q))
      (setf (cdr q) (setf (car q) (list obj)))
    (setf (cdr (cdr q)) (list obj)
	  (cdr q) (cdr (cdr q))))
  (car q))

(defun dequeue (q)
  (pop (car q)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;             rule utilities
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun trace-all (fns)
  (eval `(trace ,@(remove-if-not #'fboundp fns))))

(defun trace-nlp ()
  "Traces nlp functions"
  (let ((nlp-package (find-package :NLP)))
    (do-all-symbols (x nlp-package)
      (when (and (eq (symbol-package x) nlp-package)
                 (fboundp x))
	(eval `(trace ,x))))))

;;; order the rules by specificity
;;; the more variables, the more specific
;;; structure of the rules are ((antecedent consequent) ...)
(defun parse-rules (args rules)
  (stable-sort  rules #'> :key #'(lambda (rule)
			     (count-vars args (car rule)))))

;;; antecedents are of varying depth
(defun count-vars (vars antecedent)
  (let ((mod-antecedent (remove-duplicates (flatten-list antecedent) :test #'eq)))
    (reduce #'+ (mapcar #'(lambda (var)
			    (count var mod-antecedent)) (flatten-list vars)))))

(defun merge-keywords (args1 args2)
  (let ((keywords nil))
    (do ((kwrd (append args1 args2) (cddr kwrd)))
	((null kwrd))
      (push (car kwrd) keywords))
    (mappend #'(lambda (keyword)
		(list keyword (or (getf args1 keyword)
				  (getf args2 keyword))))
	    (remove-duplicates keywords))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;             robot utilities
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Now defined in robotalk.lisp
;; (defun talk-and-echo (speakbuf)
;;   ;;detect when operating in stleath mode 
;;   (unless (null speakbuf)
;;    (format t "~%~A will say: ~A~%" *robot-name* speakbuf)
;;    (and *evaluate*
;;   	(talk (string-trim '(#\[ #\]) speakbuf)))))

;; check-robot-name looks for the name of the robot.
;; if it agrees or if the direct address is to "robots"
;; then the test succeeds and the real defcommand occurs below

(defun check-robot-name ()
  (declare (special *robot-name* #|*you*|# *self*))
  ;; (format t "~A = robot-name; ~A = *you*; ~A = *self*~%"'
  ;;         *robot-name* *you* *self*)
  (or (eq *robot-name* (last-named-addressee))
      (member *self* (last-named-addressee))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Tinsel has the ability to send the command to other agents
;;; when appropriate but in the robots' context  we have to
;;; avoid executing the command
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun attentive-p (command)
  (or *attention-var*
      (member command *wakup-commands* :test #'eql)))


(defun applicable-p (agent)
  (eql agent *robot-name*))

(defun robots-lisp-save ()
  ;; should use something like dumplisp
 #+ign (disksave (format nil "~A/bin/robots-lisp" *robots-home*)
            :restart-function #'boot-robots :full-gc t))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; format-interim-goal first takes the input string containing goal information,
;; e.g. "4 INCH" ("4 INCHES") and transforms the goal into a list for further
;; translation functions.  It next calculates the goal to inch measurements
;; since all of the linear dimensions in the C code are in inches.  And ultimately
;; the message sent back to the robot contains only inch information.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; converted to return meters
(defun format-distance-goal (goal)
  (destructuring-bind (distance &optional scale)
      (read-from-string (format nil "(~A)" goal))
    (case scale
      (inch (values (/ distance 39.37) 'meter (format nil "~D inch~:P" distance)))
      (degree (values (* distance *degrees-to-radians-conversion*) 'radian (format nil "~D degree~:P" distance)))
      (foot (values (* distance 0.3048) 'meter
		    (format nil "~D ~:[foot~;feet~]" distance (> distance 1))))
      (meter (values distance scale (format nil "~D meter~:P" distance)))

      (kilometer (values (* distance 1000) 'meter
			 (format nil "~D kilometer~:P" distance)))
      (centimeter (values (/ distance 100) 'meter
		         (format nil "~D centimeter~:P" distance)))
      (mile (values (* distance 1609.35) 'meter
		    (format nil "~D mile~:P" distance))))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; needs some fuzzification because what's a big difference or
;;; a small difference depends on the magnitude itself and not
;;; just the relative magnitude. For example, 10% is small for
;;; small quantities like 1,2, etc but big for large quantities
;;; like 100, 200, etc.  There is also a perception issue. 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun close-enough-p (distance other-distance)
  (<= -2 (- distance other-distance) +2))

(defun within-range-p (distance)
  (<= distance *robot-distance-range*))

(defun coordinates (coords)
  (destructuring-bind (x y)
      (read-from-string (format nil "(~A)" coords))
    (values x y)))

(defun distance-goal-p (goal)
  (destructuring-bind (distance &optional scale)
      (read-from-string (format nil "(~A)" goal))
    (declare (ignore distance))
    (member scale '(INCH FOOT METER KILOMETER CENTIMETER MILE))))

(defun floor-goal-p (goal)
  (destructuring-bind (distance &optional scale)
      (read-from-string (format nil "(~A)" goal))
    (declare (ignore distance))
    (member scale '(FLOOR FLIGHT LEVEL))))

(defun vector-goal-p (goal)
  (destructuring-bind (distance &optional scale)
      (read-from-string (format nil "(~A)" goal))
    (declare (ignore distance))
    (eql scale 'DEGREE)))

(defun format-direction (direction)
  ;; in radians
  (case direction
    (your-left 1.6)
    (my-left -1.6)
    (your-right -1.6)
    (my-right 1.6)
    (:behind PI)
    (:in-front-of 0)
    (:near (* 2 PI))
    (:opposite 999.0)
    (otherwise 999.0)))

    

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; location predicates
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun home-p (loc)
  (or (eql loc 'home)
      (eql loc 'waypoint-0)))

(defun right-loc-p (loc)
  (cond ((symbolp loc)
	 (member loc '(RIGHT YOUR-RIGHT MY-LEFT)))
	((physobj-p loc)
	 (member :RIGHT (get loc :direction) ))))


(defun left-loc-p (loc)
  (cond ((symbolp loc)
	 (member loc '(LEFT YOUR-LEFT MY-RIGHT)))
	((physobj-p loc)
	 (member :LEFT (get loc :direction)))))

(defun forward-p (loc)
  (cond ((symbolp loc)
	 (member loc '(FORWARD UP IN-FRONT-OF)))
	((physobj-p loc)
	 (member :FORWARD (get loc :direction)))))

(defun back-p (loc)
  (cond ((symbolp loc)
	 (member loc '(BACK AWAY BEHIND)))
	((physobj-p loc)
	 (member :back (get loc :direction)))))

;;a more precise definition
;;does not include way and over which are too vague
(defun loc-direction-p (loc)
  (member loc '(RIGHT YOUR-RIGHT MY-LEFT LEFT YOUR-LEFT MY-RIGHT FORWARD UP BACK AWAY AROUND OPPOSITE OVER)))


(defun floor-direction-p (loc)
  (or (member loc '(DOWN UP UPSTAIRS DOWNSTAIRS))
      (has-type loc 'floor))) ; SCT was -location)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;        KR/NAUTILUS utililities
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun question-p (lf)
  (or (my-assoc 'ASKWH lf)
      (my-assoc 'REQUEST lf)))

(defun negation-p (lf)
  (my-assoc 'NOT lf))

(defun isa-request-p (lf)
  (eql (get-class (or (my-assoc 'REQUEST lf)
		      (my-assoc 'PRESENT lf))) 'P-ISA))

(defun is-exit-request-p (lf)
  (eql (get-class (my-assoc 'imper lf)) 'p-exit))

(defun location-request-p (lf)
   (eql (get-class (or (my-assoc 'REQUEST lf)
		      (my-assoc 'PRESENT lf))) 'P-LOCATION))

(defun whereis-request-p (lf)
   (eql (get-class (or (my-assoc 'REQUEST lf)
		      ;;"where am I" question
		      (my-assoc 'present (my-assoc 'ASKWH lf))))
       'BE-AT-POSITION))

(defun ami-request-p (lf)
  (eql (get-class (or (my-assoc 'REQUEST lf)
		      ;;"where am I" question
		      (my-assoc 'present (my-assoc 'ASKWH lf))))
       'BE-RELATIVE-TO))

(defun assertion-p (lf)
  (my-assoc 'PRESENT lf))

(defun command-type-p (lf)
  "Check if the parse is a command"
  (or (my-assoc 'IMPER lf)
      ;;first command in a compound command
      (my-assoc 'IMPER (my-assoc 'and lf))))

(defun frame-value (symb slot)
  (getf (get symb 'slot-specs) slot))

(defun instance-p (symb)
  (find symb *objlist* :test #'eql))

(defun obj-label (symb)
  (get symb :label))

(defun obj-dir (symb)
  (get symb :direction))

(defun obj-id (symb)
  (get symb :id))

(defun obj-descr (symb)
  (let ((description (substitute "" "-" (split-string (get symb :description) #\newline) :test #'string-equal)))
    (concatenate 'string (first description) " " (second description) ". "
		 (fourth description))))

(defun obj-typep (symb type)
  (when (symbolp symb)
    (eql (get symb 'type) type)))

(defun physobj-p (symb)
  ;;a physical object has a type of object and is not in referents.lisp
  (when (symbolp symb)
    (and (has-type (get symb 'type) 'object)
	 (not (member symb *objlist*)))))

(defun description-p (symb)
  (when (symbolp symb)
    (has-type (get symb 'type) 'description)))

(defun obj-typeof (symb)
  (when (symbolp symb)
    (get symb 'type)))

(defun obj-short-descr (symb)
  (first (split-string (get symb :description) #\newline)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  History utilities
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun is-continuable-p (context)
  (and (command-type-p (context-lfs context))
       (not (member (context-command context)
		    *non-continuable-actions*))))


(defun bind-end-of-sequence ()
  (declare (special *previous-context*))
  (setf *previous-context*
	(find-if #'(lambda (context)
		     (and (isa-request-p (context-lfs context))
			  (eql (getf (context-args context) :cotheme)
			       'end-of-sequence)))
		 *history*)))

(defun bind-exit-command ()
   (declare (special *previous-context*))
  (setf *previous-context*
	(find-if #'(lambda (context)
		     (is-exit-request-p (context-lfs context)))
		 *history*)))

(defun bind-previous-continuable-action ()
  "Find the last continuable action in the history stack"
  (declare (special *previous-context*))
  (setf *previous-context*
	 (find-if #'(lambda (context)
		      (and (is-continuable-p context)
			   (context-success context))) *history*)))

(defun bind-unparsed-request ()
  (declare (special *previous-context* ))
  (setf *previous-context*
	(find-if #'(lambda (context)
		     ;; should check that it was my request
		     (null (context-lfs context))) *history*)))

(defun bind-uncertain-vv-parse ()
  (declare (special *previous-request*))
  (setf *previous-context*
	(find-if #'(lambda (context)
		     (context-vv-uncertainty context)) *history*)))

(defun bind-previous-askwh ()
  (declare (special *previous-context*))
  (setf *previous-context*
	(find-if #'(lambda (history)
		     ;;should put a check on who made the request
		     ;;we assume for now that the robot made the request
		     (whereis-request-p (context-lfs history)))
		 *history*)))


(defun bind-previous-isa-request ()
  (declare (special *previous-context*))
  (setf *previous-context*
	(find-if #'(lambda (history)
		     ;;should put a check on who made the request
		     ;;we assume for now that the robot made the request
		     (isa-request-p (context-lfs history)))
		 *history*)))

(defun bind-previous-location-request ()
  (declare (special *previous-context*))
  (setf *previous-context*
	(find-if #'(lambda (history)
		     ;;should put a check on who made the request
		     ;;we assume for now that the robot made the request
		     (location-request-p (context-lfs history)))
		 *history*)))

(defun bind-previous-ami-request ()
  (declare (special *previous-context*))
  (setf *previous-context*
	(find-if #'(lambda (history)
		     ;;should put a check on who made the request
		     ;;we assume for now that the robot made the request
		     (ami-request-p (context-lfs history)))
		 *history*)))
       
(defun unsuccessful-command-p (context)
  (and (command-type-p (context-lfs context))
       ;;make sure the command is addressed to me
       (eql *self* (getf (context-args context) :agent))
       (not (context-success context))))

(defun unsuccessful-commands ()
  (find-all  #'unsuccessful-command-p *history*))

(defun bind-unsuccessful-command ()
  "bind the last unsuccessful request to *previous-context*"
  (declare (special *previous-context*))
  (setf *previous-context*
	(find-if #'unsuccessful-command-p *history*)))

(defun clear-history (context)
  (setf *history* (remove context *history* :test #'eq)))

(defun clear-unsuccessful-requests ()
    (mapc #'(lambda (request)
	      (setf *history* (remove request *history* :test #'eq)))
	  (unsuccessful-commands)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; testing utilities
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun discard-p (line)
  (or (empty-string-p line)
      (char= (char line 0) #\;)))

(defun prompt-for-continuation (inputstream)
    (if (y-or-n-p "Continue?")
      (read-line inputstream nil 'EOF)
    'EOF))

(defun test-sentences (filename &optional (prompt t))
  (with-open-file (s filename :direction :input)
      (do ((line (read-line s nil 'EOF) (if prompt
					    (prompt-for-continuation  s)
					  (read-line s nil 'EOF))))
	  ((eql line 'eof))
	(unless (discard-p line)
	  (format t line)
		      (interrob-process-sentence line)))))











    
