// abstract.h - Header file with declaration of abstract mechanical components

/* abstract components should have information like location and orientation
   as rarely as possible: it is defined by relations */

// Lego-like cross axis which can be used to connect to a gear
// Fixed in location and orientation-- with respect to something else
// which could be a hinge, etc...-- variable in rotation in the right slot
// length is ultimately determined by the locations of axle components
#include "component.h"
#include "elemental.h"

class CrossAxle {
public:
  CrossAxis(WithHole &comp, Measure ilen);
  CrossAxis(Vector nloc, Vector naxis);

  virtual void Draw() const;
  virtual void Info() const;

  virtual Measure updateSystem(Measure dt);
  virtual Measure dtneeded() const;
  virtual void makeChanges(Measure dt);

  void Through(AxleComponent &ngear, Measure ilen);
  void Rotate(Measure nomega);

  void setColor(int ncolor);
  void setWidth(Measure width);

  Vector getLoc() const;
  Vector getAxis() const;
private:
  Measure width;
  int color;

  Vector loc;
  Vector axis;

  bool fixed;
  Measure omega;

  SystemNode<AxleComponent> axlesys;
};

class AxleComponent : public AbstractComponent {
public:
  AxleComponent();

  virtual void Draw() const = 0;
  virtual void Info() const = 0;

  virtual Measure updateSystem(Measure dt) = 0;
  virtual Measure dtneeded() const = 0;
  virtual void makeChanges(Measure dt) = 0;

  virtual void Along(Axle &naxle, Measure nilen);
  virtual void Rotate(Measure nomega) = 0;
protected:
  CrossAxle *axle;
  Measure ilen;
};

class IdealGear : public AxleComponent {
public:
  IdealGear(unsigned nteeth);

  virtual void Draw() const;
  virtual void Info() const;

  virtual Measure updateSystem(Measure dt);
  virtual Measure dtneeded() const;
  virtual void makeChanges(Measure dt);

  virtual void Rotate(Measure nomega);

  unsigned getTeeth() const;
  void Mesh(Gear &ngear, Measure ilen);

  void setRadius(Measure nradius);
  void setColor(int ncolor);
private:
  unsigned teeth;

  Measure omega;

  SystemNode<IdealGear> meshsys;
};

// The helical spring is finally an example that uses impulses dynamically
// Note: currently needs someone else to update right + left points-- do I
// need another abstract/elemental mix or can this do as it is?  I think it can
// if I can somehow use information from the contacts and the points to find my
// new right and left points -- yes, do that way + just interact with Impulses

/* Okay, so I am going to save the position using spherical coordinates.  That
   automatically deals with changes in location, but it does not deal with
   changes in theta or axis. There are two reasonable ways to do this.  But
   since I don't want to track changes, or even initial conditions, I want to
   do this by saving the correct spherical coordinates that I can make modifi-
   cations with theta and axis to get the real spherical coordinates. */
class HelicalSpring : public AbstractComponent {
public:
  HelicalSpring(ElementalComponent &nrightcontact, Vector nrightpoint,
		ElementalComponent &nleftcontact, Vector nleftpoint,
		Measure nspringk, Measure nequilibrium);

  virtual void Draw() const;
  virtual void Info() const;

  virtual Measure updateSystem(Measure dt);
  virtual Measure dtneeded() const;
  virtual void makeChanges(Measure dt);

  void setRadius(Measure nradius);
  void setColor(int ncolor);
  void setLoops(double nloops);
private:
  Measure springk;
  Measure equilibrium;

  ElementalComponent *rightcontact;
  Vector rightpoint;
  ElementalComponent *leftcontact;
  Vector leftpoint;

  Measure oldlength;

  Measure radius;
  int color;
  double loops;
};

// Uses information for a component to convert relative points to absolue ones
Vector ToConnection(Vector relative, ElementalComponent &contact);
