// arbitrary.cpp - Function definitions to make general components

#include "arbitrary.h"

ShapedPiece::ShapedPiece(Shape &nshape, Measure nmass, Vector nloc, Vector nvel,
			 Measure nmoment, Measure ntheta, Measure nomega,
			 Vector naxis) {
  shape = &nshape;

  mass = nmass;
  loc = nloc;
  vel = nvel;

  moment = nmoment;
  theta = ntheta;
  omega = nomega;

  axis = naxis;
}

virtual void Draw() const {
  int count = 0;
  Edge &curredge;

  for (Vertex *currpt = shape->GetFirst();
       currpt && curredge = currpt->GetNext();
       currpt = curredge->GetNext())
    count += 1 + ciel(curredge->GetCurve());

  Point points[count];
  Point center;
  Point ptone, pttwo;
  Measure theta0, theta1;
  Measure theta, dtheta;

  for (Vertex *currpt = shape->GetFirst(), int i = 0;
       currpt && curredge = currpt->GetNext();
       currpt = curredge->GetNext()) {
    if (!curredge->GetCurve())
      points[i++] = currpt->GetPoint();
    else {
      ptone = currpt->GetPoint();
      pttwo = curredge->GetNext()->GetPoint()
      center = curredge->GetCenter();
      theta0 = arctan((pttwo.y - center.y) / (pttwo.x - center.x));
      theta1 = arctan((ptone.y - center.y) / (ptone.x - center.x));
      dtheta = (theta1 - theta0) / (1 + ciel(curredge->GetCurve()));
      for (theta = theta0; theta <= theta1; theta += dtheta)
	points[i++] = (curredge->GetCurve() * cos(theta),
		      curredge->GetCurve() * sin(theta));
    }
  }
  points[count - 1] = shape->GetLast()->GetPoint();

  CFillPolygon(loc, axis, points, count, shape->getWidth(), shape->getColor());
}

void ShapedPiece::Info() {
  cout << "(" << loc[1] << ", " << loc[2] << ", " << loc[3] << ") >> "
       << "(" << vel[1] << ", " << vel[2] << ", " << vel[3] << ") (+) "
       << theta << " >> " << omega;
}

Measure ShapedPiece::dtneeded() const {
  Measure mindt = largeval;
  Vertex *currpt = shape->GetFirst();
  Measure velm = vel.magnitude();
  Measure accm = acc.magnitude();
  Measure result;
  Measure dist;

  do {
    dist = currpt->GetPoint() - loc;
    result = velm + (dist / 2.0) * omega;
    if (!((result = result*result - 2.0*dr*(accm + (dist/2.0)*alpha)) < 0 ||
	(result = sqrt(result) - vel - (dist / 2) * omega) < 0))
      if (result = result / (accm + (dist / 2.0) * alpha) < mindt)
	mindt = result;
  } while (currpt = currpt->GetNext()->GetNext());
}

Vector ShapedPiece::Normal(Vector iloc, Vector dloc) {
  Vertex *curredge = shape->GetFirst()->GetNext();

  while (!curredge->Within(ThreeToTwo(iloc + dloc, axis, theta)))
    curredge = curredge->GetNext()->GetNext();

  if (!curredge->GetCurve())
    // assumes clockwise formation of shape
    return Cross(axis, curredge->GetNext()->GetPoint()
		 - curredge->GetPrev()->GetPoint()).unit();

  return iloc - TwoToThree(curredge->GetCenter(), axis, theta);
}

bool ShapedPiece::Within(Vector iloc) {
  Vertex *curredge = shape->GetFirst()->GetNext();

  if (Proj(iloc - loc, axis) > width / 2.0)
    return false;

  while (!curredge->Within(ThreeToTwo(iloc, axis, theta)))
    curredge = curredge->GetNext()->GetNext();

  return curredge->Within(ThreeToTwo(iloc, axis, theta));
}

Shape::Shape(Measure nwidth) {
  first = 0;
  last = 0;

  width = nwidth;
}

Shape &Shape::Start(Point pt) {
  if (!first)
    last = first = new Vertex(pt);
  return *this;
}

Shape &Shape::To(Point pt) {
  return To(pt, 0.0);
}

Shape &Shape::To(Point pt, Measure curve) {
  if (last)
    last = (*last).SetNext(curve).SetNext(pt);
  return *this;
}

Shape &Shape::From(Point pt) {
  return From(pt, 0.0);
}

Shape &Shape::From(Point pt, Measure curve) {
  if (first)
    first = (*first).SetPrev(curve).SetPrev(pt);
  return *this;
}

Shape &Shape::Return(Measure curve) {
  if (first)
    last = (*last).SetNext(curve).SetNext(*first);
  return *this;
}

Vertex *Shape::GetFirst() const {
  return first;
}

Vertex *Shape::GetLast() const {
  return last;
}

Shape &Shape::setColor(int ncolor) {
  color = ncolor;
}

Shape &Shape::setWidth(Measure nwidth) {
  width = nwidth;
}

int Shape::getColor() const {
  return color;
}

Measure Shape::getWidth() const {
  return width;
}

Vertex::Vertex(Point npt) {
  pt = npt;
  prev = 0;
  next = 0;
}

Edge &Vertex::SetNext(Measure ncurve) {
  return SetNext(*new Edge(ncurve));
}

Edge &Vertex::SetNext(Edge &nedge) {
  if (!next) {
    next = &nedge;
    nedge.SetPrev(*this);
  }
  return *next;
}

Edge &Vertex::SetPrev(Measure ncurve) {
  return SetPrev(*new Edge(ncurve));
}

Edge &Vertex::SetPrev(Edge &nedge) {
  if (!prev) {
    prev = &nedge;
    nedge.SetNext(*this);
  }
  return *prev;
}

Point Vertex::GetPoint() const {
  return pt;
}

Edge *Vertex::GetPrev() const {
  return prev;
}

Edge *Vertex::GetNext() const {
  return next;
}

Edge::Edge(Measure ncurve) {
  curve = ncurve;
  prev = 0;
  next = 0;
}

Vertex &Edge::SetNext(Vector npt) {
  return SetNext(*new Vertex(npt));
}

Vertex &Edge::SetNext(Vertex nvertex) {
  if (!next) {
    next = &nvertex;
    nvertex.SetPrev(*this);
  }
  return *next;
}

Vertex &Edge::SetPrev(Point npt) {
  return SetPrev(*new Vertex(npt));
}

Vertex &Edge::SetPrev(Vertex nvertex) {
  if (!prev) {
    prev = &nvertex;
    nvertex.SetNext(*this);
  }
  return *prev;
}

Measure Edge::GetCurve() const {
  return curve;
}

Point Edge::GetCenter() const {
  Point center;
  Point ptone, pttwo;
  Measure slope, xdist;

  if (!curve)
    return (0, 0);

  ptone = prev->GetPoint();
  pttwo = next->GetPoint();
  slope = (pttwo.y - ptone.y) / (pttwo.x - ptone.x);
  center = ((ptone.x + pttwo.x) / 2.0, (ptone.y + pttwo.y) / 2.0);
  // midpoint to center of circle is r, so x^2 + (slope x)^2 = r^2
  xdist = curredge->GetCurve() / sqrt(1.0 + slope * slope);
  center = (center.x - xdist, center.y - slope * xdist);

  return center;
}

bool Edge::Within(Point iloc) const {
  Point center
  Point relative;
  Point topoint;
  Point aproj;
  Measure radiusqr;
  Measure length;
  Measure projd;

  // this may not account for negaive curvatures
  if (curve) {
    center = GetCenter();
    topoint = (prev->GetPoint().x - center.x, prev->GetPoint().y)
    radiusqr = topoint.x * topoint.x + topoint.y + topoint.y;
    relative = (iloc.x - center.x, iloc.y - center.y);
    return (relative.x * relative.x + relative.y * relative.y < radiusqr &&
	    relative.x * relative.x + relative.y * relative.y
	    > radiusqr - 2 * dr * sqrt(radiusqr));
  } else {
    center = ((prev->GetPoint().x + next->GetPoint()) / 2.0,
	      (prev->GetPoint().y + next->GetPoint()) / 2.0);
    relative = (center.x - prev->GetPoint().x,
		center.y - prev->GetPoint().y);
    topoint = (iloc.x - center.x,
	       iloc.y - center.y);
    length = sqrt(relatve.x * relative.x + relative.y * relative.y);
    // 2-D Proj onto edge: along says if too far, across if close enough
    ((u . v) / v^2) v;
    if ((projd = (topoint.x * relative.x + topoint.y * relative.y) / length)
	< 2.0 * length) {  // not too far
      aproj = (topoint.x - projd * relative.x / length,
	       topoint.y - projd * relative.y / length);
      return (aproj.x * aproj.x + aproj.y * aproj.y < dr); // not far away
    }
    return false;
  }
}

Vertex *Edge::GetPrev() const {
  return prev;
}

Vertex *Edge::GetNext() const {
  return next;
}

Vector TwoToThree(Point twod, Vector axis, Measure theta) {
  Measure phi;
  Measure dist = sqrt(twod.x * twod.x + twod.y + twod.y);
  theta += tan(twod.y / twod.x) + Pi * (twod.x < 0);

  if (!axis[1] && !axis[2])   // in the x, y plane
    return (dist * sin(theta), dist * cos(theta), 0);

  /* solve Dot([l sin(t) sin(s), l cos(t) sin(s), l cos(s)], axis) = 0
     for s, then use s and t from the theta to find point */
  phi = -arctan(axis[3] / (sin(theta) * axis[1] + cos(theta) * axis[2]));
  return (dist * sin(theta) * sin(phi), dist * cos(theta) * sin(phi),
	  dist * cos(phi));
}

// One method is to find the projection onto two vectors in the plane
Point ThreeToTwo(Vector threed, Vector axis, Measure theta) {
  Vector planar = threed - Proj(threed, axis);
  Vector xprime = ((1, 0, 0) - Proj((1, 0, 0), axis)).unit();
  Vector yprime = Cross(axis, xprime).unit();
  Measure distance = planar.magnitude();
  Measure newtheta;
  Measure distx, disty;

  // Find the projection onto unit projected into the plane
  distx = Dot(planar, xprime);
  disty = Dot(planar, yprime);

  newtheta = arctan(disty / distx) - theta;

  return (distance * cos(newtheta), distance * sin(newtheta));
}
  
