#include "component.h"

/******************************* Component ******************************/

// Parent constructor for all components
Component::Component()
  : uptime(sec) {
  uptime = 0 * sec;  // uninitialized shortest-update-time
}

// Update component for time passing
//   uptime is changed to the lowest dt at any time
//   return time updates by
//   want my own copy of dt for once
Measure Component::Update(Measure &dt) {
  Measure testdt(sec);

  if (!dt)  // no change
    return dt;

  do {
    if (!uptime) {
      // Case 1: Never been called - ask system to update at my required speed
      // should divide evenly into dt
      uptime = dt / (double) ceil((dt / dtneeded())());

      uptime = updateSystem(uptime);
    } else
      // Case 2: Asking me to update at a speed that's good (dt <= uptime)
      return uptime = dt;

    // Now I have a suitably low time (and every object is in the recursion)
    //   unwind recursion, returning to top: spot where more dt than uptime
    makeChanges(uptime);

    testdt = uptime;
    uptime = 0 * sec;
  } while ((dt -= testdt) > 0 * sec);

  return testdt;
}

/*************************** Elemental Component *************************/

// Constructor for all physical components
ElementalComponent::ElementalComponent() 
  : mass(kg), moment(kg * Sqr(m)),
    theta(k), omega(1.0 / sec), alpha(1.0 / Sqr(sec)) {
  // Uninitialized state -- set all to 0
  mass = 0 * kg;
  moment = 0 * kg * Sqr(m);

  loc = Vector(0.0 * m, 0.0 * m, 0.0 * m);
  vel = Vector(0.0 * m / sec, 0.0 * m / sec, 0.0 * m / sec);
  acc = Vector(0.0 * m / Sqr(sec), 0.0 * m / Sqr(sec), 0.0 * m / Sqr(sec));

  theta = 0.0;
  omega = 0.0 / sec;
  alpha = 0.0 / Sqr(sec);
  axis = Vector(1.0 * m, 0.0 * m, 0.0 * m);
}

// Propogate effects for force applied
void ElementalComponent::Impulse(const Vector &iloc, const Vector &force) {
  acc += force / mass;

  // r x F = Torque = I alpha, but only about the axis
  // Can use either of the following
  // d omega / dt = Dot(a, Cross(r, F)) / I
  // d omega / dt = Cross(r, F - a Dot(F, a)) / I
  alpha += Dot(axis, Cross(iloc - loc, force)) / moment;
}

// Propogate changes to surrounding environment
Measure ElementalComponent::updateSystem(const Measure &dt) {
  SystemNode<ElementalComponent> *currnode;
  ElementalComponent *currcomp;

  Measure testdt(sec);

  // Step through all components to be affected
  for (currnode = &system; currnode && (currcomp = currnode->getContact());
       currnode = currnode->getNext())
    if ((testdt = currcomp->Update(uptime)) < uptime)
      uptime = testdt;

  return uptime;
}

// Update me for time passing
void ElementalComponent::makeChanges(const Measure &dt) {
  Vector loc0(loc);

  // change in location
  loc += dt * vel + dt * dt * acc / 2.0;
  // change in rotation
  theta += dt * omega + dt * dt * alpha / 2.0;

  // Check if collided with anything
  if (!checkCollisions(loc0, dt / 2.0)) {
    vel += dt * acc;
    omega += dt * alpha;
  }
}

// Determine if this component has collided with others in system
bool ElementalComponent::checkCollisions(const Vector &loc0,
					 const Measure &dt) {
  SystemNode<ElementalComponent> *currnode;
  ElementalComponent *currcomp;

  bool collisions = false;
  if (!dt)
    return false;

  // Step through items
  // Check if still in contact -- trace surface, never letting fall within
  for (currnode = &system; currnode && (currcomp = currnode->getContact());
       currnode = currnode->getNext())
    if (myCollisions(*currcomp)) {  // check for collision
      collisions = true;

      // If collision deal in component-depended way
      doCollision(*currcomp, loc0, loc - loc0, dt);

      // Cannot progress time as far (or will fall within), so try again.
      checkCollisions(loc0, dt / 2.0);
      // Also do for collided component
      currcomp->checkCollisions(loc0, dt / 2.0);
    }

  return collisions;
}

// Handle effects due to a collision
void ElementalComponent::doCollision(ElementalComponent &collcomp,
				     const Vector &iloc, const Vector &dloc,
				     const Measure &dt) {
  Vector normal;
  Vector dvel1, dvel2;
  Measure clsnforce(kg * m / Sqr(sec));
  Measure torqnorm1(1.0 / sec), torqnorm2(1.0 / sec);
  Measure domega1(1.0 / sec), domega2(1.0 / sec);

  // fell in, so apply equal and opposite forces to extract
  normal = collcomp.Normal(iloc, dloc);
  torqnorm1 = Dot(axis, Cross((iloc - loc), normal));
  torqnorm2 = Dot(axis, Cross((iloc - collcomp.loc), normal)); // see below
  /* Energy Conservation => 1/2 m1 (v10x^2 + v10y^2 + v10z^2)
       + 1/2 m2 (v20x^2 + v20y^2 + v20z^2) + 1/2 I1 omega10^2
       + 1/2 I2 omega20^2 = 1/2 m1 (v11x^2 + v11y^2 + v11z^2)
       + 1/2 m2 (v21x^2 + v21y^2 + v21z^2) + 1/2 I1 omega11^2
       + 1/2 I2 omega21^2
     Forces => vn1p = f Np dt / mn + vn0p
     Torque => omegan1 = f * Dot(a, Cross(rn, N)) * dt / In + omegan0
     a is the axis, rn is the vector to the point of contact
     Substitute in and solve for f =>
     f*dt = (Dot(N, v10 - v20) + Dot(a, Cross(r1, N)) omega10 
             - Dot(a, Cross(r2, N)) omega20)
            / (1 / m1 + 1 / m2 + Dot(a, Cross(r1, N)) / I1
	       + Dot(a, Cross(r2, N) / I2))
     Substitution into Forces and Torque equations cancels dt
     Get collision force, then change location, then change velocity */
  clsnforce = (Dot(normal, vel - collcomp.vel)
	       + torqnorm1 * omega - torqnorm2 * collcomp.omega)
    / (1.0 / mass + 1.0 / collcomp.mass
       + torqnorm1 / moment + torqnorm2 / collcomp.moment);

  // Change linear and angular velocity
  vel += dvel1 = clsnforce * normal / mass;
  collcomp.vel -= dvel2 = clsnforce * normal / collcomp.mass;
  omega += domega1 = clsnforce * torqnorm1 / moment;
  collcomp.omega -= domega2 = clsnforce * torqnorm2 / collcomp.mass;

  // Change position based on changes to derivatives
  loc += dvel1 * dt;  // Note dt is already divided by 2 or more
  collcomp.loc -= dvel2 * dt;
  theta += domega1 * dt;
  collcomp.theta -= domega2 * dt;
}

// Add new element to system
void ElementalComponent::AddContact(ElementalComponent &ncontact) {
  system.AddContact(ncontact);
  ncontact.AddContact(*this);
}

// Access functions

const Measure &ElementalComponent::getMass() const {
  return mass;
}

const Measure &ElementalComponent::getMoment() const {
  return moment;
}

const Vector &ElementalComponent::getLoc() const {
  return loc;
}

const Vector &ElementalComponent::getVel() const {
  return vel;
}

const Vector &ElementalComponent::getAcc() const {
  return acc;
}

const Measure &ElementalComponent::getTheta() const {
  return theta;
}

const Measure &ElementalComponent::getOmega() const {
  return omega;
}

const Measure &ElementalComponent::getAlpha() const {
  return alpha;
}

const Vector &ElementalComponent::getAxis() const {
  return axis;
}
