// clockmaker.h - top level declarations for ClockMaker classes
#ifndef CLOCKMAKER_H
#define CLOCKMAKER_H

#include "vector.h"
#include "system.h"
#include "drawing.h"

// for ElementalComponents
#define dr .001  // smallest width allowed (2 mm)
#define largeval 1.0 / dr
#define massless largeval

/* ----- Highest Level ----- */

// General "piece" of system
class Component {
public:
  Component();

  // Top-level functions for interacting with the UI
  virtual void Draw(DrawTablet &tablet) const = 0;
  virtual void Info() const = 0;

  // Update makes appopriate changes for the passage of time
  Measure Update(Measure &dt);
 
  // Update "up" for changes soon to be made in component
  virtual Measure updateSystem(const Measure &dt) = 0;

  virtual Measure dtneeded() const = 0;  // return largest time change to work

  // Update "within" now that the effects have propogated
  virtual void makeChanges(const Measure &dt) = 0;
protected:
  Measure uptime;  // For frequency of updates
};

/* ----- Second Level ----- */

// Intangible piece-- unimplemented
class AbstractComponent : public Component {
};

// Physical piece
class ElementalComponent : public Component {
public:
  ElementalComponent();

  // Interaction Functions
  // Assume time endpoints don't matter.   F * dt = dp = dv * m 
  // Note can't give duration, because updates won't know how much affected
  // so assume force is constant over whole time

  // Propogate effects for a force applied
  void Impulse(const Vector &iloc, const Vector &force);

  // Returns information about physical space of the component
  // iloc is the position before collision, dloc is the change which collides
  virtual Vector Normal(const Vector &iloc, const Vector &dloc) const = 0;
  virtual bool Within(const Vector &iloc) const = 0;

  // Functions for handling change due to time passing
  virtual Measure updateSystem(const Measure &dt);
  virtual void makeChanges(const Measure &dt);

  // Functions for handing collision detection and dealing
  bool checkCollisions(const Vector &loc0, const Measure &dt);
  virtual bool myCollisions(ElementalComponent &collcomp) const = 0;
  void doCollision(ElementalComponent &collcomp,
		   const Vector &iloc, const Vector &dloc, const Measure &dt);

  // Spacial relation functions (component-shape dependent)
  virtual Vector ClosestToPoint(const Vector &iloc) const = 0;
  virtual Vector ClosestToLine(const Vector &iloc, const Vector &along) const
    = 0;
  virtual Vector ClosestToPlane(const Vector &iloc, const Vector &axis) const
    = 0;

  // Set Functions
  void AddContact(ElementalComponent &ncontact);

  // Get Functions
  const Measure &getMass() const;
  const Measure &getMoment() const;
  const Vector &getLoc() const;
  const Vector &getVel() const;
  const Vector &getAcc() const;
  const Measure &getTheta() const;
  const Measure &getOmega() const;
  const Measure &getAlpha() const;
  const Vector &getAxis() const;

protected:
  // All ElementalComponents must have a mass and a moment of inertia
  Measure mass;
  Measure moment; /* Moment of inertia about center of mass and axis */

  Vector loc;
  Vector vel;
  Vector acc;     /* Linear Acceleration */

  Measure theta;  /* Angle wrt a standard about the axis below */
  Measure omega;  /* Angular velocity about the center of mass and axis */
  Measure alpha;  /* Angular acceleration */
  Vector axis;    /* unit vector for orientation along axis of rotation */

  // collection of components currently affecting this component
  SystemNode<ElementalComponent> system;
};

#endif
