// elemental.cpp - functions associated with the classes derived from
//                 ElementalComponent
#include "elemental.h"

PointMass::PointMass(Measure nmass, Vector nloc, Vector nvel) {
  mass = nmass;
  loc = nloc;
  vel = nvel;

  color = 0;
}

void PointMass::Draw(DrawTablet &tablet) const {
  CFillSphere(loc, dr, color);
}

void PointMass::Info() const {
  cout << loc << " >> " << vel << endl;
}

// calculate the time needed to travel only a distance of dr
/* dr = dt * v + dt^2 * acc / 2
   dt^2 * acc / 2 + dt * v - dr = 0
   dt = (-v +- sqrt(v^2 - 4(acc / 2)(-dr))) / (2 * acc / 2)
   dt = (sqrt(v^2 - 2 acc dr) - v) / acc  */
Measure PointMass::dtneeded() const {
  Measure velm(m / sec) = vel.magnitude();
  Measure accm(m / Sqr(sec)) = acc.magnitude();
  Measure result1(Sqr(m / sec));
  Measure result2(m / sec);

  if ((result1 = velm * velm + 2 * accm * dr) < 0.0 * Sqr(m / sec))
    return largeval * sec;
  if ((result2 = sqrt(result1) - velm) < 0.0 * m / sec)
    return largeval * sec;

  return result / accm;
}

Vector PointMass::Normal(Vector iloc, Vector dloc) {
  Vector normal = iloc - loc;
  return normal.unit();
}

bool PointMass::Within(Vector iloc) {
  return ((iloc - loc).magnitude() <= dr * m);
}

bool PointMass::myCollisions(ElementalComponent &collcomp) {
  return collcomp.Within(loc);
}

Vector PointMass::ClosestPoint(Vector iloc) const {
  return loc;
}

void PointMass::setColor(int ncolor) {
  color = ncolor;
}

SphereWeight::SphereWeight(Measure nmass, Vector nloc, Vector nvel,
			   Measure nomega, Vector naxis, Measure nradius)
  : PointMass(nmass, nloc, nvel) {
  radius(m) = nradius;
}

void SphereWeight::Draw(DrawTablet &tablet) const {
  CFillSphere(loc, radius, color);
}

void SphereWeight::Info() const {
  cout << loc << " >> " << vel
       << " <> " << nomega << endl;
}

Vector SphereWeight::ClosestPoint(Vector iloc) const {
  return radius * (iloc - loc).unit();
}

bool SphereWeight::Within(Vector iloc) {
  return ((iloc - loc).magnitude() <= radius);
}

bool SphereWeight::myCollisions(ElementalComponent &collcomp) const {
  return Within(collcomp.ClosestPoint(loc));
}

SimpleRod::SimpleRod(double nmass, Vector nloc, Vector nvel,
		     double ntheta, double nomega, Vector naxis)
  : MasslessRod(nloc, nvel, ntheta, nomega, naxis, nlength) {
  mass = nmass;
  moment = (1.0 / 12.0) * mass * length * length;  // moi of stick about com
}

// just determine the velocity of one of the endpoints
/* dr = (v + l/2 omega) dt + 1/2 (a + l/2 alpha) dt^2
   dt = (-(v + l/2 omega) +- sqrt((v + l/2 omega)^2
      - 4 (1/2 (a + l/2 alpha)) dr)) / 2 (1/2 (a + l/2 alpha))
   dt = (sqrt((v + l/2 omega)^2 - 2 dr (a + l/2 alpha)) - v - l/2 omega)
      / a + l/2 alpha   */
Measure SimpleRod::dtneeded() const {
  Measure velm = vel.magnitude();
  Measure accm = acc.magnitude();
  Measure velpt(m / sec);
  Measure velptcont(Sqr(m) / Sqr(sec));
  Measure numerator(m / sec);

  velpt = velm + (length / 2.0) * omega;
  if ((velptcont = Sqr(velpt) - 2.0 * dr * (accm + (length/2.0) * alpha)) < 0)
    return largeval * sec;
  if ((numerator = sqrt(velptcont) - velm - (length / 2) * omega) < 0)
    return largeval * sec;

  return numerator / (accm + (length / 2.0) * alpha);
}

// Just find the component of the relative position that is not along the rod
Vector SimpleRod::Normal(Vector iloc, Vector dloc) const {
  Vector along = endpoint1() - loc;
  Vector relative = iloc - loc;
  Measure alonglen = along.magnitude();

  return (relative - Proj(relative, along)).unit();
}

Vector SimpleRod::ClosestPoint(Vector iloc) const {
  Vector relative = iloc - loc;
  Vector along = Proj(relative, endpoint1() - loc);

  if (along.magnitude() > length / 2)
    return (length / 2.0) * along.unit();
  else
    return along;
}  

bool SimpleRod::Within(Vector iloc) const {
  Vector along = endpoint1() - loc;
  Vector relative = iloc - loc;

  along = Proj(relative, along);
  if (along.magnitude() < length / 2.0)
    return (relative - along <= dr);
  return false;
}

// Is any point along my length within the component?
// Now I want the closest point to a line, then I'll want the closest point
// to a plane.  Hmf...
bool SimpleRod::myCollisions(ElementalComponent &collcomp) const {
  Vector along = (endpoint1() - loc).unit();

  return Within(collcomp.ClosestToLine(loc, along));
}

RigidCombo::RigidCombo(Component &none, Component &ntwo) {
  Vector vectone, vecttwo;
  double distone, disttwo;

  one = &none;
  two = &ntwo;

  // make sure not to include massless masses
  mass = none.mass * (none.mass != massless)
    + ntwo.mass * (ntwo.mass != massless);
  loc = (none.mass * none.loc * (none.mass != massless)
    + ntwo.mass * ntwo.loc * (ntwo.mass != massless)) / mass;
  vel = none.vel;

  axis = none.axis;  /* A rigid combination already has an orientation built
			into it but needs to know how it can turn */

  vectone = none.loc - loc;
  vecttwo = ntwo.loc - loc;
  distone = (vectone - Proj(vectone, axis)).magnitude();
  disttwo = (vecttwo - Prok(vecttwo, axis)).magnitude();
  moment = (none.mass != massless)
    * (none.mass * distone * distone + none.moment)
    + (ntwo.mass != massless)
    * (ntwo.mass * disttwo * disttwo + ntwo.moment);
  omega = none.omega;

  oldloc = loc;
  oldtheta = theta;
}

void RigidCombo::Draw(DrawTablet &tablet) const {
  UpdateStatics();
  one->Draw(DrawTablet &tablet);
  two->Draw(DrawTablet &tablet);
}

void RigidCombo::Info() const {
  UpdateStatics();
  UpdateKinetics();
  one->Info();
  two->Info();
}

double RigidCombo::dtneeded() const {
  double dtone;
  double dttwo;

  UpdateKinetics();

  if ((dtone = one->dtneeded()) < (dttwo = two->dtneeded))
    return twone;
  else
    return twtwo;
}

Vector RigidCombo::Normal(Vector iloc, Vector dloc) {
  UpdateStatics();

  if (one->Within(iloc + dloc))
    return one->Normal(iloc);
  else
    return two->Normal(iloc);
}

bool RigidCombo::Within(Vector iloc) {
  UpdateStatics();

  return (one->Within(iloc) || two->Within(iloc));
}

void RigidCombo::UpdateStatics() {
  double dtheta = theta - oldtheta;
  double onedist = (one->loc - oldloc).magnitude();
  double twodist = (two->loc - oldloc).magnitude();
  Vector onead = one->axis - axis - Proj(one->axis - axis, axis);
  Vector twoad = two->axis - axis - Proj(two->axis - axis, axis);

  if (oldloc != loc || oldtheta != theta) {
    one->loc += loc - oldloc
      + onedist * tan(dtheta) * Cross(axis, one->loc - oldloc).unit()
      - onedist * (1.0 - cos(dtheta)) * (one->loc - oldloc).unit();
    two->loc += loc - oldloc
      + twodist * tan(dtheta) * Cross(axis, two->loc - oldloc).unit()
      - twodist * (1.0 - cos(dtheta)) * (two->loc - oldloc).unit();
    
    if (axis == one->axis)
      one->theta += theta - oldtheta;
    else
      one->axis += onead.magnitude() * tan(dtheta) * Cross(axis, onead).unit()
	- onead.magnitude() * (1.0 - cos(dtheta)) * onead.unit();

    if (axis == two->axis)
      two->theta += theta - oldtheta;
    else
      two->axis += twoad.magnitude() * tan(dtheta) * Cross(axis, twoad).unit()
	- twoad.magnitude() * (1.0 - cos(dtheta)) * twoad.unit();

    oldloc = loc;
    oldtheta = theta;
  }
}

void RigidCombo::UpdateKinetics() {
  // v = r omega
  one->vel = vel + Cross(axis, one->loc - oldloc).unit() * omega;
  two->vel = vel + Cross(axis, two->loc - oldloc).unit() * omega;

  if (one->axis == axis)
    one->omega = omega;
  if (two->axis == axis)
    two->omega = omega;
}

void RigidCombo::setVel(Vector nvel) {
  vel = nvel;
}

void RigidCombo::setOmega(double nomega) {
  omega = nomega;
}

void RigidCombo::setAxis(Vector naxis) {
  Vector vectone, vecttwo;
  double distone, disttwo;

  axis = naxis;

  vectone = none.loc - loc;
  vecttwo = ntwo.loc - loc;
  distone = (vectone - Proj(vectone, axis)).magnitude();
  disttwo = (vecttwo - Prok(vecttwo, axis)).magnitude();
  moment = (one->mass != massless)
    * (one->mass * distone * distone + one->moment)
    + (two->mass != massless)
    * (two->mass * disttwo * disttwo + two->moment);
}

/* ----- Bridging Components ----- */

WithHole::WithHole(ElementalComponent &ncomp, Vector nloc, Vector naxis) {
  Vector relative;
  double distance;

  comp = &ncomp;
  loc = nloc; // rotation will now be about this
  axis = naxis.unit();

  // make sure not to include massless masses
  mass = comp.mass * (none.mass != massless);
  vel = none.vel;  // beware: abstract components through hole will move with

  relative = none.loc - loc;
  distance = relative - Proj(relatve, axis);
  moment = (none.mass != massless)
    * (none.mass * distance * distance + none.moment);
  theta = none.theta;
  omega = none.omega; /* will only be correct if same axis of rotation, but if
			 not then no other source of omega */
}

void WithHole::Draw(DrawTablet &tablet) const {
  UpdateStatics();
  comp->Draw(DrawTablet &tablet);
}

void WithHole::Info() const {
  UpdateStatics();
  UpdateKinetics();
  comp->Info();
}

double WithHole::dtneeded() const {
  UpdateKinetics();
  return comp->dtneeded();
}

void WithHole::makeChanges(double dt) {
  ElementalComponent::makeChanges(dt);
  if (axle.isFixed())
    loc = axle.Loc();
  axle->Rotate(omega);
}

Vector WithHole::Normal(Vector iloc, Vector dloc) {
  UpdateStatics();
  return comp->Normal(iloc, dloc);
}

bool WithHole::Within(Vector iloc) {
  UpdateStatics();
  return comp->Within(iloc);
}

void WithHole::Rotate(double nomega) {
  omega = nomega;
}

void WithHole::Along(Axle &naxle, double ilen) {
  AxleComponent::Along(naxle, ilen);
  mass = massless;
}

void WithHole::UpdateStatics() {
  double dtheta = theta - oldtheta;
  double compdist = (comp->loc - oldloc).magnitude();
  Vector compad = comp->axis - axis - Proj(comp->axis - axis, axis);

  if (oldloc != loc || oldtheta != theta) {
    comp->loc += loc - oldloc
      + compdist * tan(dtheta) * Cross(axis, comp->loc - oldloc).unit()
      - compdist * (1.0 - cos(dtheta)) * (comp->loc - oldloc).unit();
    
    if (axis == one->axis)
      comp->theta += theta - oldtheta;
    else
      comp->axis += compad.magnitude() * tan(dtheta)
	* Cross(axis, compad).unit()
	- compad.magnitude() * (1.0 - cos(dtheta)) * compad.unit();

    oldloc = loc;
    oldtheta = theta;
  }
}

void WithHole::UpdateKinetics() {
  // v = r omega
  comp->vel = vel + Cross(axis, comp->loc - oldloc).unit() * omega;

  if (comp->axis == axis)
    comp->omega = omega;
}
