// elemental.h - classes in the ElementalComponent inheritance hierarchy
#ifndef ELEMENTAL_H
#define ELEMENTAL_H

#include <iostream.h>
#include "vector.h"
#include "system.h"
#include "component.h"
#include "massless.h"
#include "abstract.h"

/* ----- Third Level ----- */

class PointMass : public ElementalComponent {
public:
  PointMass(Measure nmass, Vector nloc, Vector nvel);

  virtual void Draw(DrawTablet &tablet) const;
  virtual void Info() const;

  virtual Measure dtneeded() const;
  virtual Vector Normal(Vector iloc, Vector dloc) const;
  virtual Vector ClosestPoint(Vector iloc) const;
  virtual bool Within(Vector iloc) const;
  virtual bool myCollisions(ElementalComponent &collcomp) const;

  void setColor(int ncolor);
private:
  int color;
};

class SphereWeight : public PointMass {
public:
  SphereWeight(Measure nmass, Vector nloc, Vector nvel,
	       Measure nomega, Vector naxis, Measure nradius);

  virtual void Draw(DrawTablet &tablet) const;
  virtual void Info() const;

  // dtneeded and normal are identical to those of a PointMass
  virtual Vector ClosestPoint(Vector iloc) const;
  virtual bool Within(Vector iloc) const;
  virtual bool myCollisions(ElementalComponent &collcomp) const;

  void setColor(int ncolor);
private:
  double radius;

  int color;
};

class SimpleRod : public MasslessRod {
  SimpleRod(double nmass, Vector nloc, Vector nvel,
	    double ntheta, double nomega, Vector naxis, double nlength);

  // Output functions do not need to be redefined
  // Only interactive functions need redefinition

  virtual Measure dtneeded() const;
  virtual Vector Normal(Vector iloc, Vector dloc) const;
  virtual Vector ClosestPoint(Vector iloc) const;
  virtual bool Within(Vector iloc) const;
  virtual bool myCollisions(ElementalComponent &collcomp) const;
};

/* Top down approach to rigid combinations: all interactions should take place
   through the RigidCombo object after a component is part of one */
/* Relative positions are implicit in the initial positions of the components
   However velocity, angular velocity, and axis (for rotation about) are
   taken from the first component, after which component data is ignored */
class RigidCombo : public Component {
public:
  RigidCombo(Component &one, Component &two);

  virtual void Draw(DrawTablet &tablet) const;
  virtual void Info() const;

  virtual Measure dtneeded() const;
  virtual Vector Normal(Vector iloc, Vector dloc);
  virtual bool Within(Vector iloc);

  /* The components' axes rotate and nothing else unless the axes coincide
     with the combo's, in which case theta rotates and nothing else */
  void UpdateStatics();
  void UpdateKinetics();

  void setVel(Vector nvel);
  void setOmega(double nomega);
  void setAxis(Vector naxis);
private:
  ElementalComponent *one;
  ElementalComponent *two;

  // to track changes (rather than force a functional beuracracy)
  Vector oldloc;
  double oldtheta;
};

/* ----- Bridging Components ----- */

// Make a component a WithHole component to make it part of an axle system
// Must be a combination of top down and bottom up
// to keep component about axle, simple keep setting loc to appropriate value
// and make mass -> infinity while moment remains the same
class WithHole : public ElementalComponent, public AxleComponent {
public:
  WithHole(ElementalComponent &ncomp, Vector nloc, Vector naxis);

  virtual void Draw(DrawTablet &tablet) const;
  virtual void Info() const;

  virtual Measure dtneeded() const;
  virtual void makeChanges(double dt);
  virtual Vector Normal(Vector iloc, Vector dloc);
  virtual bool Within(Vector iloc);

  virtual void Rotate(double nomega);
  virtual void Along(Axle &naxle, double ilen);

  void UpdateStatics();
  void UpdateKinetics();
protected:
  ElementalComponent *comp;

  Vector oldloc;
  double oldtheta;
};

#endif
