#include "massless.h"

MasslessRod::MasslessRod(const Vector &nloc, const Vector &nvel,
			 const Measure &ntheta, const Measure &nomega,
			 const Vector &naxis, const Measure &nlength) :
  length(m) {
  loc = nloc;
  vel = nvel;

  theta = ntheta;
  omega = nomega;
  axis = naxis.unit();

  length = nlength;

  color = 0;
}

// axis defines a plane, theta define the orientation within that plane
void MasslessRod::Draw(DrawTablet &tablet) const {
  CDrawLine(tablet, endpoint1(), endpoint2(), dr, color);
}

void MasslessRod::Info() const {
  cout << loc << " >> " << vel << " @ " << theta << " >> " << omega << endl;
}

void MasslessRod::Impulse(const Vector &iloc, const Vector &force) {}

Measure MasslessRod::dtneeded() const {
  return largeval * sec;
}

Vector MasslessRod::Normal(const Vector &iloc, const Vector &dloc) const {
  return Vector(0, 0, 0); // Impossible to collide with a massless rod
}

bool MasslessRod::Within(const Vector &iloc) const {
  return false;     // Impossible to collide with a massless rod
}

bool MasslessRod::myCollisions(ElementalComponent &collcomp) const {
  return false;
}

Vector MasslessRod::ClosestToPoint(const Vector &iloc) const {
  Vector relative = iloc - loc;
  Vector along = Proj(relative, endpoint1() - loc);

  if (along.magnitude() > length / 2)
    return (length / 2.0) * along.unit();
  else
    return along;
}

// Don't care about where on line that closest to, just where on me
Vector MasslessRod::ClosestToLine(const Vector &iloc, const Vector &along)
  const {
  Vector myalong = loc - endpoint1();
  Vector between = iloc - loc;

  Measure algdot = Dot(along, along);
  Measure algdmya = Dot(along, myalong);

  Measure t = (Dot(between, along) * algdmya + algdot * Dot(between, myalong))
    / (Sqr(algdmya) + algdot * Dot(myalong, myalong));

  return loc + myalong * t;
}

Vector MasslessRod::ClosestToPlane(const Vector &iloc, const Vector &axis)
  const {
  Vector myalong = loc - endpoint1();

  // Check if rod and plane are parallel: any point will do
  if (Dot(axis, myalong) == Measure(0.0 * m*m))
    return loc;

  // Check where line intersects plane
  Measure t = Dot(iloc - loc, myalong) / Dot(myalong, axis);
  if (Dot(axis, endpoint1() - iloc).Sign() ==
      Dot(axis, endpoint2() - iloc).Sign() || 
      (myalong * t).magnitude() > length / 2.0) { // rod doesn't intersect
    if (t > Measure(0.0))
      return endpoint1();
    else
      return endpoint2();
  } else
    return loc + myalong * t;
}   

Vector MasslessRod::endpoint1() const {
  double phi;

  if (!axis.xm() && !axis.ym())   // in the x, y plane
    return Vector(length * sin(theta), length * cos(theta), 0) + loc;

  /* solve Dot([l sin(t) sin(s), l cos(t) sin(s), l cos(s)], axis) = 0
     for s, then use s and t from the theta to find endpoints */
  phi = -atan(axis.zm() / (sin(theta) * axis.xm() + cos(theta) * axis.xm()));
  return Vector(length * sin(theta) * sin(phi), length * cos(theta) * sin(phi),
		length * cos(phi)) + loc;
}

Vector MasslessRod::endpoint2() const {
  Vector r1 = endpoint1() - loc;
  return loc - r1;
}

void MasslessRod::setColor(int ncolor) {
  color = ncolor;
}

StaticPlane::StaticPlane(const Vector &nloc, const Measure &ntheta,
			 const Vector &naxis, const Measure &nlength,
			 const Measure &nwidth) :
  length(m), width(m) {
  loc = nloc;
  theta = ntheta;
  axis = naxis.unit();

  length = nlength;
  width = nwidth;

  color = 0;
}

void StaticPlane::Draw(DrawTablet &tablet) const {
  CFillRectangle(tablet, cornerNW(), cornerNE(), cornerSE(), cornerSW(),
		 color);
}

void StaticPlane::Info() const {}

void StaticPlane::Impulse(const Vector &iloc, const Vector &force) {}

Measure StaticPlane::dtneeded() const {
  return largeval * sec;
}

Vector StaticPlane::Normal(const Vector &iloc, const Vector &dloc) const {
  if (Dot(iloc - loc, axis) < Measure(0 * m))
    return -1.0 * axis;
  else
    return axis;
}

// Split relative position into three components: axis, length, and width
bool StaticPlane::Within(const Vector &iloc) const {
  Vector relative = iloc - loc;

  if (Proj(relative, axis).magnitude() <= dr * m)
    if (Proj(relative, midpointN() - loc).magnitude() < width / 2.0 * m)
      if (Proj(relative, midpointE() - loc).magnitude() < length / 2.0 * m)
	return true;
  return false;
}

bool StaticPlane::myCollisions(ElementalComponent &collcomp) const {
  return Within(collcomp.ClosestToPlane(loc, axis));
}

Vector StaticPlane::ClosestToPoint(const Vector &iloc) const {
  Vector inplane(iloc - loc - Proj(iloc - loc, naxis));
  Vector northward = Proj(inplane, (midPointN() - loc).unit());
  Vector eastward = Proj(inplane, (midPointE() - loc).unit());

  Vector northcomp, eastcomp;

  if (northward.magnitude() < width / 2)
    northcomp = northward;
  else {
    if (northward.magnitude > 0)
      northcomp = midPointN();
    else
      northcomp = midPointS();
  }

  if (eastward.magnitude() < length / 2)
    eastcomp = eastward;
  else {
    if (eastward.magnitude > 0)
      eastcomp = midPointE();
    else
      eastcomp = midPointW();
  }

  return northcomp + eastcomp;
}

Vector StaticPlane::ClosestToLine

Vector StaticPlane::midpointN() const {
  MasslessRod imaginary(loc, Vector(0, 0, 0), theta + Measure(M_PI_2),
			0, axis, width);
  return imaginary.endpoint1();
}

Vector StaticPlane::midpointE() const {
  MasslessRod imaginary(loc, Vector(0, 0, 0), theta, 0, axis, length);
  return imaginary.endpoint1();
}

Vector StaticPlane::midpointS() const {
  MasslessRod imaginary(loc, Vector(0, 0, 0), theta + Measure(M_PI_2),
			0, axis, width);
  return imaginary.endpoint2();
}

Vector StaticPlane::midpointW() const {
  MasslessRod imaginary(loc, Vector(0, 0, 0), theta, 0, axis, length);
  return imaginary.endpoint2();
}

Vector StaticPlane::cornerNW() const {
  MasslessRod imaginary(midpointN(), Vector(0, 0, 0), theta, 0, axis, length);
  return imaginary.endpoint2();
}

Vector StaticPlane::cornerNE() const {
  MasslessRod imaginary(midpointN(), Vector(0, 0, 0), theta, 0, axis, length);
  return imaginary.endpoint1();
}

Vector StaticPlane::cornerSE() const {
  MasslessRod imaginary(midpointS(), Vector(0, 0, 0), theta, 0, axis, length);
  return imaginary.endpoint1();
}

Vector StaticPlane::cornerSW() const {
  MasslessRod imaginary(midpointS(), Vector(0, 0, 0), theta, 0, axis, length);
  return imaginary.endpoint2();
}

void StaticPlane::setColor(int ncolor) {
  color = ncolor;
}
