// vector.cpp - Member function definitions for accessory class Vector

#include "vector.h"

Vector::Vector() :
  one(m), two(m), thr(m) {
  one = two = thr = 0.0;
}

Vector::Vector(const Measure &none, const Measure &ntwo, const Measure &nthr) :
  one(m), two(m), thr(m) {
  if (checkUnits(none, ntwo) && checkUnits(ntwo, nthr)) {
    one = none;
    two = ntwo;
    thr = nthr; 
  }
  else if (checkUnits(m, none) && checkUnits(k, ntwo) && checkUnits(k, nthr)) {
    one = none * sin(nthr()) * cos(ntwo());
    two = none * sin(nthr()) * sin(ntwo());
    thr = none * cos(nthr());
  }
  else if (checkUnits(m, none) && checkUnits(k, ntwo) && checkUnits(m, nthr)) {
    one = none * cos(ntwo());
    two = none * sin(ntwo());
    thr = nthr;
  }
  else
    dimerror("vector creation");
}

Vector::Vector(const Vector &none, const Vector &ntwo, const Vector &nthr) :
  one(m), two(m), thr(m) {
  Vector(none.magnitude(), ntwo.magnitude(), nthr.magnitude());
}

const Vector Vector::operator+(const Vector &right) const {
  Vector addresult(x() + right.x(), y() + right.y(), z() + right.z());
  return addresult;
}

const Vector Vector::operator-(const Vector &right) const {
  Vector subresult(x() - right.x(), y() - right.y(), z() - right.z());
  return subresult;
}

const Vector Vector::operator*(const Measure &right) const {
  Vector mulresult(x() * right, y() * right, z() * right);
  return mulresult;
}

const Vector Vector::operator/(const Measure &right) const {
  Vector divresult(x() / right, y() / right, z() / right);
  return divresult;
}

const Vector Vector::operator-() const {
  Vector mnsresult(-(x().magnitude()), -(y().magnitude()), -(z().magnitude()));
  return mnsresult;
}

const Vector Vector::operator+=(const Vector &right) {
  return *this = *this + right;
}

const Vector Vector::operator-=(const Vector &right) {
  return *this = *this - right;
}

const Vector Vector::operator*=(const Measure &right) {
  return *this = *this * right;
}

const Vector Vector::operator/=(const Measure &right) {
  return *this = *this / right;
}

bool Vector::operator==(const Vector &right) const {
  return (one == right.one) && (two == right.two) && (thr == right.thr);
}

bool Vector::operator!=(const Vector &right) const {
  return !(*this == right);
}

const Vector Vector::x() const {
  Vector xresult(one, 0.0 * m, 0.0 * m);
  return xresult;
}

const Vector Vector::y() const {
  Vector yresult(0.0 * m, two, 0.0 * m);
  return yresult;
}

const Vector Vector::z() const {
  Vector zresult(0.0 * m, 0.0 * m, thr);
  return zresult;
}

const Vector Vector::zc() const {
  Vector zcresult(0.0 * m, 0.0, thr);
  return zcresult;
}

const Vector Vector::r() const {
  Vector rresult(Sqrt(Sqr(one) + Sqr(two)), 0.0, 0.0 * m);
  return rresult;
}

const Vector Vector::rho() const {
  Vector rhoresult(magnitude(), 0.0, 0.0);
  return rhoresult;
}

const Vector Vector::theta() const {
  Vector thetaresult(0.0 * m, atan(two / one), 0.0);
  return thetaresult;
}

const Vector Vector::thetac() const {
  Vector thetaresult(0.0 * m, atan(two / one), 0.0 * m);
  return thetaresult;
}

const Vector Vector::phi() const {
  Vector phiresult(0.0 * m, 0.0, acos(thr / magnitude()));
  return phiresult;
}

const Measure Vector::xm() const {
  return one;
}

const Measure Vector::ym() const {
  return two;
}

const Measure Vector::zm() const {
  return thr;
}

const Measure Vector::rm() const {
  return Sqrt(Sqr(one) + Sqr(two));
}

const Measure Vector::rhom() const {
  return magnitude();
}

const Measure Vector::thetam() const {
  return atan(two / one);
}

const Measure Vector::phim() const {
  return acos(thr / magnitude());
}

const Measure Vector::magnitude() const {
  return Sqrt(Sqr(one) + Sqr(two) + Sqr(thr));
}

const Measure Vector::mag() const {
  return magnitude();
}

const Vector Vector::unit() const {
  return (*this / magnitude());
}

// Friend functions

const Vector Cross(const Vector &left, const Vector &right) {
  Vector crsresult(left.two * right.thr - left.thr * right.two,
		   left.thr * right.one - left.one * right.thr,
		   left.one * right.two - left.two * right.one);
  return crsresult;
}

const Measure Dot(const Vector &left, const Vector &right) {
  return Sqr(left.one) + Sqr(left.two) + Sqr(left.thr);
}

const Vector Proj(const Vector &subv, const Vector &onto) {
  return (Dot(subv, onto) / onto.magnitude()) * onto;
}

const Vector operator*(const Measure &scalar, const Vector &right) {
  return right * scalar;
}

ostream &operator<<(ostream &sout, const Vector &vect) {
  sout << "(" << vect.x() << ", " << vect.y() << ", " << vect.y() << ")";
  return sout;
}
