// defield.cpp - function definitions for stuff derived from SpaceField class

#include "defield.h"

SpaceFieldByEquation::SpaceFieldByEquation() {
  Equation = '\0';
}

SpaceFieldByEquation::SpaceFieldByEquation(SpaceField &field) {
  Equation = &field;
}

SpaceFieldByEquation::~SpaceFieldByEquation() {
  delete Equation;
}

SpaceVector SpaceFieldByEquation::operator()(double x, double y,
					      double z) const {
  return (*Equation)(x, y, z);
}

SpaceFieldByFouriers::SpaceFieldByFouriers(char *filename) {
  // Open file, if already exists
  Fouriers = new fstream(filename, ios::in | ios::out | ios::nocreate);

  // File does not exist: create with no coefficients
  if (!Fouriers) {
    Fouriers = new fstream(filename, ios::in | ios::out);
    *Fouriers << 1;
    for (int i = 0; i < 27 * 3; i++)
      *Fouriers << cnum(0.0, 0.0);
    coeffcount = 1;
  }
  else
    *Fouriers >> coeffcount;

  Fouriers->clear();
  ddt = 0;
}

SpaceFieldByFouriers::~SpaceFieldByFouriers() {
  delete Fouriers;
  delete ddt;
}

void SpaceFieldByFouriers::d(SpaceField &field) {
  ddt = &field;
}

// Modify field by given amount of time, based on a simple extrapolation
// derivative.  Redefine Fourier coefficients appropriately.
void SpaceFieldByFouriers::Inc(double iinc) {
  const long tests = 2 * coeffcount;

  if (iinc <= 0)
    return;

  Complex *newcoeffs, prev;
  double ddtpoints[tests];
  double phi = -pi, theta = -pi, rho = -domain;

  Fouriers->seekg(sizeof(long));
  Fouriers->seekp(sizeof(long));

  for (char d = 0; d < 3; d++)
    for (int k = 0; k < tests; k++, rho += 2 * domain / tests) {
      for (int j = 0; j < tests; j++, theta += 2 * pi / tests) {
	for (int i = 0; i < tests; i++, phi += 2 * pi / tests)
	  ddtpoints[i] = (*ddt)(phi, theta, rho)[d];

	newcoeffs = MakeFouriers(ddtpoints, tests);

	for (int i = 0; i <= tests; i++) {
	  *Fouriers >> prev;
	  *Fouriers << newcoeffs[i] + prev;
	}
      
	delete newcoeffs;

	phi = -pi;
      }
      theta = -pi;
    }

  Fouriers->clear();
}

// Preincrement: assume a time change of the default amount
SpaceFieldByFouriers SpaceFieldByFouriers::operator++() {
  Inc(inc);

  return *this;
}

// Postincrement: same as preincrement, but returns object as before change
SpaceFieldByFouriers SpaceFieldByFouriers::operator++(int) {
  SpaceFieldByFouriers saved = *this;
  Inc(inc);

  return saved;
}

void SpaceFieldByFouriers::SetCoeffCount(long ncount) {
  const long conum = coeffcount * 2 + 1;    // Coefficients from 0 to 2m
  const long chlow = ncount - coeffcount;
  const long chhgh = ncount + coeffcount;

  Complex oldcoeffs[3][conum][conum][conum];

  if (ncount < 0)
    return;

  Fouriers->seekg(sizeof(long));

  for (char d = 0; d < 3; d++)
    for (int k = 0; k < conum; k++)
      for (int j = 0; j < conum; j++)
	for (int i = 0; i < conum; i++)
	  *Fouriers >> oldcoeffs[d][k][j][i];

  *Fouriers << ncount;

  for (char d = 0; d < 3; d++)
    for (int k = 0; k < 2 * ncount + 1; k++)
      for (int j = 0; j < 2 * ncount + 1; j++)
	for (int i = 0; i < 2 * ncount + 1; i++) {
	  if (k >= chlow && k <= chhgh && j >= chlow && j <= chhgh
	      && i >= chlow && j <= chhgh)
	    *Fouriers << oldcoeffs[d][k - chlow][j - chlow][i - chlow];
	  else
	    *Fouriers << cnum(0.0, 0.0);
	}

  coeffcount = ncount;

  Fouriers->clear();
}

long SpaceFieldByFouriers::GetCoeffCount() const {
  return coeffcount;
}

SpaceVector SpaceFieldByFouriers::operator()(double x, double y,
					      double z) const {
  const int conum = 2 * coeffcount + 1;
  double u[3] = { 0.0, 0.0, 0.0 };
  Complex currcoeff;

  Fouriers->seekg(sizeof(long));

  for (char d = 0; d < 3; d++)
    for (int k = 0; k < conum; k++)
      for (int j = 0; j < conum; j++)
	for (int i = 0; i < conum; i++) {
	  *Fouriers >> currcoeff;
	  u[d] += currcoeff.re * cos(i * x + j * y + k * z) +
	    currcoeff.im * sin(i * x + j * y + k * z);
	}
  
  Fouriers->clear();

  SpaceVector res(u[0], u[1], u[2]);
  return res;
}

SpaceFieldConstant::SpaceFieldConstant(SpaceVector nconst) {
  constant = nconst;
}

SpaceVector SpaceFieldConstant::operator()(double x, double y,
					   double z) const {
  return constant;
}

SpaceFieldVariable::SpaceFieldVariable(SpaceVector &nconst) {
  constant = &nconst;
}

SpaceVector SpaceFieldVariable::operator()(double x, double y,
					      double z) const {
  return *constant;
}

SpaceFieldAdd::SpaceFieldAdd(SpaceField &nleft, SpaceField &nright)
{
  left = &nleft;
  right = &nright;
}

SpaceVector SpaceFieldAdd::operator()(double x, double y, double z) const {
  SpaceVector vleft = (*left)(x, y, z);
  return vleft + (*right)(x, y, z);
}

SpaceFieldSub::SpaceFieldSub(SpaceField &nleft, SpaceField &nright)
{
  left = &nleft;
  right = &nright;
}

SpaceVector SpaceFieldSub::operator()(double x, double y, double z) const {
  SpaceVector vleft = (*left)(x, y, z);
  return vleft - (*right)(x, y, z);
}

SpaceFieldCrs::SpaceFieldCrs(SpaceField &nleft, SpaceField &nright)
{
  left = &nleft;
  right = &nright;
}

SpaceVector SpaceFieldCrs::operator()(double x, double y, double z) const {
  SpaceVector vleft = (*left)(x, y, z);
  return Cross(vleft, (*right)(x, y, z));
}

SpaceFieldScl::SpaceFieldScl(SpaceFunction &nleft, 
			     SpaceField &nright) {
  scalar = &nleft;
  field = &nright;
}

SpaceVector SpaceFieldScl::operator()(double x, double y, double z) const {
  SpaceVector vright = (*field)(x, y, z);
  return vright * (*scalar)(x, y, z);
}

SpaceFieldISc::SpaceFieldISc(SpaceFunction &nleft,
			     SpaceField &nright) {
  scalar = &nleft;
  field = &nright;
}

SpaceVector SpaceFieldISc::operator()(double x, double y, double z) const {
  SpaceVector vright = (*field)(x, y, z);
  return vright / (*scalar)(x, y, z);
}

SpaceFieldCrl::SpaceFieldCrl(SpaceField &nfield) {
  field = &nfield;
}

SpaceVector SpaceFieldCrl::operator()(double x, double y, double z) const {
  double ni, nj, nk;
  
  ni = ((*field)(x, y + DVAL, z)[2] - (*field)(x, y, z)[2]) / DVAL -
    ((*field)(x, y, z + DVAL)[1] - (*field)(x, y, z)[1]) / DVAL;
  nj = ((*field)(x, y, z + DVAL)[0] - (*field)(x, y, z)[0]) / DVAL -
    ((*field)(x + DVAL, y, z)[2] - (*field)(x, y, z)[2]) / DVAL;
  nk = ((*field)(x + DVAL, y, z)[1] - (*field)(x, y, z)[1]) / DVAL -
    ((*field)(x, y + DVAL, z)[0] - (*field)(x, y, z)[0]) / DVAL;

  SpaceVector res(ni, nj, nk);
  return res;
}

SpaceFieldMns::SpaceFieldMns(SpaceField &nfield) {
  field = &nfield;
}

SpaceVector SpaceFieldMns::operator()(double x, double y, double z) const {
  return -((*field)(x, y, z));
}

SpaceFunctionGrd::SpaceFunctionGrd(SpaceFunction &nfunction) {
  function = &nfunction;
}

SpaceVector SpaceFunctionGrd::operator()(double x, double y, double z) const {
  double ni, nj, nk;

  ni = ((*function)(x + DVAL, y, z) - (*function)(x, y, z)) / DVAL;
  nj = ((*function)(x, y + DVAL, z) - (*function)(x, y, z)) / DVAL;
  nk = ((*function)(x, y, z + DVAL) - (*function)(x, y, z)) / DVAL;

  SpaceVector res(ni, nj, nk);
  return res;
}
