// defuncn.cpp - functions associated with the classes derived from 
// SpaceFunction

#include "defuncn.h"

SpaceFunctionByEquation::SpaceFunctionByEquation() {
  Equation = '\0';
}

SpaceFunctionByEquation::SpaceFunctionByEquation(SpaceFunction &funcn) {
  Equation = &funcn;
}

SpaceFunctionByEquation::~SpaceFunctionByEquation() {
  delete Equation;
}

double SpaceFunctionByEquation::operator()(double x, double y,
					   double z) const {
  return (*Equation)(x, y, z);
}

SpaceFunctionByFouriers::SpaceFunctionByFouriers(char *filename) {
  // Open file, if already exists
  Fouriers = new fstream(filename, ios::in | ios::out | ios::nocreate);

  // File does not exist: create with no coefficients
  if (!Fouriers) {
    Fouriers = new fstream(filename, ios::in | ios::out);
    *Fouriers << 1;
    for (int i = 0; i < 27; i++)
      *Fouriers << cnum(0.0, 0.0);
    coeffcount = 1;
  }
  else
    *Fouriers >> coeffcount;

  Fouriers->clear();
  ddt = 0;
}

SpaceFunctionByFouriers::~SpaceFunctionByFouriers() {
  delete Fouriers;
  delete ddt;
}

void SpaceFunctionByFouriers::d(SpaceFunction &funcn) {
  ddt = &funcn;
}

// Modify function by given amount of time, based on a simple extrapolation
// derivative.  Redefine Fourier coefficients appropriately.
void SpaceFunctionByFouriers::Inc(double iinc) {
  const long tests = 2 * coeffcount;

  if (iinc <= 0)
    return;

  Complex *newcoeffs, prev;
  double ddtpoints[tests];
  double phi = -pi, theta = -pi, rho = -domain;

  Fouriers->seekg(sizeof(long));
  Fouriers->seekp(sizeof(long));

  for (int k = 0; k < tests; k++, rho += 2 * domain / tests) {
    for (int j = 0; j < tests; j++, theta += 2 * pi / tests) {
      for (int i = 0; i < tests; i++, phi += 2 * pi / tests)
	ddtpoints[i] = (*ddt)(phi, theta, rho);

      newcoeffs = MakeFouriers(ddtpoints, tests);

      for (int i = 0; i <= tests; i++) {
	*Fouriers >> prev;
	*Fouriers << newcoeffs[i] + prev;
      }

      delete newcoeffs;

      phi = -pi;
    }
    theta = -pi;
  }

  Fouriers->clear();
}

// Preincrement: assume a time change of the default amount
SpaceFunctionByFouriers SpaceFunctionByFouriers::operator++() {
  Inc(inc);

  return *this;
}

// Postincrement: same as preincrement, but returns object as before change
SpaceFunctionByFouriers SpaceFunctionByFouriers::operator++(int) {
  SpaceFunctionByFouriers saved = *this;
  Inc(inc);

  return saved;
}

void SpaceFunctionByFouriers::SetCoeffCount(long ncount) {
  const long conum = coeffcount * 2 + 1;    // Coefficients from 0 to 2m
  const long chlow = ncount - coeffcount;
  const long chhgh = ncount + coeffcount;

  Complex oldcoeffs[conum][conum][conum];

  if (ncount < 0)
    return;

  Fouriers->seekg(sizeof(long));

  for (int k = 0; k < conum; k++)
    for (int j = 0; j < conum; j++)
      for (int i = 0; i < conum; i++)
	*Fouriers >> oldcoeffs[k][j][i];

  *Fouriers << ncount;

  for (int k = 0; k < 2 * ncount + 1; k++)
    for (int j = 0; j < 2 * ncount + 1; j++)
      for (int i = 0; i < 2 * ncount + 1; i++) {
	if (k >= chlow && k <= chhgh && j >= chlow && j <= chhgh
	    && i >= chlow && j <= chhgh)
	  *Fouriers << oldcoeffs[k - chlow][j - chlow][i - chlow];
	else
	  *Fouriers << cnum(0.0, 0.0);
      }

  coeffcount = ncount;

  Fouriers->clear();
}

long SpaceFunctionByFouriers::GetCoeffCount() const {
  return coeffcount;
}

double SpaceFunctionByFouriers::operator()(double x, double y, 
					   double z) const {
  const int conum = 2 * coeffcount + 1;
  double u = 0.0;
  Complex currcoeff;

  Fouriers->seekg(sizeof(long));

  for (int k = 0; k < conum; k++)
    for (int j = 0; j < conum; j++)
      for (int i = 0; i < conum; i++) {
	*Fouriers >> currcoeff;
	u += currcoeff.re * cos(i * x + j * y + k * z) +
	  currcoeff.im * sin(i * x + j * y + k * z);
      }
  
  Fouriers->clear();

  return u;
}

SpaceFunctionConstant::SpaceFunctionConstant(double nconst) {
  constant = nconst;
}

double SpaceFunctionConstant::operator()(double x, double y, double z) const {
  return constant;
}

SpaceFunctionVariable::SpaceFunctionVariable(double &nconst) {
  constant = &nconst;
}

double SpaceFunctionVariable::operator()(double x, double y, double z) const {
  return *constant;
}

SpaceFunctionAdd::SpaceFunctionAdd(SpaceFunction &nleft,
				   SpaceFunction &nright) {
  left = &nleft;
  right = &nright;
}

double SpaceFunctionAdd::operator()(double x, double y, double z) const {
  double dleft = (*left)(x, y, z);
  return dleft + (*right)(x, y, z);
}

SpaceFunctionSub::SpaceFunctionSub(SpaceFunction &nleft,
				   SpaceFunction &nright) {
  left = &nleft;
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  eFunction &nright) {
  left = &nleft;
  right = &nright;
}

double SpaceFunctionDiv::operator()(double x, double y, double z) const {
  double dleft = (*left)(x, y, z);
  return dleft / (*right)(x, y, z);
}

SpaceFunctionPow::SpaceFunctionPow(SpaceFunction &nbase, double npower) {
  base = &nbase;
  power = npower;
}

double SpaceFunctionPow::operator()(double x, double y, double z) const {
  return pow((*base)(x, y, z), power);
}

SpaceFunctionMns::SpaceFunctionMns(SpaceFunction &nfunction) {
  function = &nfunction;
}

double SpaceFunctionMns::operator()(double x, double y, double z) const {
  return -(*function)(x, y, z);
}

SpaceFunctionDot::SpaceFunctionDot(SpaceField &nleft, 
				   SpaceField &nright) {
  left = &nleft;
  right = &nleft;
}

double SpaceFunctionDot::operator()(double x, double y, double z) const {
  double u;

  u = (*left)(x, y, z)[0] * (*right)(x, y, z)[0]
    + (*left)(x, y, z)[1] * (*right)(x, y, z)[1]
    + (*left)(x, y, z)[2] * (*right)(x, y, z)[2];

  return u;
}

SpaceFieldDvg::SpaceFieldDvg(SpaceField &nfield) {
  field = &nfield;
}

double SpaceFieldDvg::operator()(double x, double y, double z) const {
  double res;

  res =  ((*field)(x + DVAL, y, z)[0] - (*field)(x, y, z)[0]) / DVAL;
  res += ((*field)(x, y + DVAL, z)[1] - (*field)(x, y, z)[1]) / DVAL;
  res += ((*field)(x, y, z + DVAL)[2] - (*field)(x, y, z)[2]) / DVAL;

  return res;
}

PositionVariable::PositionVariable(SpaceVector nspec) {
  specific = nspec;
}

double PositionVariable::operator()(double x, double y, double z) const {
  SpaceVector pos(x, y, z);
  return Dot(specific, pos);
}
