// defuncn.h - class declarations for classes derived from SpaceFunction
#ifndef DEFUNCN_H
#define DEFUNCN_H

#include "diffeq.h"

/* ----- Third Level ----- Functions ----- */

// 3-D Function by explicit definition
class SpaceFunctionByEquation : public SpaceFunction {
public:
  SpaceFunctionByEquation();                 // Default Constructor - nothing
  SpaceFunctionByEquation(SpaceFunction &);  // Copy Constructor
  ~SpaceFunctionByEquation();                // Destructor - free memory

  virtual double operator()(double x, double y, double z) const;
private:
  SpaceFunction *Equation;
};

// 3-D function stored as the coefficients of a Fourier series in exponential
// form, in an array of doubles or a file of doubles
class SpaceFunctionByFouriers : public SpaceFunction, public SpaceDiffEqByFouriers {
public:
  SpaceFunctionByFouriers(char *);          // Constructor; passed file name
  ~SpaceFunctionByFouriers();               // Destructor - close file

  void d(SpaceFunction &);                  // Define derivative

  void Inc(double);                         // Change coefficients with a time
  SpaceFunctionByFouriers operator++();     // Change coefficients; prefix op.
  SpaceFunctionByFouriers operator++(int);  // Change coefficients; postfix op.

  void SetCoeffCount(long);                 // Change number of Fourier coeffs
  long GetCoeffCount() const;               // Return number of Fourier coeffs

  virtual double operator()(double x, double y, double z) const;
private:
  SpaceFunction *ddt;                       // equation for time changes funcn.

  fstream *Fouriers;

  long coeffcount;
};

// Constant class - saves a double
class SpaceFunctionConstant : public SpaceFunction {
public:
  SpaceFunctionConstant(double);

  virtual double operator()(double x, double y, double z) const;
private:
  double constant;
};

// Variable class - saves a reference to a double variable that may be changed
class SpaceFunctionVariable : public SpaceFunction {
public:
  SpaceFunctionVariable(double &);

  virtual double operator()(double x, double y, double z) const;
private:
  double *constant;
};

/* ----- Fourth Level ----- Functions Functions ----- */

// Sum of two functions
class SpaceFunctionAdd : public SpaceFunction {
public:
  SpaceFunctionAdd(SpaceFunction &, SpaceFunction &);

  virtual double operator()(double x, double y, double z) const;
private:
  SpaceFunction *right;
  SpaceFunction *left;
};

// Difference between two functions
class SpaceFunctionSub : public SpaceFunction {
public:
  SpaceFunctionSub(SpaceFunction &, SpaceFunction &);

  virtual double operator()(double x, double y, double z) const;
private:
  SpaceFunction *right;
  SpaceFunction *left;
};

// Product of two functions
class SpaceFunctionMul : public SpaceFunction {
public:
  SpaceFunctionMul(SpaceFunction &, SpaceFunction &);

  virtual double operator()(double x, double y, double z) const;
private:
  SpaceFunction *right;
  SpaceFunction *left;
};

// Quotient of two functions
class SpaceFunctionDiv : public SpaceFunction {
public:
  SpaceFunctionDiv(SpaceFunction &, SpaceFunction &);

  virtual double operator()(double x, double y, double z) const;
private:
  SpaceFunction *right;
  SpaceFunction *left;
};

// Function raised to a simple double power
class SpaceFunctionPow : public SpaceFunction {
public:
  SpaceFunctionPow(SpaceFunction &, double);

  virtual double operator()(double x, double y, double z) const;
private:
  SpaceFunction *base;
  double power;
};

// Unary minus operation class
class SpaceFunctionMns : public SpaceFunction {
public:
  SpaceFunctionMns(SpaceFunction &);

  virtual double operator()(double x, double y, double z) const;
private:
  SpaceFunction *function;
};

// Dot product between two space fields
class SpaceFunctionDot : public SpaceFunction {
public:
  SpaceFunctionDot(SpaceField &, SpaceField &);

  virtual double operator()(double x, double y, double z) const;
private:
  SpaceField *right;
  SpaceField *left;
};

// Divergence of a vector field
class SpaceFieldDvg : public SpaceFunction {
public:
  SpaceFieldDvg(SpaceField &);

  virtual double operator()(double x, double y, double z) const;
private:
  SpaceField *field;
};

// Accessories: objects which return pieces of the current position
//   The PositionVariable class returns a vector to the currently queried
//   position.  _phi, _theta_ and _rho are defined as the dot product
//   between this vector and <1, 0, 0>, <0, 1, 0>, and <0, 0, 1>.
class PositionVariable : public SpaceFunction {
public:
  PositionVariable(SpaceVector);

  virtual double operator()(double x, double y, double z) const;
private:
  SpaceVector specific;
};

#endif
