// diffeq.cpp - Function definitions from differential equations classes
#include "diffeq.h"
#include "defield.h"
#include "defuncn.h"
#include <math.h>

/* Non-member friend functions of class SpaceField */

SpaceFieldCrs &Cross(SpaceField &left, SpaceField &right) {
  return *(new SpaceFieldCrs(left, right));
}

SpaceFieldCrl &Curl(SpaceField &field) {
  return *(new SpaceFieldCrl(field));
}

SpaceFieldDvg &Div(SpaceField &field) {
  return *(new SpaceFieldDvg(field));
}

SpaceFunctionDot &Dot(SpaceField &left, SpaceField &right) {
  return *(new SpaceFunctionDot(left, right));
}

/* Member functions of class SpaceField */

SpaceFieldAdd &SpaceField::operator+(SpaceField &right) {
  return *(new SpaceFieldAdd(*this, right));
}

SpaceFieldSub &SpaceField::operator-(SpaceField &right) {
  return *(new SpaceFieldSub(*this, right));
}

SpaceFieldScl &SpaceField::operator*(SpaceFunction &right) {
  return *(new SpaceFieldScl(right, *this));
}

SpaceFieldISc &SpaceField::operator/(SpaceFunction &right) {
  return *(new SpaceFieldISc(right, *this));
}

SpaceFieldMns &SpaceField::operator-() {
  return *(new SpaceFieldMns(*this));
}

SpaceFunctionDot &SpaceField::operator*(SpaceField &right) {
  return *(new SpaceFunctionDot(*this, right));
}

/* Non-member friend functions of class SpaceFunction */

SpaceFunctionPow &Pow(SpaceFunction &base, double power) {
  return *(new SpaceFunctionPow(base, power));
}

SpaceFunctionGrd &Grad(SpaceFunction &function) {
  return *(new SpaceFunctionGrd(function));
}

/* Member functions of class SpaceFunction */

SpaceFunctionAdd &SpaceFunction::operator+(SpaceFunction &right) {
  return *(new SpaceFunctionAdd(*this, right));
}

SpaceFunctionSub &SpaceFunction::operator-(SpaceFunction &right) {
  return *(new SpaceFunctionSub(*this, right));
}

SpaceFunctionMul &SpaceFunction::operator*(SpaceFunction &right) {
  return *(new SpaceFunctionMul(*this, right));
}

SpaceFunctionDiv &SpaceFunction::operator/(SpaceFunction &right) {
  return *(new SpaceFunctionDiv(*this, right));
}

SpaceFunctionMns &SpaceFunction::operator-() {
  return *(new SpaceFunctionMns(*this));
}

SpaceFieldScl &SpaceFunction::operator*(SpaceField &right) {
  return *(new SpaceFieldScl(*this, right));
}

/* Class-wide definitions in class SpaceDiffEqByFouriers */

double SpaceDiffEqByFouriers::inc = 1.0;

double SpaceDiffEqByFouriers::domain = 1.0;

double SpaceDiffEqByFouriers::GetInc() {
  return inc;
}

void SpaceDiffEqByFouriers::SetInc(double ninc) {
  inc = (ninc > 0 ? ninc : 1.0);
}

double SpaceDiffEqByFouriers::GetDomain() {
  return domain;
}

void SpaceDiffEqByFouriers::SetDomain(double ndomain) {
  inc = (ndomain > 0 ? ndomain : 1.0);
}

/* Member function definitions for accessory class SpaceVector */

SpaceVector::SpaceVector(double ni, double nj, double nk) {
  i = ni;
  j = nj;
  k = nk;
}

SpaceVector SpaceVector::operator+(const SpaceVector &right) const {
  SpaceVector res(i + right.i, j + right.j, k + right.k);
  return res;
}

SpaceVector SpaceVector::operator-(const SpaceVector &right) const {
  SpaceVector res(i - right.i, j - right.j, k - right.k);
  return res;
}

SpaceVector SpaceVector::operator*(double right) const {
  SpaceVector res(i * right, j * right, k * right);
  return res;
}

SpaceVector SpaceVector::operator/(double right) const {
  SpaceVector res(i / right, j / right, k / right);
  return res;
}

SpaceVector SpaceVector::operator-() const {
  SpaceVector res(-i, -j, -k);
  return res;
}

double &SpaceVector::operator[](char dim) {
  switch (dim) {
  case (1):
  case ('i'):
  case('x'):
    return i;
  case (2):
  case ('j'):
  case ('y'):
    return j;
  case (3):
  case ('k'):
  case('z'):
    return k;
  }
}

SpaceVector Cross(const SpaceVector &left, const SpaceVector &right) {
  SpaceVector res(left.j * right.k - left.k * right.j,
		  left.k * right.i - left.i * right.k,
		  left.i * right.j - left.j * right.i);
  return res;
}

double Dot(const SpaceVector &left, const SpaceVector &right) {
  return (left.i * right.i + left.j * right.j + left.k * right.k);
}

// Complex Class Functions

Complex::Complex(double real, double imag) {
  re = real;
  im = imag;
}

Complex Complex::operator+(const Complex &right) const {
  Complex res(re + right.re, im + right.im);
  return res;
}

Complex Complex::operator-(const Complex &right) const {
  Complex res(re - right.re, im - right.im);
  return res;
}

Complex Complex::operator*(const Complex &right) const {
  Complex res(re*right.re - im*right.im, re*right.im + im*right.re);
  return res;
}

Complex Complex::operator/(double right) const {
  Complex res(re / right, im / right);
  return res;
}

Complex Complex::operator-() const {
  Complex res(-re, -im);
  return res;
}

Complex cexp(const Complex &opnd) {
  double e = exp(opnd.re);
  Complex res(e * cos(opnd.im), e * sin(opnd.im));
  return res;
}

Complex cnum(double real, double imag) {
  Complex res(real, imag);
  return res;
}

fstream &operator>>(fstream &file, Complex right) {
  file >> right.re >> right.im;
  return file;
}

fstream &operator<<(fstream &file, Complex right) {
  file << right.re << right.im;
  return file;
}
