// measure.h - Class declarations for the Mechanical Measures Class
//   makes functions formal by requiring appropriate units
#ifndef MEASURE_H
#define MEASURE_H

/* Goal is to write the following, with appropriate results:
  Measure mass1(kg) = 1.0 * kg;
  Measure mass2(kg) = 2.0 * kg;
  Measure distance1(m) = 3.0 * m;
  Measure distance2(m) = 2.0 * m;
  Measure totalmoment(kg * sqr(m));

  totalmomemt = mass1 * sqr(distance1) + mass2 * sqr(distance2);
  cout << totalmoment;  // 17.0 kg m^2;

  totalmoment = mass1 * sqr(distance1) + mass2 * distance2; // error
  cout << totalmoment;
*/

#include <iostream.h>
#include <fstream.h>
#include <math.h>
#include "misc.h"

#define dimerror(errorstr) { cout << "Units error in " << errorstr; exit(1); }

/* ----- Accessory structure ----- */

class Dimensions {
public:
  Dimensions(float ndist, float ntime, float nmass);
  bool operator==(const Dimensions &right) const;

  float dist;
  float time;
  float mass;
};

class Measure {
  friend const Measure Sqr(const Measure &);
  friend const Measure Sqrt(const Measure &);
  friend const Measure Pow(const Measure &, const Measure &);

  friend const Measure operator*(double, const Measure &);
  friend const Measure operator/(double, const Measure &);

  friend bool checkUnits(const Measure &, const Measure &);

  friend ostream &operator<<(ostream &, const Measure &);
public:
  Measure(const Measure &);            // Define dimensions
  Measure(double = 0.0);               // For unitless value

  virtual const Measure &operator=(const Measure &);

  const Measure operator+(const Measure &) const;
  const Measure operator-(const Measure &) const;
  const Measure operator*(const Measure &) const;
  const Measure operator*(double) const;
  const Measure operator/(const Measure &) const;
  const Measure operator/(double) const;
  const Measure operator-() const;

  const Measure operator+=(const Measure &);
  const Measure operator-=(const Measure &);
  // *= and /= change units of measure

  bool operator<(const Measure &) const;
  bool operator>(const Measure &) const;
  bool operator==(const Measure &) const;
  bool operator<=(const Measure &) const;
  bool operator>=(const Measure &) const;
  bool operator!=(const Measure &) const;
  bool operator!() const;
  operator bool() const;

  virtual double operator()() const;
  double operator()(const Measure &convert) const;

  int Sign() const;
protected:
  Measure(Dimensions nfactors);        // For setting const factors

  const Dimensions factors;
  double value;
};

class Dimension : public Measure {
public:
  Dimension(Measure &);              // Meant to hold conversions
  Dimension(double, double, double, double = 1.0); // For base SI

  virtual double operator()() const;
};

static const Dimension k(0, 0, 0);
static const Dimension m(1, 0, 0);
static const Dimension sec(0, 1, 0);
static const Dimension kg(0, 0, 1);

#endif
